package com.biz.crm.common.weixinsign.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.common.weixinsign.local.config.WXOpenPlatformAccountProperties;
import com.biz.crm.common.weixinsign.local.config.WXAccountProperties;
import com.biz.crm.common.weixinsign.sdk.common.constant.WXOpenConstant;
import com.biz.crm.common.weixinsign.sdk.common.enums.WXAuthorizationChangeTypeEnum;
import com.biz.crm.common.weixinsign.sdk.common.utils.AesException;
import com.biz.crm.common.weixinsign.sdk.common.utils.WXBizMsgCrypt;
import com.biz.crm.common.weixinsign.sdk.common.utils.XMLConverUtil;
import com.biz.crm.common.weixinsign.sdk.common.utils.msg.ComponentReceiveXML;
import com.biz.crm.common.weixinsign.sdk.common.utils.msg.EventReceiveXML;
import com.biz.crm.common.weixinsign.sdk.dto.ComponentVerifyTicketDto;
import com.biz.crm.common.weixinsign.sdk.dto.MsgEventDto;
import com.biz.crm.common.weixinsign.sdk.service.WXCacheVoService;
import com.biz.crm.common.weixinsign.sdk.service.WXOpenPlatformVoService;
import com.biz.crm.common.weixinsign.sdk.vo.WXJsConfigVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxCacheVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxComponentAccessTokenRespVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxJsApiTicketRespVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxWebViewAccessTokenRespVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Date;
import java.util.Formatter;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

/**
 * @author hecheng
 * @description: 微信开放平台 接口
 * @date 2021/10/16 上午10:03
 */
@Service
@Slf4j
public class WXOpenPlatformVoServiceImpl implements WXOpenPlatformVoService {

  @Autowired
  private WXAccountProperties wxAccountProperties;
  @Autowired
  private WXCacheVoService wxCache;
  @Autowired
  private RestTemplate restTemplate;
  @Autowired
  private RedisMutexService redisMutexService;


  @Override
  public void componentVerifyTicket(ComponentVerifyTicketDto componentVerifyTicketDto) {
    String signature = componentVerifyTicketDto.getSignature();
    String timestamp = componentVerifyTicketDto.getTimestamp();
    String nonce = componentVerifyTicketDto.getNonce();
    String msgSignature = componentVerifyTicketDto.getMsgSignature();
    String xmlEncrypt = componentVerifyTicketDto.getXmlEncrypt();
    Validate.notNull(wxAccountProperties, "未配置微信信息");
    WXOpenPlatformAccountProperties wxOpenPlatformAccountProperties = wxAccountProperties.getOpenPlatformAccount();
    Validate.notNull(wxOpenPlatformAccountProperties, "未配置微信第三方平台信息");
    String componentToken = wxOpenPlatformAccountProperties.getComponentToken();
    String componentEncodingaesKey = wxOpenPlatformAccountProperties.getComponentEncodingaesKey();
    String componentAppid = wxOpenPlatformAccountProperties.getComponentAppid();
    log.debug(String.format("COMPONENT_TOKEN=%s,signature=%s,timestamp=%s,nonce=%s,msgSignature=%s,xml=%s",
            componentToken, signature, timestamp, nonce, msgSignature, xmlEncrypt));
    // 判断消息是否空 （微信推送给第三方开放平台的消息一定是加过密的，无消息加密无法解密消息）
    if (StringUtils.isBlank(msgSignature)) {
      return;
    }
    boolean isValid = checkSignature(componentToken, signature, timestamp, nonce);
    log.debug(String.format("微信第三方平台消息验证isValid=%s", isValid));
    if (!isValid) {
      return;
    }
    String xml = null;
    try {
      WXBizMsgCrypt pc = new WXBizMsgCrypt(componentToken, componentEncodingaesKey, componentAppid);
      xml = pc.decryptMsg(msgSignature, timestamp, nonce, xmlEncrypt);
    } catch (AesException e) {
      log.error(e.getMessage(), e);
      throw new IllegalArgumentException("微信xml解密失败！");
    }
    ComponentReceiveXML com = XMLConverUtil.convertToObject(ComponentReceiveXML.class, xml);

    log.debug("ComponentReceiveXML=" + com.toString());
    String redisKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_VERIFY_TICKET_KEY_FORMAT, com.getAppid());
    boolean locked = false;
    try {
      locked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
      Validate.isTrue(locked, "缓存componentVerifyTicket繁忙请稍后再试！");
      if (Objects.equals(com.getInfoType(), "component_verify_ticket")) {
        String cacheKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_VERIFY_TICKET_FORMAT, com.getAppid());
        WxCacheVo cacheVo = new WxCacheVo();
        cacheVo.setCreateTimestamp(new Date().getTime());
        cacheVo.setValue(com.getComponentVerifyTicket());
        cacheVo.setExpiresIn(12 * 60 * 60);
        this.wxCache.set(cacheKey, JSONObject.toJSONString(cacheVo));
      } else {
        //这里处理授权变更事件

        if (Objects.equals(com.getInfoType(), WXAuthorizationChangeTypeEnum.AUTHORIZED.getCode())) {
          //授权成功通知
          this.getApiQueryAuth(com.getAuthorizationCode());
        } else if (Objects.equals(com.getInfoType(), WXAuthorizationChangeTypeEnum.UPDATEAUTHORIZED.getCode())) {
          //授权更新通知
          this.getApiQueryAuth(com.getAuthorizationCode());
        } else if (Objects.equals(com.getInfoType(), WXAuthorizationChangeTypeEnum.UNAUTHORIZED.getCode())) {
          String authorizerAccessTokenCacheKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_AUTHORIZER_ACCESS_TOKEN_FORMAT, com.getAuthorizerAppid());
          String authorizerRefreshTokenCacheKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_AUTHORIZER_REFRESH_TOKEN_FORMAT, com.getAuthorizerAppid());
          //取消授权通知
          this.wxCache.remove(authorizerAccessTokenCacheKey);
          //取消授权通知
          this.wxCache.remove(authorizerRefreshTokenCacheKey);
        }
      }
    } finally {
      if (locked) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  @Override
  public EventReceiveXML analyzeMsgEvent(MsgEventDto msgEventDto) {
    String timestamp = msgEventDto.getTimestamp();
    String nonce = msgEventDto.getNonce();
    String msgSignature = msgEventDto.getMsgSignature();
    String signature = msgEventDto.getSignature();
    String xmlEncrypt = msgEventDto.getXmlEncrypt();
    Validate.notNull(wxAccountProperties, "未配置微信信息");
    WXOpenPlatformAccountProperties wxOpenPlatformAccountProperties = wxAccountProperties.getOpenPlatformAccount();
    Validate.notNull(wxOpenPlatformAccountProperties, "未配置微信第三方平台信息");
    String componentToken = wxOpenPlatformAccountProperties.getComponentToken();
    String componentEncodingaesKey = wxOpenPlatformAccountProperties.getComponentEncodingaesKey();
    String componentAppid = wxOpenPlatformAccountProperties.getComponentAppid();
    log.debug(String.format("COMPONENT_TOKEN=%s,timestamp=%s,nonce=%s,msgSignature=%s,xml=%s",
            componentToken, timestamp, nonce, msgSignature, xmlEncrypt));
    // 判断消息是否空 （微信推送给第三方开放平台的消息一定是加过密的，无消息加密无法解密消息）
    if (StringUtils.isBlank(msgSignature)) {
      return null;
    }
    if (StringUtils.isNotBlank(signature)) {
      boolean isValid = checkSignature(componentToken, signature, timestamp, nonce);
      Validate.isTrue(isValid, "微信第三方平台消息验证失败");
    }
    String xml = null;
    try {
      WXBizMsgCrypt pc = new WXBizMsgCrypt(componentToken, componentEncodingaesKey, componentAppid);
      xml = pc.decryptMsg(msgSignature, timestamp, nonce, xmlEncrypt);
    } catch (AesException e) {
      log.error(e.getMessage(), e);
      throw new IllegalArgumentException("微信xml解密失败！");
    }
    EventReceiveXML eventMsg = XMLConverUtil.convertToObject(EventReceiveXML.class, xml);
    return eventMsg;
  }

  @Override
  public String getComponentAccessToken() {
    WXOpenPlatformAccountProperties wxOpenPlatformAccountProperties = wxAccountProperties.getOpenPlatformAccount();
    Validate.notNull(wxOpenPlatformAccountProperties, "未配置微信第三方平台信息");
    String componentAppid = wxOpenPlatformAccountProperties.getComponentAppid();
    String redisKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_ACCESS_TOKEN_KEY_FORMAT, componentAppid);
    boolean locked = false;
    try {
      locked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
      Validate.isTrue(locked, "缓存componentAccessToken繁忙请稍后再试！");
      String componentAppsecret = wxOpenPlatformAccountProperties.getComponentAppSecret();
      String openCacheKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_ACCESS_TOKEN_FORMAT, componentAppid);
      String accessToken = this.wxCache.get(openCacheKey);
      if (StringUtils.isNotBlank(accessToken)) {
        log.debug("第三方平台componentAccessToken:{}", accessToken);
        return accessToken;
      }
      String cacheKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_VERIFY_TICKET_FORMAT, componentAppid);
      String componentVerifyTicket = this.wxCache.get(cacheKey);
      Validate.notBlank(componentVerifyTicket, "获取第三方平台component_access_token时:componentVerifyTicket不能为空");
      HttpHeaders headers = new HttpHeaders();
      headers.setContentType(MediaType.APPLICATION_JSON);
      JSONObject postData = new JSONObject();
      postData.put("component_appid", componentAppid);
      postData.put("component_appsecret", componentAppsecret);
      postData.put("component_verify_ticket", componentVerifyTicket);
      HttpEntity<String> request = new HttpEntity<String>(postData.toJSONString(), headers);
      WxComponentAccessTokenRespVo componentAccessTokenRespVo = this.restTemplate.postForObject(WXOpenConstant.API_COMPONENT_TOKEN_URL, request, WxComponentAccessTokenRespVo.class);
      if (Objects.isNull(componentAccessTokenRespVo) || StringUtils.isNotBlank(componentAccessTokenRespVo.getErrcode())) {
        log.warn("获取第三方平台componentAccessToken失败！{}", componentAccessTokenRespVo.getErrmsg());
        return null;
      }
      accessToken = componentAccessTokenRespVo.getComponentAccessToken();
      WxCacheVo cacheVo = new WxCacheVo();
      cacheVo.setCreateTimestamp(new Date().getTime());
      cacheVo.setValue(accessToken);
      cacheVo.setExpiresIn(componentAccessTokenRespVo.getExpiresIn());
      this.wxCache.set(openCacheKey, JSONObject.toJSONString(cacheVo));
      log.debug("第三方平台componentAccessToken:{}", accessToken);
      return accessToken;
    } finally {
      if (locked) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  @Override
  public String getApiQueryAuth(String authorizationCode) {
    String redisKey = String.format("OPEN_COMPONENT_ACCESS_AUTHORIZATION_CODE_KEY", authorizationCode);
    boolean locked = false;
    try {
      locked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
      Validate.isTrue(locked, "缓存ApiQueryAuth繁忙请稍后再试！");
      WXOpenPlatformAccountProperties wxOpenPlatformAccountProperties = wxAccountProperties.getOpenPlatformAccount();
      Validate.notNull(wxOpenPlatformAccountProperties, "未配置微信第三方平台信息");
      String componentAppid = wxOpenPlatformAccountProperties.getComponentAppid();
      String componentAccessToken = this.getComponentAccessToken();
      Validate.notBlank(componentAccessToken, "获取第三方平台ApiQueryAuth时:componentAccessToken不能为空");
      HttpHeaders headers = new HttpHeaders();
      headers.setContentType(MediaType.APPLICATION_JSON);
      JSONObject postData = new JSONObject();
      postData.put("component_appid", componentAppid);
      postData.put("authorization_code", authorizationCode);
      HttpEntity<String> request = new HttpEntity<String>(postData.toJSONString(), headers);
      Map<String, String> uriVariables = new HashMap<>();
      uriVariables.put("component_access_token", componentAccessToken);
      JSONObject apiQueryAuthRespVo = this.restTemplate
              .postForObject(WXOpenConstant.API_QUERY_AUTH_URL + "?component_access_token={component_access_token}", request, JSONObject.class, uriVariables);
      if (Objects.isNull(apiQueryAuthRespVo) || StringUtils.isNotBlank(apiQueryAuthRespVo.getString("errcode"))) {
        log.warn("获取第三方平台apiQueryAuth失败！{}", apiQueryAuthRespVo.getString("errmsg"));
        return null;
      }
      JSONObject authorizationInfoJson = apiQueryAuthRespVo.getJSONObject("authorization_info");
      // 授权方appid
      String authorizerAppid = authorizationInfoJson.getString("authorizer_appid");
      // 授权方接口调用凭据（在授权的公众号或小程序具备API权限时，才有此返回值），也简称为令牌
      String authorizerAccessToken = authorizationInfoJson.getString("authorizer_access_token");
      //刷新令牌
      String authorizerRefreshToken = authorizationInfoJson.getString("authorizer_refresh_token");
      //
      String authorizerAccessTokenCacheKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_AUTHORIZER_ACCESS_TOKEN_FORMAT, authorizerAppid);
      int expires_in = authorizationInfoJson.getIntValue("expires_in");
      WxCacheVo authorizerAccessTokenCacheVo = new WxCacheVo();
      authorizerAccessTokenCacheVo.setCreateTimestamp(new Date().getTime());
      authorizerAccessTokenCacheVo.setValue(authorizerAccessToken);
      authorizerAccessTokenCacheVo.setExpiresIn(expires_in);
      this.wxCache.set(authorizerAccessTokenCacheKey, JSONObject.toJSONString(authorizerAccessTokenCacheVo));
      //
      String authorizerRefreshTokenCacheKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_AUTHORIZER_REFRESH_TOKEN_FORMAT, authorizerAppid);
      WxCacheVo authorizerRefreshTokenCacheVo = new WxCacheVo();
      authorizerRefreshTokenCacheVo.setCreateTimestamp(new Date().getTime());
      authorizerRefreshTokenCacheVo.setValue(authorizerRefreshToken);
      authorizerRefreshTokenCacheVo.setExpiresIn(60 * 60 * 24 * 30);
      this.wxCache.set(authorizerRefreshTokenCacheKey, JSONObject.toJSONString(authorizerRefreshTokenCacheVo));
      log.debug("第三方平台authorizerAccessToken:{}", authorizerAccessToken);
      return authorizerAccessToken;
    } finally {
      if (locked) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  @Override
  public String customSendMessage(JSONObject message, String authorizerAccessToken) {
    if (Objects.isNull(message) || StringUtils.isBlank(authorizerAccessToken)) {
      return null;
    }
    HttpHeaders headers = new HttpHeaders();
    headers.setContentType(MediaType.APPLICATION_JSON);
    HttpEntity<String> request = new HttpEntity<String>(message.toJSONString(), headers);
    Map<String, String> uriVariables = new HashMap<>();
    uriVariables.put("access_token", authorizerAccessToken);
    JSONObject msgRespVo = this.restTemplate
            .postForObject(WXOpenConstant.CUSTOM_MESSAGE_URL + "?access_token={access_token}", request, JSONObject.class, uriVariables);
    if (Objects.isNull(msgRespVo) || StringUtils.isNotBlank(msgRespVo.getString("errcode"))) {
      log.warn("发送客服消息失败！{}", msgRespVo.getString("errmsg"));
      return null;
    }
    return msgRespVo.toJSONString();
  }

  @Override
  public String getPreAuthCode() {
    WXOpenPlatformAccountProperties wxOpenPlatformAccountProperties = wxAccountProperties.getOpenPlatformAccount();
    Validate.notNull(wxOpenPlatformAccountProperties, "未配置微信第三方平台信息");
    String componentAppid = wxOpenPlatformAccountProperties.getComponentAppid();
    String componentAccessToken = getComponentAccessToken();
    HttpHeaders headers = new HttpHeaders();
    headers.setContentType(MediaType.APPLICATION_JSON);
    JSONObject postData = new JSONObject();
    postData.put("component_appid", componentAppid);
    HttpEntity<String> request = new HttpEntity<String>(postData.toJSONString(), headers);
    Map<String, String> uriVariables = new HashMap<>();
    uriVariables.put("component_access_token", componentAccessToken);
    JSONObject msgRespVo = this.restTemplate
            .postForObject(WXOpenConstant.API_CREATE_PREAUTHCODE_URL + "?component_access_token={component_access_token}", request, JSONObject.class, uriVariables);
    if (Objects.isNull(msgRespVo) || StringUtils.isNotBlank(msgRespVo.getString("errcode"))) {
      log.warn("发送客服消息失败！{}", msgRespVo.getString("errmsg"));
      return null;
    }
    return msgRespVo.getString("pre_auth_code");
  }

  @Override
  public String getComponentAuthUrl(String redirectUri) {
    String preAuthCode = this.getPreAuthCode();
    WXOpenPlatformAccountProperties wxOpenPlatformAccountProperties = wxAccountProperties.getOpenPlatformAccount();
    Validate.notNull(wxOpenPlatformAccountProperties, "未配置微信第三方平台信息");
    String componentAppid = wxOpenPlatformAccountProperties.getComponentAppid();
    // 获取code
    String url = String.format(
            "https://mp.weixin.qq.com/cgi-bin/componentloginpage?component_appid=%s&pre_auth_code=%s&redirect_uri=%s&auth_type=%s",
            componentAppid, preAuthCode, redirectUri, 1);
    return url;
  }

  @Override
  public String getAuthorizeUrl(String redirectUri, String scope, String appid) {
    WXOpenPlatformAccountProperties wxOpenPlatformAccountProperties = wxAccountProperties.getOpenPlatformAccount();
    Validate.notNull(wxOpenPlatformAccountProperties, "未配置微信第三方平台信息");
    String componentAppid = wxOpenPlatformAccountProperties.getComponentAppid();
    String urlEncodeRedirectUri = null;
    try {
      urlEncodeRedirectUri = URLEncoder.encode(redirectUri, "utf-8");
    } catch (UnsupportedEncodingException e) {
      throw new IllegalArgumentException("redirectUri不能为空！");
    }
    String authorizeUrl = String.format(WXOpenConstant.COMPONENT_AUTHORIZE_URL_FORMAT, appid, urlEncodeRedirectUri, scope, componentAppid);
    return authorizeUrl;
  }

  @Override
  public WxWebViewAccessTokenRespVo findWebViewAccessToken(String code, String appid) {
    WXOpenPlatformAccountProperties wxOpenPlatformAccountProperties = wxAccountProperties.getOpenPlatformAccount();
    Validate.notNull(wxOpenPlatformAccountProperties, "未配置微信第三方平台信息");
    String componentAppid = wxOpenPlatformAccountProperties.getComponentAppid();
    if (StringUtils.isBlank(code)) {
      log.warn("获取网页授权access_token失败!code为空");
      return null;
    }
    String componentAccessToken = getComponentAccessToken();
    Validate.notBlank("componentAccessToken", "获取第三方平台accesstoken失败");
    Map<String, String> uriVariables = new HashMap<>();
    uriVariables.put("appid", appid);
    uriVariables.put("component_appid", componentAppid);
    uriVariables.put("component_access_token", componentAccessToken);
    uriVariables.put("code", code);
    String jsonStr = this.restTemplate
            .getForObject(WXOpenConstant.COMPONENT_WEB_VIEW_ACCESS_TOKEN_URL
                            + "?appid={appid}&code={code}&grant_type=authorization_code&component_appid={component_appid}&component_access_token={component_access_token}", String.class,
                    uriVariables);
    WxWebViewAccessTokenRespVo accessTokenRespVo = JSONObject.parseObject(jsonStr, WxWebViewAccessTokenRespVo.class);
    if (Objects.isNull(accessTokenRespVo) || StringUtils.isNotBlank(accessTokenRespVo.getErrcode())) {
      log.warn("获取网页授权access_token失败！{}", accessTokenRespVo.getErrmsg());
      return null;
    }
    log.info("获取网页授权access_token！{}", JSONObject.toJSONString(accessTokenRespVo));
    return accessTokenRespVo;
  }

  @Override
  public WXJsConfigVo findWXJsConfig(String url, String appid) {
    Validate.notBlank(url, "获取jssdk的url不能为空");
    String jsapiTicket = this.findJsApiTicket(appid);
    if (StringUtils.isBlank(jsapiTicket)) {
      return null;
    }
    //
    String nonceStr = UUID.randomUUID().toString();
    long timestamp = System.currentTimeMillis() / 1000;
    String signOriginal = String.format(WXOpenConstant.SIGN_FORMAT, jsapiTicket, nonceStr, timestamp, url);
    log.info("wxopen:signOriginal:{}", signOriginal);
    String signature = null;
    try {
      MessageDigest crypt = MessageDigest.getInstance("SHA-1");
      crypt.reset();
      crypt.update(signOriginal.getBytes("UTF-8"));
      signature = this.byteToHex(crypt.digest());
    } catch (NoSuchAlgorithmException e) {
      log.error(e.getMessage(), e);
    } catch (UnsupportedEncodingException e) {
      log.error(e.getMessage(), e);
    }
    Validate.notBlank(signature, "获取jssdk时签名失败");
    WXJsConfigVo vo = new WXJsConfigVo();
    vo.setAppId(appid);
    vo.setNonceStr(nonceStr);
    vo.setSignature(signature);
    vo.setTimestamp(timestamp);
    return vo;
  }

  @Override
  public String findJsApiTicket(String appid) {
    log.info("wxopen:jsApiTicket:wxProperties:{}", appid);
    String cacheKey = String.format(WXOpenConstant.OPEN_CACHE_JSAPI_TICKET_FORMAT, appid);
    String jsApiTicket = this.wxCache.get(cacheKey);
    if (StringUtils.isNotBlank(jsApiTicket)) {
      log.debug("wxopen:jsApiTicket:{}", jsApiTicket);
      return jsApiTicket;
    }
    String accessToken = this.findApiAuthorizerToken(appid);
    if (StringUtils.isBlank(accessToken)) {
      return null;
    }
    Map<String, String> uriVariables = new HashMap<>();
    uriVariables.put("access_token", accessToken);
    uriVariables.put("type", "jsapi");
    WxJsApiTicketRespVo jsApiTicketRespVo = this.restTemplate.getForObject(WXOpenConstant.JSAPI_TICKET_URL + "?access_token={access_token}&type={type}", WxJsApiTicketRespVo.class, uriVariables);
    if (Objects.isNull(jsApiTicketRespVo) || !Objects.equals(jsApiTicketRespVo.getErrcode(), WXOpenConstant.SUCCESS)) {
      log.warn("wxopen:获取微信全局JSAPI_TICKET失败！{}", jsApiTicketRespVo.getErrmsg());
      return null;
    }
    jsApiTicket = jsApiTicketRespVo.getTicket();
    WxCacheVo cacheVo = new WxCacheVo();
    cacheVo.setCreateTimestamp(new Date().getTime());
    cacheVo.setValue(jsApiTicket);
    cacheVo.setExpiresIn(jsApiTicketRespVo.getExpiresIn());
    this.wxCache.set(cacheKey, JSONObject.toJSONString(cacheVo));
    log.debug("wxopen:jsApiTicket:{}", jsApiTicket);
    return jsApiTicket;
  }

  //此接口建议每天晚上刷新
  @Override
  public String findApiAuthorizerToken(String appid) {
    log.info("wxopen:jsApiTicket:wxProperties:{}", appid);
    String redisKey = String.format("OPEN_CACHE_COMPONENT_AUTHORIZER_ACCESS_TOKEN_KEY_FORMAT", appid);
    boolean locked = false;
    try {
      locked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
      Validate.isTrue(locked, "缓存ApiAuthorizerToken繁忙请稍后再试！");
      String authorizerAccessTokenCacheKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_AUTHORIZER_ACCESS_TOKEN_FORMAT, appid);
      String authorizerAccessToken = this.wxCache.get(authorizerAccessTokenCacheKey);
      if (StringUtils.isNotBlank(authorizerAccessToken)) {
        log.debug("wxopen:authorizer_access_token:{}", authorizerAccessToken);
        return authorizerAccessToken;
      }
      WXOpenPlatformAccountProperties wxOpenPlatformAccountProperties = wxAccountProperties.getOpenPlatformAccount();
      Validate.notNull(wxOpenPlatformAccountProperties, "未配置微信第三方平台信息");
      String componentAppid = wxOpenPlatformAccountProperties.getComponentAppid();
      //
      String authorizerRefreshTokenCacheKey = String.format(WXOpenConstant.OPEN_CACHE_COMPONENT_AUTHORIZER_REFRESH_TOKEN_FORMAT, appid);
      String authorizerRefreshToken = this.wxCache.get(authorizerRefreshTokenCacheKey);
      Validate.notBlank("authorizerRefreshToken", "获取第三方平台authorizerRefreshToken失败,请先授权，或者重新授权");

      Map<String, String> uriVariables = new HashMap<>();
      String accessToken = this.getComponentAccessToken();
      Validate.notBlank("componentAccessToken", "获取第三方平台accesstoken失败");
      uriVariables.put("component_access_token", accessToken);
      HttpHeaders headers = new HttpHeaders();
      headers.setContentType(MediaType.APPLICATION_JSON);
      JSONObject postData = new JSONObject();
      postData.put("component_appid", componentAppid);
      postData.put("authorizer_appid", appid);
      postData.put("authorizer_refresh_token", authorizerRefreshToken);
      HttpEntity<String> request = new HttpEntity<String>(postData.toJSONString(), headers);
      JSONObject respVo = this.restTemplate
              .postForObject(WXOpenConstant.COMPONENT_API_AUTHORIZER_TOKEN_URL + "?component_access_token={component_access_token}", request, JSONObject.class, uriVariables);
      if (Objects.isNull(respVo) || StringUtils.isNotBlank(respVo.getString("errcode"))) {
        log.warn("wxopen:获取微信全局authorizer_access_token失败！{}", respVo.getString("errmsg"));
        return null;
      }
      // 授权方接口调用凭据（在授权的公众号或小程序具备API权限时，才有此返回值），也简称为令牌
      authorizerAccessToken = respVo.getString("authorizer_access_token");
      //刷新令牌
      authorizerRefreshToken = respVo.getString("authorizer_refresh_token");
      //
      int expires_in = respVo.getIntValue("expires_in");
      WxCacheVo authorizerAccessTokenCacheVo = new WxCacheVo();
      authorizerAccessTokenCacheVo.setCreateTimestamp(new Date().getTime());
      authorizerAccessTokenCacheVo.setValue(authorizerAccessToken);
      authorizerAccessTokenCacheVo.setExpiresIn(expires_in);
      this.wxCache.set(authorizerAccessTokenCacheKey, JSONObject.toJSONString(authorizerAccessTokenCacheVo));
      //
      WxCacheVo authorizerRefreshTokenCacheVo = new WxCacheVo();
      authorizerRefreshTokenCacheVo.setCreateTimestamp(new Date().getTime());
      authorizerRefreshTokenCacheVo.setValue(authorizerRefreshToken);
      authorizerRefreshTokenCacheVo.setExpiresIn(60 * 60 * 24 * 30);
      this.wxCache.set(authorizerRefreshTokenCacheKey, JSONObject.toJSONString(authorizerRefreshTokenCacheVo));
      log.debug("第三方平台authorizerAccessToken:{}", authorizerAccessToken);
      return authorizerAccessToken;
    } finally {
      if (locked) {
        redisMutexService.unlock(redisKey);
      }
    }
  }


  /**
   * 验证
   *
   * @param componentToken
   * @param signature
   * @param timestamp
   * @param nonce
   * @return
   */
  private boolean checkSignature(String componentToken, String signature, String timestamp,
          String nonce) {
    boolean flag = false;
    if (StringUtils.isAnyBlank(signature, timestamp, nonce)) {
      return flag;
    }
    String signatureValue = "";
    String[] arr = new String[]{componentToken, timestamp, nonce};
    Arrays.sort(arr);
    for (String temp : arr) {
      signatureValue += temp;
    }
    String signatureVerify = null;
    try {
      signatureVerify = DigestUtils.shaHex(signatureValue);
    } catch (Exception e) {
      log.error(e.getMessage(), e);
    }
    if (signature.equals(signatureVerify)) {
      flag = true;
    }
    return flag;
  }

  /**
   * 进制转换
   *
   * @param hash
   * @return
   */
  private String byteToHex(final byte[] hash) {
    Formatter formatter = new Formatter();
    for (byte b : hash) {
      formatter.format("%02x", b);
    }
    String result = formatter.toString();
    formatter.close();
    return result;
  }
}
