package com.biz.crm.cpcnpay.core.strategy.tx;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.cpcnpay.common.http.HttpsConnection;
import com.biz.crm.cpcnpay.common.http.NameValuePair;
import com.biz.crm.cpcnpay.core.common.enums.CpcnRequestType;
import com.biz.crm.cpcnpay.core.config.CpcnConfig;
import com.biz.crm.cpcnpay.core.tx.TxBaseRequest;
import com.biz.crm.cpcnpay.core.tx.TxBaseResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.ObjectUtils;

import javax.validation.ValidatorFactory;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 中金接口调用抽象类
 * 该抽象类实现spring InitializingBean接口，在类属性初始化完成后执行afterPropertiesSet方法
 *
 * @author Keller
 */
@Slf4j
public abstract class AbstractTxStrategy<T extends TxBaseRequest, R extends TxBaseResponse> {
  @Autowired
  protected HttpsConnection httpsConnection;
  @Autowired
  protected CpcnConfig cpcnConfig;
  @Autowired
  protected ValidatorFactory validatorFactory;

  /**
   * 策略统一执行处理方法
   *
   * @param request 请求封装报文
   * @return TxBaseResponse 返回响应报文
   */
  public R handler(T request) {
    request.setCpcnConfig(cpcnConfig);
    request.setValidator(validatorFactory.getValidator());
    // 执行属性验证
    request.validate();
    try {
      request.process();
    } catch (Exception e) {
      log.error("数据封装异常", e);
      throw new RuntimeException("数据封装异常", e);
    }
    // 继续接口请求业务
    R response = null;
    log.error("请求中金请求参数为{}", JSONObject.toJSONString(request));
    // 调用接口请求
    String[] resp = this.send(request.getRequestMessage(), request.getRequestSignature(), request.getRequestPlainText());
    if (ObjectUtils.isEmpty(resp) || resp.length <= 1) {
      throw new IllegalArgumentException("返回数据不符合格式");
    }
    // 检查策略处理器
    Validate.notBlank(this.getCode(), "接口编号不能为空");
    Validate.notNull(this.getResponse(), "响应报文类不能为空");
    Class<R> responseType = this.getResponse();
    Validate.notNull(responseType, "responseType获取类型为空");
    try {
      Constructor<R> constructor = responseType.getConstructor(String.class, String.class, CpcnConfig.class);
      response = constructor.newInstance(resp[0], resp[1], cpcnConfig);
      log.error("请求中金返回参数为{}", JSONObject.toJSONString(response));
    } catch (Exception e) {
      log.error("处理数据出现错误", e);
    }
    Validate.notNull(response, "获取数据为空");
    return response;
  }

  /**
   * 请求中金接口
   *
   * @param requestMessage   请求的信息
   * @param requestSignature 签名信息
   * @param xml              组成的xml
   * @return 结果
   */
  protected String[] send(String requestMessage, String requestSignature, String xml) {
    String responseText;
    String[] response = new String[2];
    // 请求参数初始化
    String url = getRequestUrl();
    List<NameValuePair> list = new ArrayList<>();
    NameValuePair message = new NameValuePair("message", requestMessage);
    NameValuePair signature = new NameValuePair("signature", requestSignature);
    list.add(message);
    list.add(signature);
    // 调试信息
    log.error("请求的地址=" + url + "]");
    log.error("message=[" + requestMessage + "]");
    log.error("signature=[" + requestSignature + "]");
    log.error(String.format("请求中金数据:[%s],xml为:[%s],url:[%s]", requestMessage, xml, url));
    try {
      responseText = httpsConnection.execute(url, list);
    } catch (Exception e) {
      log.error(String.format("请求中金出现错误,请求数据为%s", xml), e);
      throw new RuntimeException("支付接口请求错误，请联系平台管理员！");
    }
    int index = responseText.indexOf(',');
    if (index > 0) {
      response[0] = responseText.substring(0, index);
      response[1] = responseText.substring(index + 1, responseText.length());
    } else {
      log.error("错误的请求返回：{}", responseText);
      String errorMessage = "响应数据格式不正确";
      throw new RuntimeException(errorMessage);
    }
    log.error(String.format("请求中金返回原始数据为:[%s]", Arrays.toString(response)));
    return response;
  }

  /**
   * 获取请求url地址
   */
  protected String getRequestUrl() {
    CpcnRequestType type = CpcnRequestType.getByCode(this.getCode());
    switch (type.getUrlType()) {
      case PAYMENT_TX:
        return cpcnConfig.getTxUrl();
      case PAYMENT_PAYMENT:
        return cpcnConfig.getPaymentUrl();
      case GATEWAY4FILE:
        return cpcnConfig.getGateway4fileUrl();
      case GATEWAY4AGGREGATE_PAYMENT:
        return cpcnConfig.getGateway4aggregatePaymentUrl();
      case GATEWAY4AGGREGATE_TX:
        return cpcnConfig.getGateway4aggregateTxUrl();
      default:
        throw new RuntimeException("无匹配的URL地址，请检查配置信息");
    }
  }

  /**
   * 返回处理的编号
   */
  public abstract String getCode();

  /**
   * 返回响应的信息类
   */
  public abstract Class<R> getResponse();

}
