package com.biz.crm.cpcnpay.core.tx;

import com.biz.crm.cpcnpay.common.utils.PayUtils;
import com.biz.crm.cpcnpay.common.utils.XmlUtils;
import com.biz.crm.cpcnpay.core.config.CpcnConfig;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.springframework.util.Base64Utils;
import org.springframework.util.CollectionUtils;
import org.w3c.dom.Document;

import javax.validation.ConstraintViolation;
import javax.validation.Validator;
import javax.validation.constraints.NotBlank;
import java.nio.charset.StandardCharsets;
import java.util.Set;

/**
 * @author Keller
 */
@Getter
@Setter
@ToString
public abstract class TxBaseRequest {
  /**
   * 交易编码
   * 如：4600(影印件采集）
   */
  @NotBlank(message = "交易编码不能诶空")
  protected String txCode;
  /**
   * 请求明文
   */
  protected String requestPlainText;
  /**
   * 请求消息
   */
  protected String requestMessage;
  /**
   * 请求签名
   */
  protected String requestSignature;
  /**
   * 签名算法
   */
  protected String signAlgorithm;
  /**
   * 签名序列号
   */
  protected String signSN;
  /**
   * 加密序列号
   */
  protected String encryptSN;
  /**
   *
   */
  protected String dgtlEnvlp;
  /**
   * 机构配置信息
   */
  protected CpcnConfig cpcnConfig;
  /**
   * 属性验证方法
   */
  protected Validator validator;

  /**
   * 执行请求接口
   */
  public abstract void process() throws Exception;

  /**
   * 执行属性校验
   */
  public void validate() {
    Set<ConstraintViolation<TxBaseRequest>> violationSet = validator.validate(this);
    if(!CollectionUtils.isEmpty(violationSet)){
      throw new IllegalArgumentException(violationSet.iterator().next().getMessage());
    }
  }

  /**
   * 生成请求信息与签名
   *
   * @param document
   */
  protected void postProcess(Document document) throws Exception {
    this.requestPlainText = XmlUtils.createPrettyFormat(document).trim();
    byte[] data = this.requestPlainText.getBytes(StandardCharsets.UTF_8);
    this.requestMessage = new String(Base64Utils.encode(data));
    byte[] signature = cpcnConfig.getSigner().sign(data);
    this.requestSignature = PayUtils.bytes2hex(signature);
  }
}
