package com.biz.crm.cps.business.signtask.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.cps.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.cps.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.cps.business.signtask.local.entity.SignTask;
import com.biz.crm.cps.business.signtask.local.entity.SignTaskChannelRelationship;
import com.biz.crm.cps.business.signtask.local.entity.SignTaskCusRelationship;
import com.biz.crm.cps.business.signtask.local.entity.SignTaskMaterialRelationship;
import com.biz.crm.cps.business.signtask.local.entity.SignTaskOrgRelationship;
import com.biz.crm.cps.business.signtask.local.entity.SignTaskTagRelationship;
import com.biz.crm.cps.business.signtask.local.entity.SignTaskTerminalRelationship;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskChannelRelationshipRepository;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskCusRelationshipRepository;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskMaterialRelationshipRepository;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskOrgRelationshipRepository;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskRepository;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskTagRelationshipRepository;
import com.biz.crm.cps.business.signtask.local.repository.SignTaskTerminalRelationshipRepository;
import com.biz.crm.cps.business.signtask.local.service.SignTaskService;
import com.biz.crm.cps.business.signtask.local.utils.SignTaskActivityUtils;
import com.biz.crm.cps.business.signtask.sdk.constant.SignTaskCodeConstant;
import com.biz.crm.cps.business.signtask.sdk.dto.SignTaskDto;
import com.biz.crm.cps.business.signtask.sdk.event.SignTaskEventListener;
import com.biz.crm.cps.business.signtask.sdk.vo.SignTaskVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * @description: 活动签约配置
 * @author: rentao
 * @date: 2021/12/20 17:39
 */
@Service
@Slf4j
public class SignTaskServiceImpl<T> implements SignTaskService {

  @Autowired
  private SignTaskRepository signTaskRepository;
  @Autowired
  private SignTaskOrgRelationshipRepository signTaskOrgRelationshipRepository;
  @Autowired
  private SignTaskChannelRelationshipRepository signTaskChannelRelationshipRepository;
  @Autowired
  private SignTaskTagRelationshipRepository signTaskTagRelationshipRepository;
  @Autowired
  private SignTaskTerminalRelationshipRepository signTaskTerminalRelationshipRepository;
  @Autowired
  private SignTaskMaterialRelationshipRepository signTaskMaterialRelationshipRepository;
  @Autowired
  private SignTaskCusRelationshipRepository signTaskCusRelationshipRepository;
  @Autowired(required = false)
  private List<SignTaskEventListener> signTaskEventListeners;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<SignTask> findByConditions(Pageable pageable, SignTaskDto signTaskDto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (signTaskDto == null) {
      signTaskDto = new SignTaskDto();
    }
    Page<SignTask> signTaskPage = this.signTaskRepository.findByConditions(pageable, signTaskDto);
    return signTaskPage;
  }

  @Override
  @Transactional
  public SignTask create(SignTask signTask) {
    this.createForm(signTask);
    this.signTaskRepository.save(signTask);
    this.createDetails(signTask);
    if (!CollectionUtils.isEmpty(signTaskEventListeners)) {
      SignTaskVo signTaskVo = nebulaToolkitService
          .copyObjectByBlankList(signTask, SignTaskVo.class, HashSet.class, ArrayList.class);
      for (SignTaskEventListener event : signTaskEventListeners) {
        event.onCreate(signTaskVo);
      }
    }
    return signTask;
  }

  @Override
  @Transactional
  public SignTask update(SignTask signTask) {
    //修改签约任务配置
    SignTask signTaskOld = this.findById(signTask.getId());
    Validate.notNull(signTaskOld, "修改操作时，未查询到签约任务配置信息！");
    signTask = this.updateDetails(signTask, signTaskOld);
    if (!CollectionUtils.isEmpty(signTaskEventListeners)) {
      SignTaskVo signTaskVo = nebulaToolkitService
          .copyObjectByBlankList(signTask, SignTaskVo.class, HashSet.class, ArrayList.class);
      for (SignTaskEventListener event : signTaskEventListeners) {
        event.onUpdate(signTaskVo);
      }
    }
    return signTask;
  }

  @Override
  public SignTask findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.signTaskRepository.getById(id);
  }

  @Override
  public SignTask findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    SignTask signTask = this.signTaskRepository.findById(id);
    if (Objects.isNull(signTask)) {
      return null;
    }
    this.transformSet(signTask);
    return signTask;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选中要操作的数据");
    this.signTaskRepository.updateEnableStatusByIds(EnableStatusEnum.ENABLE, ids);
    if (!CollectionUtils.isEmpty(signTaskEventListeners)) {
      List<SignTask> terminals = this.signTaskRepository.listByIds(ids);
      List<SignTaskVo> voList = (List<SignTaskVo>) this.nebulaToolkitService
          .copyCollectionByWhiteList(terminals, SignTask.class, SignTaskVo.class, HashSet.class,
              ArrayList.class);
      for (SignTaskEventListener event : signTaskEventListeners) {
        event.onEnable(voList);
      }
    }
  }

  @Transactional
  @Override
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选中要操作的数据");
    this.signTaskRepository.updateEnableStatusByIds(EnableStatusEnum.DISABLE, ids);
    if (!CollectionUtils.isEmpty(signTaskEventListeners)) {
      List<SignTask> terminals = this.signTaskRepository.listByIds(ids);
      List<SignTaskVo> voList = (List<SignTaskVo>) this.nebulaToolkitService
          .copyCollectionByWhiteList(terminals, SignTask.class, SignTaskVo.class, HashSet.class,
              ArrayList.class);
      for (SignTaskEventListener event : signTaskEventListeners) {
        event.onDisable(voList);
      }
    }
  }

  /**
   * 新增数据填充
   *
   * @param signTask
   * @author rentao
   * @date
   */
  private SignTask createForm(SignTask signTask) {
    Validate.notNull(signTask, "进行当前操作时，信息对象必须传入!!");
    SignTaskActivityUtils.createValidation(signTask);
    Date now = new Date();
    signTask.setCreateTime(now);
    signTask.setModifyTime(now);
    if (Objects.nonNull(generateCodeService)) {
      signTask.setSignTaskCode(
          generateCodeService.generateCode(SignTaskCodeConstant.SIGN_TASK_CODE, 1).get(0));
    }
    signTask.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    signTask.setTenantCode(TenantUtils.getTenantCode());
    signTask.setCreateAccount(loginUserService.getLoginAccountName());
    signTask.setModifyAccount(loginUserService.getLoginAccountName());
    return signTask;
  }

  /**
   * 修改签约任务配置详情，以及修改其子表信息
   *
   * @param signTask
   * @param signTaskOld
   * @author rentao
   * @date
   */
  private SignTask updateDetails(SignTask signTask, SignTask signTaskOld) {
    //修改签约任务配置
    signTask.setId(signTaskOld.getId());
    signTask.setModifyAccount(loginUserService.getLoginAccountName());
    signTask.setModifyTime(new Date());
    SignTaskActivityUtils.updateValidation(signTask);
    this.signTaskRepository.updateById(signTask);
    String signTaskCode = signTaskOld.getSignTaskCode();
    //修改关联组织
    this.signTaskOrgRelationshipRepository.deleteBySignTaskCode(signTaskCode);
    Set<SignTaskOrgRelationship> signTaskOrgRelationships = signTask.getSignTaskOrgRelationships();
    if (!CollectionUtils.isEmpty(signTaskOrgRelationships)) {
      signTaskOrgRelationships.forEach(
          signTaskOrgRelationship -> {
            signTaskOrgRelationship.setSignTaskCode(signTaskCode);
            signTaskOrgRelationship.setId(null);
          });
      this.signTaskOrgRelationshipRepository.saveBatch(signTaskOrgRelationships);
    }
    //修改关联标签
    this.signTaskTagRelationshipRepository.deleteBySignTaskCode(signTaskCode);
    Set<SignTaskTagRelationship> signTaskTagRelationships = signTask.getSignTaskTagRelationships();
    if (!CollectionUtils.isEmpty(signTaskTagRelationships)) {
      signTaskTagRelationships.forEach(
          signTaskTagRelationship -> {
            signTaskTagRelationship.setSignTaskCode(signTaskCode);
            signTaskTagRelationship.setId(null);
          });
      this.signTaskTagRelationshipRepository.saveBatch(signTaskTagRelationships);
    }
    //修改关联渠道
    this.signTaskChannelRelationshipRepository.deleteBySignTaskCode(signTaskCode);
    Set<SignTaskChannelRelationship> signTaskChannelRelationships = signTask
        .getSignTaskChannelRelationships();
    if (!CollectionUtils.isEmpty(signTaskChannelRelationships)) {
      signTaskChannelRelationships
          .forEach(channelRelationship -> {
            channelRelationship.setSignTaskCode(signTaskCode);
            channelRelationship.setId(null);
          });
      this.signTaskChannelRelationshipRepository.saveBatch(signTaskChannelRelationships);
    }
    //保存指定终端关联
    this.signTaskTerminalRelationshipRepository.deleteBySignTaskCode(signTaskCode);
    Set<SignTaskTerminalRelationship> signTaskTerminalRelationships = signTask
        .getSignTaskTerminalRelationships();
    if (!CollectionUtils.isEmpty(signTaskTerminalRelationships)) {
      signTaskTerminalRelationships.forEach(
          signTaskTerminalRelationship -> {
            signTaskTerminalRelationship.setSignTaskCode(signTaskCode);
            signTaskTerminalRelationship.setId(null);
          });
      this.signTaskTerminalRelationshipRepository.saveBatch(signTaskTerminalRelationships);
    }
    //保存物料关联
    this.signTaskMaterialRelationshipRepository.deleteBySignTaskCode(signTaskCode);
    Set<SignTaskMaterialRelationship> signTaskMaterialRelationships = signTask
        .getSignTaskMaterialRelationships();
    if (!CollectionUtils.isEmpty(signTaskMaterialRelationships)) {
      signTaskMaterialRelationships.forEach(
          signTaskMaterialRelationship -> {
            signTaskMaterialRelationship.setSignTaskCode(signTaskCode);
            signTaskMaterialRelationship.setId(null);
          });
      this.signTaskMaterialRelationshipRepository.saveBatch(signTaskMaterialRelationships);
    }
    //保存指定终端关联
    this.signTaskCusRelationshipRepository.deleteBySignTaskCode(signTaskCode);
    Set<SignTaskCusRelationship> signTaskCusRelationships = signTask
        .getSignTaskCusRelationships();
    if (!CollectionUtils.isEmpty(signTaskCusRelationships)) {
      signTaskCusRelationships.forEach(
          signTaskCusRelationship -> {
            signTaskCusRelationship.setSignTaskCode(signTaskCode);
            signTaskCusRelationship.setId(null);
          });
      this.signTaskCusRelationshipRepository.saveBatch(signTaskCusRelationships);
    }
    return signTask;
  }

  /**
   * 新增签约任务配置详情
   *
   * @param signTask
   * @author rentao
   * @date
   */
  private void createDetails(SignTask signTask) {
    //保存关联组织
    if (!CollectionUtils.isEmpty(signTask.getSignTaskOrgRelationships())) {
      signTask.getSignTaskOrgRelationships().forEach(
          templateOrgRelationship -> {
            templateOrgRelationship.setSignTaskCode(signTask.getSignTaskCode());
            templateOrgRelationship.setId(null);
          }
      );
      this.signTaskOrgRelationshipRepository.saveBatch(signTask.getSignTaskOrgRelationships());
    }
    //保存关联标签
    if (!CollectionUtils.isEmpty(signTask.getSignTaskTagRelationships())) {
      signTask.getSignTaskTagRelationships()
          .forEach(templateTag -> {
            templateTag.setSignTaskCode(signTask.getSignTaskCode());
            templateTag.setId(null);
          });
      this.signTaskTagRelationshipRepository.saveBatch(signTask.getSignTaskTagRelationships());
    }
    //保存关联渠道
    if (!CollectionUtils.isEmpty(signTask.getSignTaskChannelRelationships())) {
      signTask.getSignTaskChannelRelationships().forEach(signTaskChannelRelationship -> {
        signTaskChannelRelationship.setSignTaskCode(signTask.getSignTaskCode());
        signTaskChannelRelationship.setId(null);
      });
      this.signTaskChannelRelationshipRepository
          .saveBatch(signTask.getSignTaskChannelRelationships());
    }
    //保存指定终端关联
    if (!CollectionUtils.isEmpty(signTask.getSignTaskTerminalRelationships())) {
      signTask.getSignTaskTerminalRelationships().forEach(
          terminalRelationship -> {
            terminalRelationship.setSignTaskCode(signTask.getSignTaskCode());
            terminalRelationship.setId(null);
          });
      this.signTaskTerminalRelationshipRepository
          .saveBatch(signTask.getSignTaskTerminalRelationships());
    }
    //保存物料关联
    if (!CollectionUtils.isEmpty(signTask.getSignTaskMaterialRelationships())) {
      signTask.getSignTaskMaterialRelationships().forEach(
          materialRelationship -> {
            materialRelationship.setSignTaskCode(signTask.getSignTaskCode());
            materialRelationship.setId(null);
          });
      this.signTaskMaterialRelationshipRepository
          .saveBatch(signTask.getSignTaskMaterialRelationships());
    }
    //保存客户关联
    if (!CollectionUtils.isEmpty(signTask.getSignTaskCusRelationships())) {
      signTask.getSignTaskCusRelationships().forEach(
          cusRelationship -> {
            cusRelationship.setSignTaskCode(signTask.getSignTaskCode());
            cusRelationship.setId(null);
          });
      this.signTaskCusRelationshipRepository.saveBatch(signTask.getSignTaskCusRelationships());
    }
  }

  /**
   *
   * 明细list转set
   * @param signTask
   * @author rentao
   * @date
   */
  private void transformSet(SignTask signTask){
    List<SignTaskOrgRelationship> signTaskOrgRelationships = signTaskOrgRelationshipRepository
        .findBySignTaskCode(signTask.getSignTaskCode());
    List<SignTaskChannelRelationship> signTaskChannelRelationships = signTaskChannelRelationshipRepository
        .findBySignTaskCode(signTask.getSignTaskCode());
    List<SignTaskTagRelationship> signTaskTagRelationships = signTaskTagRelationshipRepository
        .findBySignTaskCode(signTask.getSignTaskCode());
    List<SignTaskTerminalRelationship> signTaskTerminalRelationships = signTaskTerminalRelationshipRepository
        .findBySignTaskCode(signTask.getSignTaskCode());
    List<SignTaskCusRelationship> signTaskCusRelationships = signTaskCusRelationshipRepository
        .findBySignTaskCode(signTask.getSignTaskCode());
    List<SignTaskMaterialRelationship> signTaskMaterialRelationships = signTaskMaterialRelationshipRepository
        .findBySignTaskCode(signTask.getSignTaskCode());
    signTask.setSignTaskOrgRelationships(CollectionUtils.isEmpty(signTaskOrgRelationships) ? null
        : CollectionUtil.newHashSet(false, signTaskOrgRelationships.iterator()));
    signTask.setSignTaskChannelRelationships(CollectionUtils.isEmpty(signTaskChannelRelationships) ? null
        : CollectionUtil.newHashSet(false, signTaskChannelRelationships.iterator()));
    signTask.setSignTaskTagRelationships(CollectionUtils.isEmpty(signTaskTagRelationships) ? null
        : CollectionUtil.newHashSet(false, signTaskTagRelationships.iterator()));
    signTask.setSignTaskTerminalRelationships(CollectionUtils.isEmpty(signTaskTerminalRelationships) ? null
        : CollectionUtil.newHashSet(false, signTaskTerminalRelationships.iterator()));
    signTask.setSignTaskCusRelationships(CollectionUtils.isEmpty(signTaskCusRelationships) ? null
        : CollectionUtil.newHashSet(false, signTaskCusRelationships.iterator()));
    signTask.setSignTaskMaterialRelationships(CollectionUtils.isEmpty(signTaskMaterialRelationships) ? null
        : CollectionUtil.newHashSet(false, signTaskMaterialRelationships.iterator()));
  }
}
