package com.biz.crm.business.common.local.config;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.data.redis.connection.Message;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.listener.KeyExpirationEventMessageListener;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;

/**
 * 向redis注册系统服务
 *
 * 使用场景: 系统的基础服务例如导入导出的任务，错误日志等都是按照服务名进行区分任务，
 * 故需要将当前系统上线的服务名存入到Redis中去，方便基础服务时使用。
 * @author HanJiaJun
 * @describe:
 * @createTime 2022年05月12日 15:26:00
 */
@Component
public class RegistrarServerMsgHandler extends KeyExpirationEventMessageListener implements ApplicationRunner {

  @Value("${spring.application.name:}")
  private String systemName;
  @Autowired
  private RedisTemplate<String , Object> redisTemplate;

  /**
   * 创建监听器
   *
   * @param listenerContainer must not be {@literal null}.
   */
  public RegistrarServerMsgHandler(RedisMessageListenerContainer listenerContainer) {
    super(listenerContainer);
  }

  /**
   * 过期检测，判断当前过期的服务名是否与当前yml中的服务名是否相等，相等则重新添加到redis
   * @param message
   * @param pattern
   */
  @Override
  public void onMessage(Message message, byte[] pattern) {
    //过期的KEY
    String key = message.toString();
    if (key.equals("springboot-system-".concat(systemName))) {
      //重新添加
      redisTemplate.opsForValue().set(key,systemName,6,TimeUnit.HOURS);
    }
  }

  /**
   * spring启动时执行方法，将当前启动的服务名注册到redis中
   * @param args
   * @throws Exception
   */
  @Override
  public void run(ApplicationArguments args) throws Exception {
    String key = "springboot-system-" + systemName;
    if(redisTemplate.hasKey(key)){
      redisTemplate.expire(key,6,TimeUnit.HOURS);
    }
    //Redis保存当前启动的服务名称
    redisTemplate.opsForValue().set(key, systemName,6, TimeUnit.HOURS);
  }
}
