package com.biz.crm.business.common.sdk.service;

import com.biz.crm.business.common.local.config.SignProperties;
import com.biz.crm.business.common.sdk.utils.Md5Utils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * 签名服务，主要用于在服务之间数据传输时的数据签名认证，
 * 提供数据签名和签名认证
 * 签名的生成依赖于配置文件中apps的应用配置
 * 目前只支持单个文本内容的签名
 * 注：为保证数据在服务之间传递的一致性，json数据皆用文本内容传输，用文本内容接收再解析对象
 * sign:
 *   appId: test // 当前系统的appId标识
 *   apps:
 *     test:
 *      appKey: 123456
 * 如上的配置中，test为appId，appKey为app对应的加密key
 * @Author: Paul Chan
 * @Date: 2020/8/3 11:11
 */
public class SignService {

  @Autowired
  private SignProperties signProperties;

  /**
   * 获取当前系统的appId
   * @return
   */
  public String getCurrentAppId() {
    return signProperties.getAppId();
  }

  /**
   * 根据输入的内容生成签名，输入的内容不能为null,但可以是空字符串</br>
   * 目前的签名生成方式为appId+content+appKey再md5加密，然后将md5的二进制HEX字符串输出</br>
   * appId会使用当前系统配置的appId标识
   * @param content
   * @return
   */
  public String sign(String content) {
    String appId = signProperties.getAppId();
    Validate.notBlank(appId, "当前系统未配置appId");
    return this.sign(appId, content);
  }

  /**
   * 根据输入的内容生成签名，输入的内容不能为null,但可以是空字符串</br>
   * appId是配置文件中配置的app的id，</br>
   * 目前的签名生成方式为appId+content+appKey再md5加密，然后将md5的二进制HEX字符串输出</br>
   * @param appId
   * @param content
   * @return
   */
  public String sign(String appId, String content) {
    Validate.notBlank(appId, "appId不能为空");
    Validate.notNull(content, "签名数据内容不能为null");
    SignProperties.AppInfo app = signProperties.getApp(appId);
    Validate.notNull(app, "未找到配置的app:%s", appId);
    String signContent = StringUtils.join(appId, content, app.getAppKey());
    return Md5Utils.encode(signContent, Md5Utils.EncodeType.HEX);
  }

  /**
   * 验证数据的签名是否正确</br>
   * 首先会调用sign方法将输入的content进行签名，然后将content生成的签名与传入的签名进行比较</br>
   * 注：传入的content是源数据，不是进行appId+content+appKey拼装后的数据
   * @param appId
   * @param content
   * @param sign
   */
  public void verify(String appId, String content, String sign) {
    Validate.notBlank(sign, "验证的签名不能为空");
    String contentSign = this.sign(appId, content);
    Validate.isTrue(contentSign.equals(sign), "签名不正确");
  }

  /**
   * 对k-v参数进行签名，根据key的升序排序后，再去除空值的参数后根据k=v&k1=k2方式拼接，再将拼接后的字符串进行签名
   * @param params
   * @return
   */
  public String sign(Map<String, String> params) {
    String appId = signProperties.getAppId();
    Validate.notBlank(appId, "当前系统未配置appId");
    return this.sign(appId, params);
  }

  /**
   * 对k-v参数进行签名，根据key的升序排序后，再去除空值的参数后根据k=v&k1=k2方式拼接，再将拼接后的字符串进行签名
   * @param appId
   * @param params
   * @return
   */
  public String sign(String appId, Map<String, String> params) {
    Validate.notBlank(appId, "appId不能为空");
    Validate.notNull(params, "数据内容不能为null");
    SignProperties.AppInfo app = signProperties.getApp(appId);
    Validate.notNull(app, "未找到配置的app:%s", appId);
    TreeMap<String, String> treeParams = new TreeMap<>(params);
    String paramsContent = this.getParamsContent(treeParams);
    return this.sign(appId, paramsContent);
  }

  /**
   * 验证数据的签名是否正确</br>
   * 对k-v参数进行签名，根据key的升序排序后，再去除空值的参数后根据k=v&k1=k2方式拼接，再将拼接后的字符串进行签名
   * 将得到的签名和传入的签名进行比较
   * @param appId
   * @param params
   * @param sign
   */
  public void verify(String appId, Map<String, String> params, String sign) {
    Validate.notNull(params, "数据内容不能为null");
    String contentSign = this.sign(appId, params);
    Validate.isTrue(contentSign.equals(sign), "签名不正确");
  }

  /**
   * 将参数map转化为内容,根据key的升序排序后，再去除空值的参数后根据k=v&k1=k2方式拼接
   * @param treeParams
   * @return
   */
  private String getParamsContent(TreeMap<String, String> treeParams) {
    List<String> paramList = new ArrayList<>();
    treeParams.forEach((k, v) -> {
      if(StringUtils.isNotBlank(v) && !"sign".equals(k)) {
        paramList.add(StringUtils.join(k, "=", v));
      }
    });
    return StringUtils.join(paramList, "&");
  }

}
