package com.biz.crm.business.common.sdk.utils.wx.open;

import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.biz.crm.business.common.sdk.utils.wx.open.WxOpenUrlConstant.CREATE_QRCODE_URL;
import static com.biz.crm.business.common.sdk.utils.wx.open.WxOpenUrlConstant.CREATE_WXACODE_UNLIMITED_URL;
import static com.biz.crm.business.common.sdk.utils.wx.open.WxOpenUrlConstant.CREATE_WXACODE_URL;


/**
 * 微信第三方小程序微信二维码相关工具类
 */
public class WxOpenQrcodeProcessor {
  private static final Logger LOGGER = LoggerFactory.getLogger(WxOpenQrcodeProcessor.class);

  /**
   * Htt调用执行器
   */
  private final WxOpenHttpExecute wxOpenHttpExecutor;
  private final WxOpenServiceProvision wxOpenServiceProvision;

  WxOpenQrcodeProcessor(WxOpenServiceProvision wxOpenServiceProvision, WxOpenHttpExecute wxOpenHttpExecutor) {
    this.wxOpenServiceProvision = wxOpenServiceProvision;
    this.wxOpenHttpExecutor = wxOpenHttpExecutor;
  }

  /**
   * 获取第三方小程序二维码图片字节码
   *
   * @param path 小程序路径
   * @return
   */
  public byte[] createQrcode(String path) {
    Validate.notBlank(path, "获取第三方小程序二维码时,小程序跳转路径不能为空！");
    return this.createQrcode(path, 430);
  }

  /**
   * 接口C: 获取小程序页面二维码.
   * <pre>
   * 适用于需要的码数量较少的业务场景
   * 通过该接口，仅能生成已发布的小程序的二维码。
   * 可以在开发者工具预览时生成开发版的带参二维码。
   * 带参二维码只有 100000 个，请谨慎调用。
   * </pre>
   *
   * <p>https://developers.weixin.qq.com/doc/oplatform/Third-party_Platforms/2.0/api/qrcode/getwxacodeunlimit.html</>
   *
   * @param path  不能为空，最大长度 128 字节
   * @param width 默认430 二维码的宽度
   * @return 文件内容字节数组
   */
  public byte[] createQrcode(String path, Integer width) {
    Validate.notBlank(path, "获取第三方小程序二维码图片时，跳转路径不能为空！！！");
    Validate.isTrue(path.length() <= 128, "小程序页面路径: %s , 超出最大长度不能超过 128 字节！！！", path);
    Validate.notNull(width, "获取第三方小程序二维码图片时，图片大小不能为空！！！");
    Validate.isTrue(width >= 280 && width <= 1280, "二维码的宽度 %s 超出限制：最小 280px，最大 1280px！！！", width);

    HashMap<Object, Object> dataMap = new HashMap<>();
    dataMap.put("path", path);
    dataMap.put("width", width);
    return this.createQrcodeOnExecute(CREATE_QRCODE_URL, dataMap);
  }

  /**
   * 接口B: 获取小程序码（永久有效、数量暂无限制）.
   * <pre>
   * 通过该接口生成的小程序码，永久有效，数量暂无限制。
   * 用户扫描该码进入小程序后，将统一打开首页，开发者需在对应页面根据获取的码中 scene 字段的值，再做处理逻辑。
   * 使用如下代码可以获取到二维码中的 scene 字段的值。
   * 调试阶段可以使用开发工具的条件编译自定义参数 scene=xxxx 进行模拟，开发工具模拟时的 scene 的参数值需要进行 urlencode
   * </pre>
   *
   * <p>https://developers.weixin.qq.com/doc/oplatform/Third-party_Platforms/2.0/api/qrcode/getwxacode.html</p>
   *
   * @param scene 最大32个可见字符，只支持数字，大小写英文以及部分特殊字符：!#$&'()*+,/:;=?@-._~，
   *              其它字符请自行编码为合法字符（因不支持%，中文无法使用 urlencode 处理，请使用其他编码方式）
   * @param page  必须是已经发布的小程序页面，例如 "pages/index/index" ,如果不填写这个字段，默认跳主页面
   * @param width 默认430 二维码的宽度
   * @return 文件内容字节数组
   */
  public byte[] createWxaCodeUnlimitBytes(String scene, String page, Integer width) {
    return createWxaCodeUnlimitBytes(scene, page, width, false, 0, 0, 0, false);
  }

  /**
   * 接口B: 获取小程序码（永久有效、数量暂无限制）.
   * <pre>
   * 通过该接口生成的小程序码，永久有效，数量暂无限制。
   * 用户扫描该码进入小程序后，将统一打开首页，开发者需在对应页面根据获取的码中 scene 字段的值，再做处理逻辑。
   * 使用如下代码可以获取到二维码中的 scene 字段的值。
   * 调试阶段可以使用开发工具的条件编译自定义参数 scene=xxxx 进行模拟，开发工具模拟时的 scene 的参数值需要进行 urlencode
   * </pre>
   *
   * @param scene     最大32个可见字符，只支持数字，大小写英文以及部分特殊字符：!#$&'()*+,/:;=?@-._~，
   *                  其它字符请自行编码为合法字符（因不支持%，中文无法使用 urlencode 处理，请使用其他编码方式）
   * @param page      必须是已经发布的小程序页面，例如 "pages/index/index" ,如果不填写这个字段，默认跳主页面
   * @param width     默认430 二维码的宽度
   * @param autoColor 默认false 自动配置线条颜色，如果颜色依然是黑色，则说明不建议配置主色调
   * @param r         autoColor 为 true 时生效，使用 rgb 设置颜色 例如 {"r":"xxx","g":"xxx","b":"xxx"}
   * @param g         autoColor 为 true 时生效，使用 rgb 设置颜色 例如 {"r":"xxx","g":"xxx","b":"xxx"}
   * @param b         autoColor 为 true 时生效，使用 rgb 设置颜色 例如 {"r":"xxx","g":"xxx","b":"xxx"}
   * @param isHyaline 默认false 是否需要透明底色， is_hyaline 为true时，生成透明底色的小程序码
   * @return 文件内容字节数组
   */
  public byte[] createWxaCodeUnlimitBytes(String scene, String page, Integer width, boolean autoColor,
                                          Integer r, Integer g, Integer b, boolean isHyaline) {
    Validate.isTrue(StringUtils.isBlank(page) || !page.startsWith("/"), "路径:【%s】根路径前不要填加 /", page);
    Validate.notBlank(scene, "请求 scene 参数不能为空");

    HashMap<Object, Object> dataMap = new HashMap<>();
    dataMap.put("scene", scene);
    dataMap.put("page", page);
    dataMap.put("width", width);
    dataMap.put("auto_color", autoColor);
    dataMap.put("is_hyaline", isHyaline);
    if (autoColor && Objects.nonNull(r) && Objects.nonNull(g) && Objects.nonNull(b)) {
      dataMap.put("line_color", ImmutableMap.of("r", r, "g", g, "b", b));
    }

    return this.createQrcodeOnExecute(CREATE_WXACODE_UNLIMITED_URL, dataMap);
  }


  /**
   * 接口B: 获取小程序码（永久有效、数量暂无限制）.
   * <pre>
   * 通过该接口生成的小程序码，永久有效，数量暂无限制。
   * 用户扫描该码进入小程序后，将统一打开首页，开发者需在对应页面根据获取的码中 scene 字段的值，再做处理逻辑。
   * 使用如下代码可以获取到二维码中的 scene 字段的值。
   * 调试阶段可以使用开发工具的条件编译自定义参数 scene=xxxx 进行模拟，开发工具模拟时的 scene 的参数值需要进行 urlencode
   * </pre>
   *
   *                  其它字符请自行编码为合法字符（因不支持%，中文无法使用 urlencode 处理，请使用其他编码方式）
   * @param page      必须是已经发布的小程序页面，例如 "pages/index/index" ,如果不填写这个字段，默认跳主页面
   * @param width     默认430 二维码的宽度
   * @return 文件内容字节数组
   */
  public byte[] createWxaCodeBytes(String page, Integer width) {
    return createWxaCodeBytes(page, width, false, 0, 0, 0, false);
  }


  /**
   * 接口B: 获取小程序码（永久有效、数量暂无限制）.
   * <pre>
   * 通过该接口生成的小程序码，永久有效，数量暂无限制。
   * 用户扫描该码进入小程序后，将统一打开首页，开发者需在对应页面根据获取的码中 scene 字段的值，再做处理逻辑。
   * 使用如下代码可以获取到二维码中的 scene 字段的值。
   * 调试阶段可以使用开发工具的条件编译自定义参数 scene=xxxx 进行模拟，开发工具模拟时的 scene 的参数值需要进行 urlencode
   * </pre>
   *
   *                  其它字符请自行编码为合法字符（因不支持%，中文无法使用 urlencode 处理，请使用其他编码方式）
   * @param page      必须是已经发布的小程序页面，例如 "pages/index/index" ,这个字段必填
   * @param width     默认430 二维码的宽度
   * @param autoColor 默认true 自动配置线条颜色，如果颜色依然是黑色，则说明不建议配置主色调
   * @param r         autoColor 为 true 时生效，使用 rgb 设置颜色 例如 {"r":"xxx","g":"xxx","b":"xxx"}
   * @param g         autoColor 为 true 时生效，使用 rgb 设置颜色 例如 {"r":"xxx","g":"xxx","b":"xxx"}
   * @param b         autoColor 为 true 时生效，使用 rgb 设置颜色 例如 {"r":"xxx","g":"xxx","b":"xxx"}
   * @param isHyaline 是否需要透明底色， is_hyaline 为true时，生成透明底色的小程序码
   * @return 文件内容字节数组
   */
  public byte[] createWxaCodeBytes(String page, Integer width, boolean autoColor,
                                          Integer r, Integer g, Integer b, boolean isHyaline) {
    Validate.isTrue(StringUtils.isBlank(page) || !page.startsWith("/"), "路径:【%s】根路径前不要填加 /", page);

    HashMap<Object, Object> dataMap = new HashMap<>();
    dataMap.put("path", page);
    dataMap.put("width", width);
    dataMap.put("auto_color", autoColor);
    dataMap.put("is_hyaline", isHyaline);
    if (autoColor && Objects.nonNull(r) && Objects.nonNull(g) && Objects.nonNull(b)) {
      dataMap.put("line_color", ImmutableMap.of("r", r, "g", g, "b", b));
    }

    return this.createQrcodeOnExecute(CREATE_WXACODE_URL, dataMap);
  }


  /**
   * 获取第三方小程序二维码图片字节码
   *
   * @param url     请求路径
   * @param dataMap 请求参数
   * @return
   */
  public byte[] createQrcodeOnExecute(String url, HashMap<Object, Object> dataMap) {
    final Supplier<HttpRequestBase> request = () -> {
      WxOpenInfo wxOpenInfo = wxOpenServiceProvision.getWxOpenInfo();
      Validate.notNull(wxOpenInfo, "未找到第三方小程序配置信息！");
      Validate.notBlank(wxOpenInfo.getAuthorizerAccessToken(),
              "调用第三方小程序授权token为返回的授权小程序AccessToken信息空");

      String urlToken = String.format(url, wxOpenInfo.getAuthorizerAccessToken());
      HttpPost httpPost = new HttpPost(urlToken);
      httpPost.setHeader("Content-Type", ContentType.APPLICATION_JSON.toString());
      StringEntity stringEntity;
      try {
        stringEntity = new StringEntity(JSONObject.toJSONString(dataMap));
      } catch (UnsupportedEncodingException e) {
        LOGGER.error(e.getMessage(), e);
        throw new UncheckedIOException(e);
      }
      httpPost.setEntity(stringEntity);
      return httpPost;
    };
    final Function<CloseableHttpResponse, byte[]> response = (CloseableHttpResponse httpResponse) -> {
      try (final InputStream inputStream = httpResponse.getEntity().getContent()) {
        if (StringUtils.equalsIgnoreCase(httpResponse.getEntity().getContentType().getValue(), ContentType.APPLICATION_JSON.toString())) {
          String responseStr = EntityUtils.toString(httpResponse.getEntity());
          Validate.isTrue(StringUtils.isBlank(responseStr)
                          || Objects.equals(JSONObject.parseObject(responseStr).get("errcode"), 0),
                  "获取第三方小程序二维码图片失败：【%s】", responseStr);
        }

        return IOUtils.toByteArray(inputStream);
      } catch (IOException e) {
        LOGGER.error(e.getMessage(), e);
        throw new UncheckedIOException(e);
      }
    };
    return wxOpenHttpExecutor.doExecute(request, response);
  }
}
