package com.biz.crm.business.common.sdk.utils.wx.open;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.google.common.util.concurrent.UncheckedExecutionException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 第三方小程序服务提供接口实现
 */
public class WxOpenServiceProvisionImpl implements WxOpenServiceProvision {
  private static final Logger LOGGER = LoggerFactory.getLogger(WxOpenServiceProvisionImpl.class);

  /**
   * Htt调用执行器
   */
  private final WxOpenHttpExecute wxOpenHttpExecutor = new WxOpenHttpExecuteImpl();

  private final InternalWxOpenInfoProcessor internalWxOpenInfoProcessor = new InternalWxOpenInfoProcessor();
  private final WxOpenUserProcessor wxOpenUserProcessor = new WxOpenUserProcessor(this, wxOpenHttpExecutor);
  private final WxOpenQrcodeProcessor wxOpenQrcodeProcessor = new WxOpenQrcodeProcessor(this, wxOpenHttpExecutor);

  private RedisMutexService redisMutexService;
  /**
   * 第三方微信开放平台缓存锁前缀
   */
  private static final String WX_OPEN_INFO_CACHE_LOCK = "wx:open:info:lock:%s:%s";
  /**
   * 第三方微信开放平台缓存
   */
  private static final String WX_OPEN_INFO_CACHE = "wx:open:info:cache";

  public void enableCache(RedisMutexService redisMutexService) {
    this.redisMutexService = redisMutexService;
  }

  /**
   * 获取第三方小程序信息
   *
   * @return
   */
  @Override
  public WxOpenInfo getWxOpenInfo() {
    InternalWxOpenInfoProcessor.InternalWxOpenInfoParam param = internalWxOpenInfoProcessor.getInternalWxOpenInfoParam();
    Validate.notBlank(param.getTenantCode(), "配置的第三方小程序的租户信息为空");
    Validate.notNull(param.getAppletType(), "配置的第三方小程序的小程序类型信息为空");

    if (Objects.isNull(redisMutexService)) {
      WxOpenInfo wxOpenInfo = getWxOpenParams(param);
      Validate.notNull(wxOpenInfo, "调用第三方小程序授权token为返回信息为空！");
      Validate.notBlank(wxOpenInfo.getAuthorizerAppid(), "调用第三方小程序授权token为返回的APPID信息为空");
      Validate.notBlank(wxOpenInfo.getComponentAppid(), "调用第三方小程序授权token为返回的第三方APPID信息为空");
      Validate.notBlank(wxOpenInfo.getComponentAccessToken(), "调用第三方小程序授权token为返回的第三方AccessToken信息为空");
      Validate.notNull(wxOpenInfo.getEndTime(), "调用第三方小程序授权token为返回的第三方AccessToken过期时间信息为空");
      Validate.isTrue(wxOpenInfo.getEndTime() > 0, "第三方小程序授权token为返回的AccessToken已过期，请重试！！！");

      return wxOpenInfo;
    }

    String mapKey = StringUtils.join(param.getTenantCode(), "-", param.getAppletType());
    String cache = redisMutexService.getMCode(WX_OPEN_INFO_CACHE, mapKey);
    if (StringUtils.isNotBlank(cache)) {
      return JSONObject.parseObject(cache, WxOpenInfo.class);
    }
    String lockKey = String.format(WX_OPEN_INFO_CACHE_LOCK, param.getTenantCode(), param.getAppletType());
    try {
      // 此处的锁定时间应该打印HTTP的调用链接时间
      boolean locked = redisMutexService.tryLock(lockKey, TimeUnit.MILLISECONDS, 15000);
      if (locked) {
        cache = redisMutexService.getMCode(WX_OPEN_INFO_CACHE, mapKey);
        if (StringUtils.isNotBlank(cache)) {
          return JSONObject.parseObject(cache, WxOpenInfo.class);
        }
        // 不存在或者已经过期则需要重新拉去内部系统第三方小程序信息
        WxOpenInfo wxOpenInfo = getWxOpenParams(param);
        Validate.notNull(wxOpenInfo, "调用第三方小程序授权token为返回信息为空！");
        Validate.notBlank(wxOpenInfo.getComponentAppid(), "调用第三方小程序授权token为返回的第三方APPID信息为空");
        Validate.notBlank(wxOpenInfo.getComponentAccessToken(), "调用第三方小程序授权token为返回的第三方AccessToken信息为空");
        Validate.notBlank(wxOpenInfo.getAuthorizerAppid(), "调用第三方小程序授权token为返回的授权APPID信息为空");
        Validate.notBlank(wxOpenInfo.getAuthorizerAccessToken(), "调用第三方小程序授权token为返回的授权AccessToken信息为空");
        Validate.notNull(wxOpenInfo.getEndTime(), "调用第三方小程序授权token为返回的AccessToken过期时间信息为空");
        Validate.isTrue(wxOpenInfo.getEndTime() > 0, "第三方小程序授权token为返回的AccessToken已过期，请重试！！！");

        redisMutexService.setMCode(WX_OPEN_INFO_CACHE, mapKey, JSONObject.toJSONString(wxOpenInfo), wxOpenInfo.getEndTime() * 1000);
        return wxOpenInfo;
      }
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      throw new UncheckedExecutionException(e);
    } finally {
      if (redisMutexService.islock(lockKey)) {
        redisMutexService.unlock(lockKey);
      }
    }

    return null;
  }

  /**
   * 调用内部系统获取第三方小程序信息
   *
   * @param param 内部系统第三方小程序配置信息
   * @return
   */
  public WxOpenInfo getWxOpenParams(InternalWxOpenInfoProcessor.InternalWxOpenInfoParam param) {
    WxOpenInfo wxOpenInfo = wxOpenHttpExecutor.doExecute(() ->
                    new HttpGet(String.format(param.getUrl(), param.getTenantCode(), param.getAppletType())),
            (CloseableHttpResponse httpResponse) -> {
              String response;
              try {
                response = EntityUtils.toString(httpResponse.getEntity());
              } catch (IOException e) {
                LOGGER.error(e.getMessage(), e);
                throw new UncheckedIOException(e);
              }

              JSONObject parseObject = JSONObject.parseObject(response);
              Validate.isTrue((Boolean) parseObject.get("success"), "调用内部第三方小程序失败源信息：%s", response);
              Validate.notNull(parseObject.get("data"), "调用内部第三方小程序数据为空");

              return ((JSONObject) parseObject.get("data")).toJavaObject(WxOpenInfo.class);
            });
    return wxOpenInfo;
  }

  /**
   * 创建设置获取第三方小程序需要的URL等信息工具类实例
   *
   * @return
   */
  public InternalWxOpenInfoProcessor getWxOpenInitializationProcessor() {
    return this.internalWxOpenInfoProcessor;
  }

  /**
   * 创建第三方小程序用户相关工具类实例
   *
   * @return
   */
  public WxOpenUserProcessor getWxOpenUserProcessor() {
    return this.wxOpenUserProcessor;
  }

  /**
   * 创建第三方小程序二维码相关工具类实例
   *
   * @return
   */
  public WxOpenQrcodeProcessor getWxOpenQrcodeProcessor() {
    return this.wxOpenQrcodeProcessor;
  }
}
