package com.biz.crm.business.common.local.service.internal;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.geo.Distance;
import org.springframework.data.geo.GeoResults;
import org.springframework.data.geo.Point;
import org.springframework.data.redis.connection.BitFieldSubCommands;
import org.springframework.data.redis.connection.RedisGeoCommands;
import org.springframework.data.redis.core.GeoOperations;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import com.biz.crm.business.common.sdk.service.RedisService;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * redis服务接口实现类
 *
 * @author ning.zhang
 * @date 2021/12/2
 */
@Service
public class RedisServiceImpl implements RedisService {

  @Autowired
  private RedisTemplate<String, Object> redisTemplate;

  @Override
  public void set(String key, Object value, long time) {
    redisTemplate.opsForValue().set(key, value, time, TimeUnit.SECONDS);
  }

  @Override
  public void set(String key, Object value) {
    redisTemplate.opsForValue().set(key, value);
  }

  @Override
  public void setRPushAll(String key, List<?> value, long timeout, TimeUnit unit) {
    List<Object> objValue = value.stream().map(Object::toString).collect(Collectors.toList());
    redisTemplate.opsForList().rightPushAll(key, objValue.toArray());
    redisTemplate.expire(key, timeout, unit);
  }

  @Override
  public Object get(String key) {
    return redisTemplate.opsForValue().get(key);
  }

  @Override
  public Object getRightPop(String key) {
    return redisTemplate.opsForList().rightPop(key);
  }

  @Override
  public Boolean del(String key) {
    return redisTemplate.delete(key);
  }

  @Override
  public Long del(List<String> keys) {
    return redisTemplate.delete(keys);
  }

  @Override
  public Boolean expire(String key, long time) {
    return redisTemplate.expire(key, time, TimeUnit.SECONDS);
  }

  @Override
  public Long getExpire(String key) {
    return redisTemplate.getExpire(key, TimeUnit.SECONDS);
  }

  @Override
  public Boolean hasKey(String key) {
    return redisTemplate.hasKey(key);
  }

  @Override
  public Long incr(String key, long delta) {
    return redisTemplate.opsForValue().increment(key, delta);
  }

  @Override
  public Long decr(String key, long delta) {
    return redisTemplate.opsForValue().increment(key, -delta);
  }

  @Override
  public Object hGet(String key, String hashKey) {
    return redisTemplate.opsForHash().get(key, hashKey);
  }

  @Override
  public Boolean hSet(String key, String hashKey, Object value, long time) {
    redisTemplate.opsForHash().put(key, hashKey, value);
    return expire(key, time);
  }

  @Override
  public void hSet(String key, String hashKey, Object value) {
    redisTemplate.opsForHash().put(key, hashKey, value);
  }

  @Override
  public Map<Object, Object> hGetAll(String key) {
    return redisTemplate.opsForHash().entries(key);
  }

  @Override
  public Boolean hSetAll(String key, Map<String, Object> map, long time) {
    redisTemplate.opsForHash().putAll(key, map);
    return expire(key, time);
  }

  @Override
  public void hSetAll(String key, Map<String, ?> map) {
    redisTemplate.opsForHash().putAll(key, map);
  }

  @Override
  public void hDel(String key, Object... hashKey) {
    redisTemplate.opsForHash().delete(key, hashKey);
  }

  @Override
  public Boolean hHasKey(String key, String hashKey) {
    return redisTemplate.opsForHash().hasKey(key, hashKey);
  }

  @Override
  public Long hIncr(String key, String hashKey, Long delta) {
    return redisTemplate.opsForHash().increment(key, hashKey, delta);
  }

  @Override
  public Long hDecr(String key, String hashKey, Long delta) {
    return redisTemplate.opsForHash().increment(key, hashKey, -delta);
  }

  @Override
  public Set<Object> sMembers(String key) {
    return redisTemplate.opsForSet().members(key);
  }

  @Override
  public Long sAdd(String key, Object... values) {
    return redisTemplate.opsForSet().add(key, values);
  }

  @Override
  public Long sAdd(String key, long time, Object... values) {
    Long count = redisTemplate.opsForSet().add(key, values);
    expire(key, time);
    return count;
  }

  @Override
  public Boolean sIsMember(String key, Object value) {
    return redisTemplate.opsForSet().isMember(key, value);
  }

  @Override
  public Long sSize(String key) {
    return redisTemplate.opsForSet().size(key);
  }

  @Override
  public Long sRemove(String key, Object... values) {
    return redisTemplate.opsForSet().remove(key, values);
  }

  @Override
  public List<Object> lRange(String key, long start, long end) {
    return redisTemplate.opsForList().range(key, start, end);
  }

  @Override
  public Long lSize(String key) {
    return redisTemplate.opsForList().size(key);
  }

  @Override
  public Object lIndex(String key, long index) {
    return redisTemplate.opsForList().index(key, index);
  }

  @Override
  public Long lPush(String key, Object value) {
    return redisTemplate.opsForList().rightPush(key, value);
  }

  @Override
  public Long lPush(String key, Object value, long time) {
    Long index = redisTemplate.opsForList().rightPush(key, value);
    expire(key, time);
    return index;
  }

  @Override
  public Long lPushAll(String key, Object... values) {
    return redisTemplate.opsForList().rightPushAll(key, values);
  }

  @Override
  public Long lPushAll(String key, Long time, Object... values) {
    Long count = redisTemplate.opsForList().rightPushAll(key, values);
    expire(key, time);
    return count;
  }

  @Override
  public Long lRemove(String key, long count, Object value) {
    return redisTemplate.opsForList().remove(key, count, value);
  }

  @Override
  public Boolean bitAdd(String key, int offset, boolean b) {
    return redisTemplate.opsForValue().setBit(key, offset, b);
  }

  @Override
  public Boolean bitGet(String key, int offset) {
    return redisTemplate.opsForValue().getBit(key, offset);
  }

  @Override
  public Long bitCount(String key) {
    return redisTemplate.execute((RedisCallback<Long>) con -> con.bitCount(key.getBytes()));
  }

  @Override
  public List<Long> bitField(String key, int limit, int offset) {
    return redisTemplate.execute((RedisCallback<List<Long>>) con ->
        con.bitField(key.getBytes(),
            BitFieldSubCommands.create().get(BitFieldSubCommands.BitFieldType.unsigned(limit)).valueAt(offset)));
  }

  @Override
  public byte[] bitGetAll(String key) {
    return redisTemplate.execute((RedisCallback<byte[]>) con -> con.get(key.getBytes()));
  }

  @Override
  public Long geoAdd(String key, Double x, Double y, String name) {
    return redisTemplate.opsForGeo().add(key, new Point(x, y), name);
  }

  @Override
  public List<Point> geoGetPointList(String key, Object... place) {
    return redisTemplate.opsForGeo().position(key, place);
  }

  @Override
  public Distance geoCalculationDistance(String key, String placeOne, String placeTow) {
    return redisTemplate.opsForGeo()
        .distance(key, placeOne, placeTow, RedisGeoCommands.DistanceUnit.KILOMETERS);
  }

  @Override
  public GeoResults<RedisGeoCommands.GeoLocation<Object>> geoNearByPlace(String key, String place, Distance distance, long limit, Sort.Direction sort) {
    RedisGeoCommands.GeoRadiusCommandArgs args = RedisGeoCommands.GeoRadiusCommandArgs.newGeoRadiusArgs().includeDistance().includeCoordinates();
    // 判断排序方式
    if (Sort.Direction.ASC == sort) {
      args.sortAscending();
    } else {
      args.sortDescending();
    }
    args.limit(limit);
    return redisTemplate.opsForGeo()
        .radius(key, place, distance, args);
  }

  @Override
  public List<String> geoGetHash(String key, String... place) {
    GeoOperations<String, Object> geoOperations = redisTemplate.opsForGeo();
    List<String> hash = geoOperations.hash(key, (Object[])place);
    return hash;
  }
}