package com.biz.crm.business.common.local.service.internal;

import com.biz.crm.business.common.sdk.dto.TreeDto;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.business.common.sdk.utils.TreeRuleCode;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Service;

/**
 * @author hecheng
 * @description: 降维码生成的一个默认实现
 * @date 2021/9/30 下午2:09
 */
@Service("defaultTreeRuleCodeStrategy")
public class DefaultTreeGenerateRuleCodeServiceImpl implements TreeRuleCodeStrategy {

  private final static String KEY = "default";
  /**
   * 缓存 生成降维码的方式，以降维码的长度为key缓存
   */
  private Map<Integer, TreeRuleCode> cache = new HashMap<>();

  private TreeRuleCode get(int ruleCodeSplitLength) {
    TreeRuleCode treeRuleCode = this.cache.get(ruleCodeSplitLength);
    if (treeRuleCode != null) {
      return treeRuleCode;
    } else {
      treeRuleCode = new TreeRuleCode(ruleCodeSplitLength);
      this.cache.put(ruleCodeSplitLength, treeRuleCode);
      return treeRuleCode;
    }
  }


  @Override
  public String getKey() {
    return KEY;
  }

  @Override
  public String generate(int ruleCodeSplitLength, String parentRuleCode, List<TreeDto> children) {
    String ruleCode = "";
    String prefix = "";
    TreeRuleCode treeRuleCode = this.get(ruleCodeSplitLength);
    if (StringUtils.isNotEmpty(parentRuleCode)) {
      prefix = parentRuleCode;
    }
    if (CollectionUtils.isEmpty(children)) {
      ruleCode = prefix + treeRuleCode.generateByNum(1);
    } else {
      Set<Integer> set = new LinkedHashSet<>();
      for (TreeDto item : children) {
        if (StringUtils.isNotEmpty(item.getRuleCode())) {
          Integer integer = treeRuleCode.findLevelNumByRuleCode(item.getRuleCode());
          set.add(integer);
        }
      }
      int maxRuleCodeValueInLevel = treeRuleCode.getMaxRuleCode();
      for (int i = 1; i <= maxRuleCodeValueInLevel; i++) {
        Validate.isTrue(i != maxRuleCodeValueInLevel, "降维编码越界，请联系管理员处理");
        if (!set.contains(i)) {
          ruleCode = prefix + treeRuleCode.generateByNum(i);
          break;
        }
      }
    }
    return ruleCode;
  }

  @Override
  public String generateByNum(int ruleCodeSplitLength, int num) {
    TreeRuleCode treeRuleCode = this.get(ruleCodeSplitLength);
    return treeRuleCode.generateByNum(num);
  }

  @Override
  public Set<String> findParentRuleCodeByRuleCodesExcludeAnySelf(int ruleCodeSplitLength, List<String> ruleCodes) {
    TreeRuleCode treeRuleCode = this.get(ruleCodeSplitLength);
    return treeRuleCode.findParentRuleCodesByRuleCodesExcludeAnySelf(ruleCodes);
  }

  @Override
  public Set<String> findParentRuleCodesByRuleCodeExcludeSelf(int ruleCodeSplitLength, String ruleCode) {
    TreeRuleCode treeRuleCode = this.get(ruleCodeSplitLength);
    return treeRuleCode.findParentRuleCodesByRuleCodeExcludeSelf(ruleCode);
  }

  @Override
  public Set<String> findParentRuleCodesByRuleCodesExcludeSelf(int ruleCodeSplitLength, List<String> ruleCodes) {
    TreeRuleCode treeRuleCode = this.get(ruleCodeSplitLength);
    return treeRuleCode.findParentRuleCodesByRuleCodesExcludeSelf(ruleCodes);
  }

  @Override
  public Set<String> findParentRuleCodeByRuleCode(int ruleCodeSplitLength, String ruleCode) {
    TreeRuleCode treeRuleCode = this.get(ruleCodeSplitLength);
    return treeRuleCode.findParentRuleCodesByRuleCode(ruleCode);
  }

  @Override
  public Set<String> findParentRuleCodeByRuleCodes(int ruleCodeSplitLength, List<String> ruleCodes) {
    TreeRuleCode treeRuleCode = this.get(ruleCodeSplitLength);
    return treeRuleCode.findParentRuleCodesByRuleCodes(ruleCodes);
  }
}
