package com.biz.crm.business.common.local.config;

import com.biz.crm.business.common.sdk.model.Result;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Arrays;
import java.util.Objects;

/**
 * @author jerry7
 * @date 2021/11/4 9:45
 * 初步的全局异常捕获，后续完善
 */
@ControllerAdvice
@Slf4j
public class GlobalExceptionHandler {

    @ExceptionHandler(value = Exception.class)
    @ResponseBody
    public Result<String> errorHandler(Exception e) {
        log.error(e.getMessage(), e);
        //拦截器已打印错误日志
        return Result.error(e.getMessage());
    }

    @ExceptionHandler(value = HttpMessageNotReadableException.class)
    @ResponseBody
    public Result<String> httpMessageNotReadableExceptionHandler(HttpMessageNotReadableException e) {
        //拦截器已打印错误日志
        String errorMsg = "请求类型异常:接口";
        try {
            ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
            HttpServletRequest request = null;
            if (requestAttributes != null) {
                request = requestAttributes.getRequest();
            }
            if (Objects.nonNull(request)) {
                errorMsg = errorMsg + "[" + request.getRequestURI() + "]";
            }
            errorMsg = errorMsg + "字段类型错误:" + e.getCause().getMessage();
        } catch (Exception ex) {
            log.error(ex.getMessage(), ex);
        }
        //拦截器已打印错误日志
        log.error(errorMsg, e);
        return Result.error(errorMsg);
    }

    @ExceptionHandler(value = HttpRequestMethodNotSupportedException.class)
    @ResponseBody
    public Result<String> httpRequestMethodNotSupportedExceptionHandler(HttpRequestMethodNotSupportedException e) {
        String errorMsg = "请求类型异常:接口";
        try {
            ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
            HttpServletRequest request = null;
            if (requestAttributes != null) {
                request = requestAttributes.getRequest();
            }
            if (Objects.nonNull(request)) {
                errorMsg = errorMsg + "[" + request.getRequestURI() + "]";
            }
            errorMsg = errorMsg + "支持类型" + Arrays.toString(e.getSupportedMethods()) + ";当前请求类型[" + e.getMethod() + "]";
        } catch (Exception ex) {
            log.error(ex.getMessage(), ex);
        }
        //拦截器已打印错误日志
        log.error(errorMsg, e);
        return Result.error(errorMsg);
    }

    @ExceptionHandler(value = MissingServletRequestParameterException.class)
    @ResponseBody
    public Result<String> missingServletRequestParameterExceptionHandler(MissingServletRequestParameterException e) {
        String errorMsg = "请求接口异常:接口";
        try {
            ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
            HttpServletRequest request = null;
            if (requestAttributes != null) {
                request = requestAttributes.getRequest();
            }
            if (Objects.nonNull(request)) {
                errorMsg = errorMsg + "[" + request.getRequestURI() + "]";
            }
            errorMsg = errorMsg + "缺失必填参数;参数类型[" + e.getParameterType() + "]参数[" + e.getParameterName() + "]";
        } catch (Exception ex) {
            log.error(ex.getMessage(), ex);
        }
        //拦截器已打印错误日志
        log.error(errorMsg, e);
        return Result.error(errorMsg);
    }
}
