package com.biz.crm.business.common.local.config;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.constant.CommonConstant;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.security.sdk.config.SimpleSecurityProperties;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Maps;
import feign.RequestInterceptor;
import feign.RequestTemplate;
import io.jsonwebtoken.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.Base64Utils;

import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
/**
 * 这个类的作用是：
 *  在一个没有带jwt请求中去调用feign接口时，就会自动加上系统自动会用feignuser的jwt去请求接口
 * @Author scgaopan
 * @Date 2021/9/20 12:08
 * @Version 1.0
 * @Desc
 */
public class FeignClientConfiguration implements RequestInterceptor {

    @Autowired(required = false)
    private SimpleSecurityProperties simpleSecurityProperties;
    private static volatile Cache<String, String> cache = CacheBuilder.newBuilder()
            .initialCapacity(10)
            .expireAfterWrite(24*7, TimeUnit.DAYS)
            .maximumSize(100000)
            .build();

    /***
     * 当feign调用时，发现request中没有带jwt，就用系统默认配置的用户Jwt进行访问
     * @param requestTemplate
     */
    @Override
    public void apply(RequestTemplate requestTemplate) {
//        String url = requestTemplate.url();
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();

        //判断当前有没有用户认证信息
        boolean authedFlag = authentication == null ||
                authentication.getPrincipal() == null
                || StringUtils.equals("anonymousUser", authentication.getPrincipal().toString());
        //这里默认对所有feign接口调用都进行默认jwt处理，可以
        if (authedFlag) {
            String tempJwt = getDefaultJwt();
            requestTemplate.header("jwt", tempJwt);
        }
        String traceId = MDC.get(CommonConstant.HEADER_TRACE_ID);
        if (StringUtils.isNotBlank(traceId)) {
            requestTemplate.header(CommonConstant.HEADER_TRACE_ID, traceId);
        }
    }

    /***
     * 生成一个默认的JWT
     * @return
     */
    private String getDefaultJwt() {
        String tempJwt = cache.getIfPresent(TenantUtils.getTenantCode());
        if (tempJwt == null) {
            synchronized (FeignClientConfiguration.class) {
                while (tempJwt == null) {
                    AbstractCrmUserIdentity abstractCrmUserIdentity = new AbstractCrmUserIdentity() {
                        @Override
                        public String getPostCode() {
                            return null;
                        }

                        @Override
                        public String getPostName() {
                            return null;
                        }
                    };
                    abstractCrmUserIdentity.setLoginType(1);
                    abstractCrmUserIdentity.setIdentityType("u");
                    abstractCrmUserIdentity.setTenantCode(TenantUtils.getTenantCode());
                    abstractCrmUserIdentity.setUsername(simpleSecurityProperties.getIndependencyUser());
                    abstractCrmUserIdentity.setAccount(simpleSecurityProperties.getIndependencyUser());
                    abstractCrmUserIdentity.setRealName("系统用户或系统定时任务");
                    abstractCrmUserIdentity.setRoleCodes(simpleSecurityProperties.getIndependencyRoles());
                    tempJwt = encode(abstractCrmUserIdentity, -1, simpleSecurityProperties.getSecretKey());
                    cache.put(TenantUtils.getTenantCode(), tempJwt);
                }
            }
        }
        return tempJwt;
    }

    /***
     * 判断当前url是否需要加入默认的jwt，只有当前url包含在在array中才会加入默认的jwt
     * @param array 配置需要过滤的url请求
     * @param url 当前url请求
     * @return
     */
    public static boolean containsString(String[] array, String url) {
        if (array == null || array.length == 0 || StringUtils.isBlank(url)) {
            return true;
        }
        //如果是get请求 url后面可能带有?a=xx&b=xx
        //配置了通配符*
        int i = url.indexOf("?");
        final AtomicReference<String> urlRef = new AtomicReference<>(i > 0 ? url.substring(0, i) : url);
        return Arrays.stream(array).anyMatch(str -> {
            if (str.indexOf("*") > 0) {
                str = str.replaceAll("\\*", "");
                return url.startsWith(str);
            }
            return StringUtils.equals(str, urlRef.get());
        });

    }

    /**
     * 构造JWT信息  该方法copy nebula-core方法
     * 与com.bizunited.nebula.security.local.utils.JwtUtils#encode保持同步
     *
     * @param ttl       jwtToken的有效时长(单位秒)，如果不传入，则有效时长为168小时
     * @param secretKey 秘钥信息（未加密的）
     * @return
     */
    private static String encode(UserIdentity userIdentity, int ttl, String secretKey) {
        // 指定签名的时候使用的签名算法
        SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.HS512;
        Date now = new Date();

        // 首先构造JWT头信息
        Map<String, Object> jwsHeader = Maps.newHashMap();
        jwsHeader.put(JwsHeader.ALGORITHM, signatureAlgorithm.getValue());
        jwsHeader.put(JwsHeader.KEY_ID, userIdentity.getAccount());
        jwsHeader.put(JwsHeader.TYPE, userIdentity.getIdentityType());
        jwsHeader.put("ttl", ttl);
        jwsHeader.put("tenantCode", userIdentity.getTenantCode());

        // 然后构造有效载荷(有效载荷中，存储了当前用户的完整字段信息，以便业务系统能根据自己的需求进行扩充)
        JSONObject userJson = (JSONObject) JSONObject.toJSON(userIdentity);
        Claims claims = Jwts.claims();
        claims.put("userinfo", userJson);

        // 然后是可能的过期时间
        Date expiration = null;
        if (ttl > 0) {
            expiration = DateUtils.addMilliseconds(now, ttl);
        } else if (ttl == -1) {
            //10年
            expiration = DateUtils.addHours(now, 87600);
        } else {
            expiration = DateUtils.addHours(now, 168);
        }

        // 最后是签名(这个加密工具类，同学们可以根据自己选择的加密方式进行替换)
        // TODO 先使用RAS256对秘钥本身进行加密
        String base64EncodedSecretKey = Base64Utils.encodeToString(secretKey.getBytes());

        // 构造JWT_TOKEN字符串信息
        JwtBuilder builder = Jwts.builder()
                .compressWith(CompressionCodecs.GZIP)
                .setHeader(jwsHeader)
                .setClaims(claims)
                .setExpiration(expiration)
                .signWith(signatureAlgorithm, base64EncodedSecretKey);
        return builder.compact();
    }
}
