package com.biz.crm.dms.business.allow.sale.local.list.service.internal;

import com.biz.crm.dms.business.allow.sale.local.list.entity.AllowSaleList;
import com.biz.crm.dms.business.allow.sale.local.list.repository.AllowSaleListRepository;
import com.biz.crm.dms.business.allow.sale.local.list.service.AllowSaleListCacheService;
import com.biz.crm.dms.business.allow.sale.sdk.constant.AllowSaleRuleConstant;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleListTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.AllowSaleListModifyNoticeDto;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.AllowSaleListModifyNoticeItemDto;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.Redisson;
import org.redisson.api.RTopic;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * 允销清单缓存实现
 *
 * @author sunx
 * @date 2022/6/20
 */
@Slf4j
@Service
public class AllowSaleListCacheServiceImpl implements AllowSaleListCacheService {

  @Autowired(required = false)
  private Redisson redisson;

  @Autowired(required = false)
  @Qualifier("_allowSaleListHandlerExecutor")
  private ThreadPoolExecutor threadPoolExecutor;

  @Autowired(required = false)
  private AllowSaleListRepository allowSaleListRepository;

  /** 保证在缓存刷新过程中，需要读取缓存的线程进行阻塞(一个锁) */
  private static ReentrantReadWriteLock loadLock = new ReentrantReadWriteLock();
  /**
   * 第一级Key：listType
   *
   * <p>第二级Key：（经销商、终端）编码。 value：可购商品编码集合
   */
  private static Map<String, Map<String, Set<String>>> cacheMapping = Maps.newConcurrentMap();

  @Override
  public void notifyCacheRefresh(AllowSaleListModifyNoticeDto dto) {
    Validate.notNull(dto, "刷新缓存参数不能为空");
    RTopic topic = this.redisson.getTopic(AllowSaleRuleConstant.ALLOW_SALE_LIST_NOTIFY);
    topic.publish(dto);
  }

  @Override
  public void clearCache(AllowSaleListModifyNoticeDto dto) {
    Validate.notNull(dto, "刷新缓存参数不能为空");
    ReentrantReadWriteLock.WriteLock writeLock = loadLock.writeLock();
    try {
      writeLock.lock();
      // 重新加载
      this.reloadCache(dto);
    } finally {
      writeLock.unlock();
    }
  }

  @Override
  public Set<String> findCache(String listType, String businessCode) {
    return cacheMapping.getOrDefault(listType, Maps.newHashMap()).get(businessCode);
  }

  /**
   * 重新加载缓存数据
   *
   * @param dto
   * @return
   */
  private void reloadCache(AllowSaleListModifyNoticeDto dto) {
    if (dto == null) {
      return;
    }
    if (CollectionUtils.isEmpty(dto.getList())) {
      for (AllowSaleListTypeEnums value : AllowSaleListTypeEnums.values()) {
        this.reloadCacheByListTypeAndBusinessCodes(value.getCode(), Sets.newHashSet());
      }
    } else {
      for (AllowSaleListModifyNoticeItemDto item : dto.getList()) {
        final String listType = item.getListType();
        final Set<String> businessCodes = item.getBusinessCodes();
        this.reloadCacheByListTypeAndBusinessCodes(listType, businessCodes);
      }
    }
  }

  /**
   * 重新加载缓存
   *
   * @param listType
   * @param businessCodes
   */
  private void reloadCacheByListTypeAndBusinessCodes(String listType, Set<String> businessCodes) {
    if (StringUtils.isBlank(listType)) {
      return;
    }
    boolean reloadAll = CollectionUtils.isEmpty(businessCodes);
    if (CollectionUtils.isEmpty(businessCodes)) {
      businessCodes =
          this.allowSaleListRepository.findBusinessCodesByListTypeAndRuleType(
              listType, AllowSaleRuleTypeEnums.ALLOW.getDictCode());
    }
    if (CollectionUtils.isEmpty(businessCodes)) {
      if (cacheMapping.get(listType) != null) {
        cacheMapping.get(listType).clear();
      }
      return;
    }
    Lists.partition(Lists.newArrayList(businessCodes), 500)
        .forEach(
            a ->
                CompletableFuture.runAsync(
                    () -> this.reloadCacheItem(a, listType), threadPoolExecutor));
    if (Boolean.TRUE.equals(reloadAll)) {
      final Map<String, Set<String>> map = cacheMapping.getOrDefault(listType, Maps.newHashMap());
      final Set<String> set = map.keySet();
      final SetView<String> difference = Sets.difference(set, businessCodes);
      if (!CollectionUtils.isEmpty(difference)) {
        for (String item : difference) {
          cacheMapping.getOrDefault(listType, Maps.newHashMap()).remove(item);
        }
      }
    }
  }

  /**
   * 分组刷新
   *
   * @param businessCodes
   * @param listType
   */
  private void reloadCacheItem(List<String> businessCodes, String listType) {
    if (CollectionUtils.isEmpty(businessCodes)) {
      return;
    }
    List<AllowSaleList> list =
        this.allowSaleListRepository.findByBusinessCodesAndListTypeAndRuleType(
            businessCodes, listType, AllowSaleRuleTypeEnums.ALLOW.getDictCode());
    Map<String, Set<String>> map = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(list)) {
      map =
          list.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getProductCode(), a.getBusinessCode()))
              .collect(
                  Collectors.groupingBy(
                      AllowSaleList::getBusinessCode,
                      Collectors.mapping(AllowSaleList::getProductCode, Collectors.toSet())));
    }
    final Map<String, Set<String>> mapItem = cacheMapping.getOrDefault(listType, Maps.newHashMap());
    for (String businessCode : businessCodes) {
      mapItem.put(businessCode, map.get(businessCode));
    }
    cacheMapping.put(listType, mapItem);
  }
}
