package com.biz.crm.dms.business.allow.sale.local.rule.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.allow.sale.local.dimension.service.DimensionService;
import com.biz.crm.dms.business.allow.sale.local.product.entity.AllowSaleRuleProduct;
import com.biz.crm.dms.business.allow.sale.local.product.service.AllowSaleRuleProductService;
import com.biz.crm.dms.business.allow.sale.local.rule.entity.AllowSaleRule;
import com.biz.crm.dms.business.allow.sale.local.rule.repository.AllowSaleRuleRepository;
import com.biz.crm.dms.business.allow.sale.local.rule.service.AllowSaleRuleService;
import com.biz.crm.dms.business.allow.sale.sdk.config.AllowSaleProperties;
import com.biz.crm.dms.business.allow.sale.sdk.constant.AllowSaleRuleConstant;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleCreateTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleProductRelateTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleRuleTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.rule.dto.AllowSaleRulePaginationDto;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.spu.sdk.service.ProductSpuVoService;
import com.biz.crm.mdm.business.product.spu.sdk.vo.ProductSpuVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 允销规则主细信息(AllowSaleRule)表服务实现类
 *
 * @author sunx
 * @date 2021-11-19 15:16:25
 */
@Slf4j
@Service("allowSaleRuleService")
public class AllowSaleRuleServiceImpl implements AllowSaleRuleService {

  @Autowired(required = false)
  private AllowSaleRuleRepository allowSaleRuleRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private AllowSaleRuleProductService allowSaleRuleProductService;

  @Autowired(required = false)
  private DimensionService dimensionService;

  @Autowired(required = false)
  private ProductLevelVoSdkService productLevelVoSdkService;

  @Autowired(required = false)
  private ProductSpuVoService productSpuVoService;

  @Autowired(required = false)
  private AllowSaleProperties allowSaleProperties;

  @Autowired(required = false)
  private AllowSaleRuleListAdapter allowSaleRuleListAdapter;

  @Override
  public Page<AllowSaleRule> findByConditions(Pageable pageable, AllowSaleRulePaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new AllowSaleRulePaginationDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<AllowSaleRule> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<AllowSaleRule> pageResult = this.allowSaleRuleRepository.findByConditions(page, dto);
    if (pageResult != null && CollectionUtils.isNotEmpty(pageResult.getRecords())) {
      for (AllowSaleRule item : pageResult.getRecords()) {
        item.setFlag(this.allowSaleProperties.checkByCreateType(item.getCreateType()));
      }
    }
    return pageResult;
  }

  @Override
  public AllowSaleRule findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    AllowSaleRule re = this.allowSaleRuleRepository.findById(id);
    if (Objects.isNull(re)) {
      return null;
    }
    JSONObject jsonObject =
        this.dimensionService.findByRuleCode(re.getRuleCode(), re.getDimensionType());

    List<AllowSaleRuleProduct> ruleProductList =
        this.allowSaleRuleProductService.findByRuleCode(re.getRuleCode());
    setAllowSaleRuleProductList(ruleProductList);

    re.setDimensionJson(jsonObject);
    re.setProductList(ruleProductList);
    return re;
  }

  @Override
  @Transactional
  public AllowSaleRule create(AllowSaleRule allowSaleRule) {
    this.createValidation(allowSaleRule);
    allowSaleRule.setTenantCode(TenantUtils.getTenantCode());
    allowSaleRule.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    if (StringUtils.isBlank(allowSaleRule.getCreateType())) {
      allowSaleRule.setCreateType(AllowSaleRuleCreateTypeEnums.DEFAULT.getCode());
    }
    if (StringUtils.isBlank(allowSaleRule.getEnableStatus())) {
      allowSaleRule.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    }
    if (StringUtils.isBlank(allowSaleRule.getRuleCode())) {
      allowSaleRule.setRuleCode(
          generateCodeService.generateCode(AllowSaleRuleConstant.RULE_CODE, 1).get(0));
    } else {
      Integer count = this.allowSaleRuleRepository.countByRuleCode(allowSaleRule.getRuleCode());
      Validate.isTrue(null == count || 1 > count, allowSaleRule.getRuleCode() + "编码已存在(已逻辑删除)");
    }
    this.allowSaleRuleRepository.saveOrUpdate(allowSaleRule);

    this.bindExtInfo(allowSaleRule);

    if (EnableStatusEnum.ENABLE.getCode().equals(allowSaleRule.getEnableStatus())) {
      this.allowSaleRuleListAdapter.createByRule(allowSaleRule);
    }
    return allowSaleRule;
  }

  @Override
  @Transactional
  public AllowSaleRule update(AllowSaleRule allowSaleRule) {
    this.updateValidation(allowSaleRule);
    Validate.isTrue(
        AllowSaleRuleCreateTypeEnums.DEFAULT.getCode().equals(allowSaleRule.getCreateType()),
        "当前规则关联合同，不能执行该操作!");
    String currentId = allowSaleRule.getId();
    AllowSaleRule current = this.allowSaleRuleRepository.findById(currentId);
    Validate.notNull(current, "修改信息不存在");
    Validate.isTrue(allowSaleRule.getRuleCode().equals(current.getRuleCode()), "规则编码不能修改");
    this.allowSaleRuleRepository.saveOrUpdate(allowSaleRule);

    this.bindExtInfo(allowSaleRule);

    if (EnableStatusEnum.ENABLE.getCode().equals(allowSaleRule.getEnableStatus())) {
      this.allowSaleRuleListAdapter.createByRule(allowSaleRule);
    }
    return allowSaleRule;
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<AllowSaleRule> list = this.allowSaleRuleRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    final Optional<AllowSaleRule> first =
        list.stream()
            .filter(a -> !AllowSaleRuleCreateTypeEnums.DEFAULT.getCode().equals(a.getCreateType()))
            .findFirst();
    Validate.isTrue(!first.isPresent(), "存在非手动创建的规则，不能执行该操作!");
    this.allowSaleRuleRepository.updateDelFlagByIds(ids);
    for (AllowSaleRule item : list) {
      this.allowSaleRuleListAdapter.deleteByRule(item);
    }
  }

  @Override
  public List<AllowSaleRule> findByCodesAndTenantCode(
      List<String> ruleCodeList, String tenantCode) {
    if (CollectionUtils.isEmpty(ruleCodeList)) {
      return Lists.newLinkedList();
    }
    if (StringUtils.isBlank(tenantCode)) {
      tenantCode = TenantUtils.getTenantCode();
    }
    return this.allowSaleRuleRepository.findByCodesAndTenantCode(ruleCodeList, tenantCode);
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    final List<AllowSaleRule> list = this.allowSaleRuleRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    final Optional<AllowSaleRule> first =
        list.stream()
            .filter(a -> !AllowSaleRuleCreateTypeEnums.DEFAULT.getCode().equals(a.getCreateType()))
            .findFirst();
    Validate.isTrue(!first.isPresent(), "存在非手动创建的规则，不能执行该操作!");

    final List<AllowSaleRule> list1 =
        list.stream()
            .filter(a -> EnableStatusEnum.DISABLE.getCode().equals(a.getEnableStatus()))
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(list1)) {
      return;
    }
    final Set<String> set = list1.stream().map(AllowSaleRule::getId).collect(Collectors.toSet());
    this.allowSaleRuleRepository.updateEnableStatusByIds(
        Lists.newArrayList(set), EnableStatusEnum.ENABLE);
    for (AllowSaleRule allowSaleRule : list1) {
      JSONObject jsonObject =
          this.dimensionService.findByRuleCode(
              allowSaleRule.getRuleCode(), allowSaleRule.getDimensionType());
      List<AllowSaleRuleProduct> ruleProductList =
          this.allowSaleRuleProductService.findByRuleCode(allowSaleRule.getRuleCode());
      allowSaleRule.setProductList(ruleProductList);
      allowSaleRule.setDimensionJson(jsonObject);
      this.allowSaleRuleListAdapter.createByRule(allowSaleRule);
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    final List<AllowSaleRule> list = this.allowSaleRuleRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    final Optional<AllowSaleRule> first =
        list.stream()
            .filter(a -> !AllowSaleRuleCreateTypeEnums.DEFAULT.getCode().equals(a.getCreateType()))
            .findFirst();
    Validate.isTrue(!first.isPresent(), "存在非手动创建的规则，不能执行该操作!");

    final List<AllowSaleRule> list1 =
        list.stream()
            .filter(a -> EnableStatusEnum.ENABLE.getCode().equals(a.getEnableStatus()))
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(list1)) {
      return;
    }
    final Set<String> set = list1.stream().map(AllowSaleRule::getId).collect(Collectors.toSet());
    this.allowSaleRuleRepository.updateEnableStatusByIds(
        Lists.newArrayList(set), EnableStatusEnum.DISABLE);
    for (AllowSaleRule item : list) {
      this.allowSaleRuleListAdapter.deleteByRule(item);
    }
  }

  @Override
  @Transactional
  public void updateByContractCodesAndEnableStatus(
      List<String> contractCodes, String enableStatus) {
    if (CollectionUtils.isEmpty(contractCodes) || StringUtils.isBlank(enableStatus)) {
      return;
    }
    List<AllowSaleRule> list =
        this.allowSaleRuleRepository.findByRelateCodesAndCreateType(
            contractCodes, AllowSaleRuleCreateTypeEnums.CONTRACT.getCode());
    if (CollectionUtils.isEmpty(list)) {
      return;
    }

    final List<AllowSaleRule> saleRuleList =
        list.stream()
            .filter(a -> !a.getEnableStatus().equals(enableStatus))
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(saleRuleList)) {
      return;
    }

    for (AllowSaleRule allowSaleRule : saleRuleList) {
      allowSaleRule.setEnableStatus(enableStatus);
    }
    this.allowSaleRuleRepository.saveOrUpdateBatch(saleRuleList);

    if (enableStatus.equals(EnableStatusEnum.DISABLE.getCode())) {
      for (AllowSaleRule item : saleRuleList) {
        this.allowSaleRuleListAdapter.deleteByRule(item);
      }
    } else if (enableStatus.equals(EnableStatusEnum.ENABLE.getCode())) {
      for (AllowSaleRule allowSaleRule : saleRuleList) {
        JSONObject jsonObject =
            this.dimensionService.findByRuleCode(
                allowSaleRule.getRuleCode(), allowSaleRule.getDimensionType());
        List<AllowSaleRuleProduct> ruleProductList =
            this.allowSaleRuleProductService.findByRuleCode(allowSaleRule.getRuleCode());
        allowSaleRule.setProductList(ruleProductList);
        allowSaleRule.setDimensionJson(jsonObject);
        this.allowSaleRuleListAdapter.createByRule(allowSaleRule);
      }
    }
  }

  @Override
  public List<AllowSaleRule> findByContractCodes(Set<String> contractCodes) {
    if (CollectionUtils.isEmpty(contractCodes)) {
      return Lists.newLinkedList();
    }
    return this.allowSaleRuleRepository.findByContractCodes(contractCodes);
  }

  /** 绑定扩展信息，商品+规则 */
  private void bindExtInfo(AllowSaleRule allowSaleRule) {
    Validate.notNull(allowSaleRule, "允销规则不能为空");
    allowSaleRule.getDimensionJson().put("ruleCode", allowSaleRule.getRuleCode());
    this.dimensionService.saveDimension(
        allowSaleRule.getDimensionJson(), allowSaleRule.getDimensionType());
    if (CollectionUtils.isNotEmpty(allowSaleRule.getProductList())) {
      for (AllowSaleRuleProduct item : allowSaleRule.getProductList()) {
        item.setRuleCode(allowSaleRule.getRuleCode());
        item.setTenantCode(TenantUtils.getTenantCode());
      }
      this.allowSaleRuleProductService.batchSave(allowSaleRule.getProductList());
    }
  }

  private void createValidation(AllowSaleRule allowSaleRule) {
    this.validation(allowSaleRule);
  }

  private void updateValidation(AllowSaleRule allowSaleRule) {
    Validate.notBlank(allowSaleRule.getId(), "规则id不能为空");
    Validate.notBlank(allowSaleRule.getRuleCode(), "规则编码不能为空");
    Validate.isTrue(
        allowSaleRule.getCreateType().equals(AllowSaleRuleCreateTypeEnums.DEFAULT.getCode()),
        "非手动创建信息不能编辑");
    this.validation(allowSaleRule);
  }

  private void validation(AllowSaleRule allowSaleRule) {
    if (StringUtils.isBlank(allowSaleRule.getRuleType())) {
      allowSaleRule.setRuleType(AllowSaleRuleTypeEnums.ALLOW.getDictCode());
    }
    Validate.notBlank(allowSaleRule.getRuleName(), "允销规则名称不能为空");
    Validate.notBlank(allowSaleRule.getDimensionType(), "规则维度不能为空");
    Validate.notBlank(allowSaleRule.getCreateType(), "创建方式不能为空");
    Validate.notNull(allowSaleRule.getDimensionJson(), "维度规则配置不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(allowSaleRule.getProductList()), "可购商品组织、终端或客户不能为空");
  }

  /**
   * 设置可购商品规则扩展信息 spu名 产品层级名
   *
   * @param list
   */
  private void setAllowSaleRuleProductList(List<AllowSaleRuleProduct> list) {
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    Map<String, String> mapSpu = Maps.newHashMap();
    Map<String, String> mapLevel = Maps.newHashMap();
    // key relateType, value List<relateCode>
    Map<String, List<String>> map =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNotBlank(a.getRelateType())
                        && StringUtils.isNotBlank(a.getRelateCode()))
            .collect(
                Collectors.groupingBy(
                    AllowSaleRuleProduct::getRelateType,
                    Collectors.mapping(AllowSaleRuleProduct::getRelateCode, Collectors.toList())));

    for (Entry<String, List<String>> item : map.entrySet()) {
      if (AllowSaleRuleProductRelateTypeEnums.SPU.getCode().equals(item.getKey())) {
        List<ProductSpuVo> curList = this.productSpuVoService.findBySpuCodes(item.getValue());
        if (CollectionUtils.isNotEmpty(curList)) {
          mapSpu =
              curList.stream()
                  .filter(a -> StringUtils.isNoneBlank(a.getSpuCode(), a.getSpuName()))
                  .collect(
                      Collectors.toMap(
                          ProductSpuVo::getSpuCode, ProductSpuVo::getSpuName, (a, b) -> a));
        }
      } else if (AllowSaleRuleProductRelateTypeEnums.PRODUCT_LEVEL
          .getCode()
          .equals(item.getKey())) {
        List<ProductLevelVo> curList =
            this.productLevelVoSdkService.findListByCodes(item.getValue());
        if (CollectionUtils.isNotEmpty(curList)) {
          mapLevel =
              curList.stream()
                  .filter(
                      a ->
                          StringUtils.isNoneBlank(a.getProductLevelCode(), a.getProductLevelName()))
                  .collect(
                      Collectors.toMap(
                          ProductLevelVo::getProductLevelCode,
                          ProductLevelVo::getProductLevelName,
                          (a, b) -> a));
        }
      }
    }

    for (AllowSaleRuleProduct item : list) {
      AllowSaleRuleProductRelateTypeEnums typeEnums =
          AllowSaleRuleProductRelateTypeEnums.findByCode(item.getRelateType());
      if (Objects.isNull(typeEnums)) {
        return;
      }
      if (AllowSaleRuleProductRelateTypeEnums.SPU.equals(typeEnums)) {
        item.setRelateName(mapSpu.getOrDefault(item.getRelateCode(), StringUtils.EMPTY));
      } else if (AllowSaleRuleProductRelateTypeEnums.PRODUCT_LEVEL.equals(typeEnums)) {
        item.setRelateName(mapLevel.getOrDefault(item.getRelateCode(), StringUtils.EMPTY));
      }
    }
  }
}
