package com.biz.crm.dms.business.order.config.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.order.common.sdk.enums.OrderTypeEnum;
import com.biz.crm.dms.business.order.common.sdk.vo.OrderConfigDimensionVo;
import com.biz.crm.dms.business.order.common.sdk.vo.OrderConfigVerificationVo;
import com.biz.crm.dms.business.order.common.sdk.vo.OrderConfigVo;
import com.biz.crm.dms.business.order.config.local.entity.OrderConfigDimensionEntity;
import com.biz.crm.dms.business.order.config.local.entity.OrderConfigEntity;
import com.biz.crm.dms.business.order.config.local.entity.OrderConfigVerificationEntity;
import com.biz.crm.dms.business.order.config.local.repository.OrderConfigDimensionRepository;
import com.biz.crm.dms.business.order.config.local.repository.OrderConfigRepository;
import com.biz.crm.dms.business.order.config.local.repository.OrderConfigVoRepository;
import com.biz.crm.dms.business.order.config.local.repository.OrderVerificationRepository;
import com.biz.crm.dms.business.order.config.sdk.constant.OrderConfigConstant;
import com.biz.crm.dms.business.order.config.sdk.dto.OrderConfigDimensionDto;
import com.biz.crm.dms.business.order.config.sdk.dto.OrderConfigDto;
import com.biz.crm.dms.business.order.config.sdk.dto.OrderConfigPageDto;
import com.biz.crm.dms.business.order.config.sdk.dto.OrderConfigVerificationDto;
import com.biz.crm.dms.business.order.config.sdk.enums.EffectiveDimensionEnum;
import com.biz.crm.dms.business.order.config.sdk.service.OrderConfigVoService;
import com.biz.crm.dms.business.order.verification.sdk.dto.OrderValidateDetailDto;
import com.biz.crm.dms.business.order.verification.sdk.dto.OrderValidateDto;

import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 订单配置Vo服务接口实现类
 *
 * @author pengxi
 * @date 2021/12/30
 */
@Slf4j
@Service
public class OrderConfigVoServiceImpl implements OrderConfigVoService {

  @Autowired(required = false)
  private OrderConfigRepository orderConfigRepository;

  @Autowired(required = false)
  private OrderConfigVoRepository orderConfigVoRepository;

  @Autowired(required = false)
  private OrderVerificationRepository orderVerificationRepository;

  @Autowired(required = false)
  private OrderConfigDimensionRepository orderConfigDimensionRepository;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private OrgVoService orgVoService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<OrderConfigVo> findByConditions(Pageable pageable, OrderConfigPageDto dto) {
    // 非默认设定才需要根据维度范围查找
    Set<String> codes = new HashSet<>();
    if (Objects.nonNull(dto.getIsDefault()) && !dto.getIsDefault()) {
      OrderConfigDimensionDto orderConfigDimensionDto = new OrderConfigDimensionDto();
      if (StringUtils.isNotBlank(dto.getCustomerCode())) {
        codes.add(dto.getCustomerCode());
        // 如果是客户编码，还需要找到客户所属组织，并根据所属组织搜索
        CustomerVo customerVo = this.findByCustomerCode(dto.getCustomerCode());
        if(customerVo != null && StringUtils.isNotBlank(customerVo.getOrgCode())) {
          codes.add(customerVo.getOrgCode());
        }
      }
      if (StringUtils.isNotBlank(dto.getOrgCode())) {
        codes.add(dto.getOrgCode());
      }
      if (CollectionUtils.isNotEmpty(codes)) {
        orderConfigDimensionDto.setCodes(codes);
        List<OrderConfigDimensionVo> orderConfigDimensionVoList = this.orderConfigDimensionRepository.findByConditions(orderConfigDimensionDto);
        Set<String> configCodes = orderConfigDimensionVoList.stream().map(OrderConfigDimensionVo::getConfigCode).collect(Collectors.toSet());
        dto.setConfigCodes(configCodes);
      }
    }
    OrderConfigDto orderConfigDto = new OrderConfigDto();
    orderConfigDto.setOrderType(dto.getOrderType());
    orderConfigDto.setIsDefault(dto.getIsDefault());
    orderConfigDto.setConfigCodes(dto.getConfigCodes());
    return this.orderConfigVoRepository.findByConditions(pageable, orderConfigDto);
  }

  @Override
  public OrderConfigVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    OrderConfigEntity entity = this.orderConfigRepository.getById(id);
    if (entity == null) {
      return null;
    }
    OrderConfigVo orderConfigVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, OrderConfigVo.class, HashSet.class, ArrayList.class);
    // 查询校验设置
    List<OrderConfigVerificationEntity> verificationEntities = this.orderVerificationRepository.findByConfigCodeIn(TenantUtils.getTenantCode(), Collections.singletonList(orderConfigVo.getConfigCode()));
    if (CollectionUtils.isNotEmpty(verificationEntities)) {
      List<OrderConfigVerificationVo> voList = (List<OrderConfigVerificationVo>) this.nebulaToolkitService.copyCollectionByWhiteList(verificationEntities, OrderConfigVerificationEntity.class, OrderConfigVerificationVo.class, HashSet.class, ArrayList.class);
      orderConfigVo.setVerificationVoList(voList);
    }
    // 查询生效范围
    OrderConfigDimensionDto orderConfigDimensionDto = new OrderConfigDimensionDto();
    orderConfigDimensionDto.setTenantCode(orderConfigVo.getTenantCode());
    orderConfigDimensionDto.setConfigCode(orderConfigVo.getConfigCode());
    List<OrderConfigDimensionVo> orderConfigDimensionVoList = this.orderConfigDimensionRepository.findByConditions(orderConfigDimensionDto);
    if (CollectionUtils.isNotEmpty(orderConfigDimensionVoList)) {
      // 查询客户或组织名称
      List<String> codes = orderConfigDimensionVoList.stream().map(OrderConfigDimensionVo::getCode).collect(Collectors.toList());
      if (EffectiveDimensionEnum.CUSTOMER.getDictCode().equals(orderConfigVo.getEffectiveType())) {
        List<CustomerVo> customerVos = this.customerVoService.findByCustomerCodes(codes);
        Map<String, CustomerVo> customerVoMap = customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity()));
        orderConfigDimensionVoList.forEach(ocd->{
          CustomerVo customerVo = customerVoMap.get(ocd.getCode());
          if (customerVo == null) {
            return;
          }
          ocd.setName(customerVo.getCustomerName());
        });
      } else if (EffectiveDimensionEnum.ORG.getDictCode().equals(orderConfigVo.getEffectiveType())) {
        List<OrgVo> orgVos = this.orgVoService.findByOrgCodes(codes);
        Map<String, OrgVo> orgVoMap = orgVos.stream().collect(Collectors.toMap(OrgVo::getOrgCode, Function.identity()));
        orderConfigDimensionVoList.forEach(ocd->{
          OrgVo orgVo = orgVoMap.get(ocd.getCode());
          if (orgVo == null) {
            return;
          }
          ocd.setName(orgVo.getOrgName());
        });
      }
    }
    orderConfigVo.setDimensionVoList(orderConfigDimensionVoList);
    return orderConfigVo;
  }

  @Override
  @Transactional
  public void create(OrderConfigDto dto) {
    this.createValidation(dto);
    if (StringUtils.isBlank(dto.getConfigCode())) {
      // redis生物料编码code
      List<String> codeList = this.generateCodeService.generateCode(OrderConfigConstant.ORDER_CONFIG_CODE, 1);
      Validate.isTrue(CollectionUtils.isNotEmpty(codeList), "添加信息时，生成订单配置编码失败！");
      OrderConfigEntity dbEntity = this.orderConfigRepository.findByConfigCode(TenantUtils.getTenantCode(), codeList.get(0));
      Validate.isTrue(Objects.isNull(dbEntity), String.format("添加信息时，订单配置编码[%s]已存在！", codeList.get(0)));
      dto.setConfigCode(codeList.get(0));
      dto.setTenantCode(TenantUtils.getTenantCode());
      dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    } else {
      OrderConfigEntity dbEntity = this.orderConfigRepository.findByConfigCode(TenantUtils.getTenantCode(), dto.getConfigCode());
      Validate.isTrue(Objects.isNull(dbEntity), String.format("添加信息时，订单配置编码[%s]已存在！", dto.getConfigCode()));
    }
    OrderConfigEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, OrderConfigEntity.class, HashSet.class, ArrayList.class);
    this.orderConfigRepository.save(entity);
    // 新增订单校验设置
    this.saveBatchVerification(entity, dto.getVerificationDtoList());
    // 新增订单配置维度
    this.saveBatchDimension(entity, dto.getDimensionDtoList());
  }

  @Override
  @Transactional
  public void update(OrderConfigDto dto) {
    this.updateValidation(dto);
    OrderConfigEntity oldEntity = this.orderConfigRepository.getById(dto.getId());
    Validate.notNull(oldEntity, "修改信息不存在");
    OrderConfigEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, OrderConfigEntity.class, HashSet.class, ArrayList.class);
    this.orderConfigRepository.updateById(entity);
    // 删除订单校验设置
    this.orderVerificationRepository.deleteByConfigCode(entity.getTenantCode(), Sets.newHashSet(entity.getConfigCode()));
    // 新增订单校验设置
    this.saveBatchVerification(entity, dto.getVerificationDtoList());
    // 删除订单配置维度
    this.orderConfigDimensionRepository.deleteByConfigCode(entity.getTenantCode(), Sets.newHashSet(entity.getConfigCode()));
    // 新增订单配置维度
    this.saveBatchDimension(entity, dto.getDimensionDtoList());
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<OrderConfigEntity> entities = this.orderConfigRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.orderConfigRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<OrderConfigEntity> entities = this.orderConfigRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.orderConfigRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<OrderConfigEntity> entities = this.orderConfigRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.orderConfigRepository.updateDelFlagByIds(ids);
    Set<String> orderConfigCodes = entities.stream().map(OrderConfigEntity::getConfigCode).collect(Collectors.toSet());
    this.orderVerificationRepository.deleteByConfigCode(TenantUtils.getTenantCode(), orderConfigCodes);
    this.orderConfigDimensionRepository.deleteByConfigCode(TenantUtils.getTenantCode(), orderConfigCodes);
  }

  /**
   * Feign查询客户信息
   */
  private CustomerVo findByCustomerCode(String customerCode){
    try {
      return customerVoService.findDetailsByIdOrCode(null, customerCode);
    } catch (Exception e) {
      log.error("Feign查询客户信息异常：{}", e.getMessage(), e);
    }
    return null;
  }

  @Override
  public OrderConfigVo findByDefaultAndOrderType(String orderType) {
    if (StringUtils.isBlank(orderType)) {
      return null;
    }
    OrderConfigEntity entity = this.orderConfigRepository.findByDefaultAndOrderType(TenantUtils.getTenantCode(), orderType);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, OrderConfigVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 批量新增订单校验设置
   */
  private void saveBatchVerification(OrderConfigEntity entity, List<OrderConfigVerificationDto> verificationDtoList) {
    if (CollectionUtils.isEmpty(verificationDtoList)) {
      return;
    }
    List<OrderConfigVerificationEntity> orderVerificationEntities = Lists.newArrayList();
    verificationDtoList.forEach(d->{
      OrderConfigVerificationEntity verificationEntity = new OrderConfigVerificationEntity();
      verificationEntity.setTenantCode(entity.getTenantCode());
      verificationEntity.setConfigCode(entity.getConfigCode());
      Validate.notBlank(d.getName(), "订单校验策略实现名称不能为空");
      Validate.notBlank(d.getTitle(), "订单校验策略实现标题不能为空");
      verificationEntity.setName(d.getName());
      verificationEntity.setTitle(d.getTitle());
      verificationEntity.setIsOpen(d.getIsOpen());
      orderVerificationEntities.add(verificationEntity);
    });
    if (CollectionUtils.isNotEmpty(orderVerificationEntities)) {
      this.orderVerificationRepository.saveBatch(orderVerificationEntities);
    }
  }

  /**
   * 批量新增订单配置维度
   */
  private void saveBatchDimension(OrderConfigEntity entity, List<OrderConfigDimensionDto> dimensionDtoList) {
    if (CollectionUtils.isEmpty(dimensionDtoList)) {
      return;
    }
    List<OrderConfigDimensionEntity> orderConfigDimensionEntities = Lists.newArrayList();
    dimensionDtoList.forEach(d->{
      Validate.notBlank(d.getCode(), "订单生效维度编码不能为空");
      OrderConfigDimensionEntity dimensionEntity = new OrderConfigDimensionEntity();
      dimensionEntity.setTenantCode(entity.getTenantCode());
      dimensionEntity.setConfigCode(entity.getConfigCode());
      dimensionEntity.setOrderType(entity.getOrderType());
      dimensionEntity.setType(entity.getEffectiveType());
      dimensionEntity.setCode(d.getCode());
      orderConfigDimensionEntities.add(dimensionEntity);
    });
    if (CollectionUtils.isNotEmpty(orderConfigDimensionEntities)) {
      this.orderConfigDimensionRepository.saveBatch(orderConfigDimensionEntities);
    }
  }

  /**
   * 新增校验差异部分
   * @param dto
   */
  private void createValidation(OrderConfigDto dto) {
    dto.setId(null);
    this.validation(dto);
  }

  /**
   * 编辑校验差异部分
   * @param dto
   */
  private void updateValidation(OrderConfigDto dto) {
    Validate.notBlank(dto.getId(), "ID不能为空");
    Validate.notBlank(dto.getConfigCode(), "订单配置编码不能为空");
    this.validation(dto);
  }

  /**
   * 公共校验部分
   * @param dto
   *
   */
  private void validation(OrderConfigDto dto) {
    Validate.notBlank(dto.getOrderType(), "订单类型不能为空");
    Validate.notNull(dto.getIsDefault(), "是否默认设定不能为空");
    Validate.notEmpty(dto.getVerificationDtoList(), "订单配置校验设置不能为空");
    OrderTypeEnum orderTypeEnum = OrderTypeEnum.findByCode(dto.getOrderType());
    Validate.notNull(orderTypeEnum, "订单类型不正确");
    if (dto.getIsDefault()) {
      OrderConfigVo orderConfigVo = this.findByDefaultAndOrderType(dto.getOrderType());
      Validate.isTrue(Objects.isNull(orderConfigVo) || orderConfigVo.getId().equals(dto.getId()), String.format("%s类型已存在默认配置，请勿重复设置", orderTypeEnum.getValue()));
    } else {
      Validate.notBlank(dto.getEffectiveType(), "生效类型不能为空");
      EffectiveDimensionEnum effectiveDimensionEnum = EffectiveDimensionEnum.findByCode(dto.getEffectiveType());
      Validate.notNull(effectiveDimensionEnum, "生效类型不正确，客户 customer；组织 org");
      Validate.notEmpty(dto.getDimensionDtoList(), "非默认设定时生效维度不能为空");
      // 订单类型+客户编码，订单类型+组织编码作为唯一
      Set<String> codes = dto.getDimensionDtoList().stream().map(OrderConfigDimensionDto::getCode).collect(Collectors.toSet());
      OrderConfigDimensionDto orderConfigDimensionDto = new OrderConfigDimensionDto();
      orderConfigDimensionDto.setTenantCode(TenantUtils.getTenantCode());
      orderConfigDimensionDto.setCodes(codes);
      List<OrderConfigDimensionVo> orderConfigDimensionVoList = this.orderConfigDimensionRepository.findByConditions(orderConfigDimensionDto);
      List<OrderConfigDimensionVo> existConfigList = Lists.newArrayList();
      orderConfigDimensionVoList.forEach(ocd->{
        // 排除自己，订单号+客户或组织编码相同时，及存在重复数据
        if (!ocd.getConfigCode().equals(dto.getConfigCode()) && ocd.getOrderType().equals(dto.getOrderType()) && codes.contains(ocd.getCode())) {
          existConfigList.add(ocd);
        }
      });
      Validate.isTrue(CollectionUtils.isEmpty(existConfigList), "当前订单类型已存在包含生效范围的配置信息（包括禁用状态）！");
    }
    // 根据订单类型检查订单校验设置
    OrderValidateDto orderValidateDto = this.nebulaToolkitService.copyObjectByWhiteList(dto, OrderValidateDto.class, HashSet.class, ArrayList.class);
    List<OrderValidateDetailDto> voList = (List<OrderValidateDetailDto>) this.nebulaToolkitService.copyCollectionByWhiteList(dto.getVerificationDtoList(), OrderConfigVerificationDto.class, OrderValidateDetailDto.class, HashSet.class, ArrayList.class);
    orderValidateDto.setDetailDtoList(voList);
    // TODO 所有校验策略实现之后打开
    //orderTypeStrategy.validate(dto.getOrderType(), orderValidateVo);
  }
}
