package com.biz.crm.dms.business.order.config.local.service.internal;

import com.biz.crm.dms.business.order.common.sdk.vo.OrderConfigVerificationVo;
import com.biz.crm.dms.business.order.common.sdk.vo.OrderConfigVo;
import com.biz.crm.dms.business.order.config.local.entity.OrderConfigDimensionEntity;
import com.biz.crm.dms.business.order.config.local.entity.OrderConfigEntity;
import com.biz.crm.dms.business.order.config.local.entity.OrderConfigVerificationEntity;
import com.biz.crm.dms.business.order.config.local.repository.OrderConfigDimensionRepository;
import com.biz.crm.dms.business.order.config.local.repository.OrderConfigRepository;
import com.biz.crm.dms.business.order.config.local.repository.OrderVerificationRepository;
import com.biz.crm.dms.business.order.config.sdk.enums.EffectiveDimensionEnum;
import com.biz.crm.dms.business.order.config.sdk.service.OrderValidateVoService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;

import java.util.stream.Collectors;

/**
 * 订单配置Vo服务接口实现类
 *
 * @author pengxi
 * @date 2021/12/30
 */
@Slf4j
@Service
public class OrderValidateVoServiceImpl implements OrderValidateVoService {

  @Autowired(required = false)
  private OrderConfigRepository orderConfigRepository;

  @Autowired(required = false)
  private OrderVerificationRepository orderVerificationRepository;

  @Autowired(required = false)
  private OrderConfigDimensionRepository orderConfigDimensionRepository;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  /**
   * Feign查询客户信息
   */
  private CustomerVo findByCustomerCode(String customerCode){
    try {
      return customerVoService.findDetailsByIdOrCode(null, customerCode);
    } catch (Exception e) {
      log.error("Feign查询客户信息异常：{}", e.getMessage(), e);
    }
    return null;
  }

  /**
   * 根据优先级规则获取订单配置信息
   * 第一优先：客户编码
   * 第二优先：客户所属组织
   * 第三优先：默认设置（系统初始化）
   *
   * @param orderType 订单类型
   * @param customerCode 客户编码
   * @return OrderConfigVo
   */
  @Override
  public OrderConfigVo findByOrderTypeAndCustomerCode(String orderType, String customerCode) {
    if (StringUtils.isAnyBlank(orderType, customerCode)) {
      return null;
    }
    OrderConfigVo orderConfigVo = null;
    // 1、客户编码--查询客户维度上设置的订单配置
    List<OrderConfigDimensionEntity> customerList = this.orderConfigDimensionRepository.findByOrderTypeAndTypeAndCodeIn(TenantUtils.getTenantCode(), orderType, EffectiveDimensionEnum.CUSTOMER.getDictCode(), Collections.singleton(customerCode));
    if (CollectionUtils.isNotEmpty(customerList)) {
      orderConfigVo = this.findByConfigDimension(customerList);
    }
    if (orderConfigVo!=null) {
      return orderConfigVo;
    }
    // 2、查询组织维度上设置的订单配置
    // --调用MDM侧，查询客户所属组织
    CustomerVo customerVo = this.findByCustomerCode(customerCode);
    if(customerVo != null && StringUtils.isNotBlank(customerVo.getOrgCode())) {
      // --使用组织编码，查询组织维度上设置的订单配置
      List<OrderConfigDimensionEntity> orgList = this.orderConfigDimensionRepository.findByOrderTypeAndTypeAndCodeIn(TenantUtils.getTenantCode(), orderType, EffectiveDimensionEnum.ORG.getDictCode(), Collections.singleton(customerVo.getOrgCode()));
      if (CollectionUtils.isNotEmpty(orgList)) {
        orderConfigVo = this.findByConfigDimension(orgList);
      }
    }
    if (orderConfigVo!=null) {
      return orderConfigVo;
    }
    // 3、查询系统默认的订单配置
    OrderConfigEntity orderConfigEntity = this.orderConfigRepository.findByDefaultAndOrderType(TenantUtils.getTenantCode(), orderType);
    if (orderConfigEntity == null) {
      return null;
    }
    orderConfigVo = this.nebulaToolkitService.copyObjectByWhiteList(orderConfigEntity, OrderConfigVo.class, HashSet.class, ArrayList.class);
    List<OrderConfigVerificationEntity> verificationEntities = this.orderVerificationRepository.findByConfigCodeIn(TenantUtils.getTenantCode(), Collections.singletonList(orderConfigEntity.getConfigCode()));
    if (CollectionUtils.isNotEmpty(verificationEntities)) {
      List<OrderConfigVerificationVo> voList = (List<OrderConfigVerificationVo>) this.nebulaToolkitService.copyCollectionByWhiteList(verificationEntities, OrderConfigVerificationEntity.class, OrderConfigVerificationVo.class, HashSet.class, ArrayList.class);
      orderConfigVo.setVerificationVoList(voList);
    }
    return orderConfigVo;
  }

  /**
   * 根据订单配置维度查询订单配置
   * @param customerList
   * @return
   */
  private OrderConfigVo findByConfigDimension(List<OrderConfigDimensionEntity> customerList) {
    if (CollectionUtils.isEmpty(customerList)) {
      return null;
    }
    List<String> configCodes = customerList.stream().map(OrderConfigDimensionEntity::getConfigCode).distinct().collect(Collectors.toList());
    OrderConfigEntity orderConfigEntity = this.orderConfigRepository.findByConfigCode(TenantUtils.getTenantCode(), configCodes.get(0));
    if (orderConfigEntity == null) {
      return null;
    }
    OrderConfigVo orderConfigVo = this.nebulaToolkitService.copyObjectByWhiteList(orderConfigEntity, OrderConfigVo.class, HashSet.class, ArrayList.class);
    List<OrderConfigVerificationEntity> verificationEntities = this.orderVerificationRepository.findByConfigCodeIn(TenantUtils.getTenantCode(), configCodes);
    if (CollectionUtils.isNotEmpty(verificationEntities)) {
      List<OrderConfigVerificationVo> voList = (List<OrderConfigVerificationVo>) this.nebulaToolkitService.copyCollectionByWhiteList(verificationEntities, OrderConfigVerificationEntity.class, OrderConfigVerificationVo.class, HashSet.class, ArrayList.class);
      orderConfigVo.setVerificationVoList(voList);
    }
    return orderConfigVo;
  }

}
