package com.biz.crm.common.jdb.file.local.internal;


import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.StringPool;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.local.entity.UuidCrmOpEntity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.common.jdb.file.local.constant.TbAttachmentConstant;
import com.biz.crm.common.jdb.file.local.entity.TbAttachmentEntity;
import com.biz.crm.common.jdb.file.local.repository.TbAttachmentRepository;
import com.biz.crm.common.jdb.file.sdk.service.TbAttachmentService;
import com.biz.crm.common.jdb.file.sdk.vo.TbAttachmentVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.collect.Lists;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 附件管理 Service 实现.
 * <p>
 *
 * @author liukai
 * @version v1.0
 */
@Service
@Slf4j
public class TbAttachmentServiceImpl implements TbAttachmentService {

    @Autowired(required = false)
    private TbAttachmentRepository tbAttachmentRepository;

    @Autowired(required = false)
    private FileHandleService fileHandleService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Override
    public List<TbAttachmentVo> findFileList(String businessKey) {
        if (StringUtils.isBlank(businessKey)) {
            return Collections.emptyList();
        }
        return this.tbAttachmentRepository.findFileList(Collections.singletonList(businessKey), null);
    }

    @Override
    public List<TbAttachmentVo> findFileList(String businessKey, String attachmentType) {
        if (StringUtils.isBlank(businessKey)) {
            return Collections.emptyList();
        }
        return this.tbAttachmentRepository.findFileList(Collections.singletonList(businessKey), attachmentType);
    }

    @Override
    public List<TbAttachmentVo> findFileList(List<String> businessKeys) {
        if (CollectionUtils.isEmpty(businessKeys)) {
            return Collections.emptyList();
        }
        return this.tbAttachmentRepository.findFileList(businessKeys, null);
    }

    @Override
    public List<TbAttachmentVo> findFileList(List<String> businessKeys, String attachmentType) {
        if (CollectionUtils.isEmpty(businessKeys)) {
            return Collections.emptyList();
        }
        return this.tbAttachmentRepository.findFileList(businessKeys, attachmentType);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveFile(String fileCode, String businessKey) {
        this.saveFile(fileCode, businessKey, null);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveFile(String fileCode, String businessKey, String attachmentType) {
        Assert.hasLength(fileCode, "文件id不能为空");
        Assert.hasLength(businessKey, "业务id不能为空");
        this.saveFile(Collections.singletonList(fileCode), businessKey, attachmentType);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveFile(List<String> fileCodes, String businessKey) {
        this.saveFile(fileCodes, businessKey, null);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveFile(List<String> fileCodes, String businessKey, String attachmentType) {
        Assert.notEmpty(fileCodes, "文件id集合不能为空");
        Assert.hasLength(businessKey, "业务id不能为空");
        List<TbAttachmentEntity> attachmentEntityList = new ArrayList<>();
        fileCodes.forEach(item -> {
            Assert.hasLength(item, "文件id不能为空");
        });
        List<OrdinaryFileVo> files = fileHandleService.findByIds(fileCodes);
        Assert.notEmpty(files, "文件不存在");
        Assert.isTrue(files.size() == fileCodes.size(), "文件id数量与实际文件数量不一致");
        FacturerUserDetails loginUserDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
        String postId = loginUserDetails.getPostId();
        files.forEach(file -> {
            TbAttachmentEntity attachment = new TbAttachmentEntity();
            attachment.setAttachmentType(attachmentType);
            attachment.setPositionId(postId);
            attachment.setBusinessKey(businessKey);
            String attachmentTitle = file.getOriginalFileName();
            if (StringUtils.isNotBlank(attachmentTitle)) {
                attachmentTitle = attachmentTitle.replace(".".concat(file.getPrefix()), "");
            }
            attachment.setAttachmentTitle(attachmentTitle);
            attachment.setRealPath(file.getRelativeLocal());
            attachment.setExtend(file.getPrefix());
            attachment.setFileCode(file.getId());
            attachment.setFileVersion(TbAttachmentConstant.FILE_NEW_VERSION);
            attachmentEntityList.add(attachment);
        });
        tbAttachmentRepository.saveBatch(attachmentEntityList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<TbAttachmentVo> saveOrUpdateBatch(List<TbAttachmentVo> files) {
        Assert.notEmpty(files, "文件不能为空");
        files.forEach(file -> {
            Assert.hasLength(file.getBusinessKey(), "业务id不能为空");
            Assert.hasLength(file.getFileCode(), "文件id不能为空");
            file.setFileVersion(TbAttachmentConstant.FILE_NEW_VERSION);
        });
        List<String> fileCodes = files.stream().map(TbAttachmentVo::getFileCode).filter(StringUtils::isNotEmpty).distinct().collect(Collectors.toList());
        List<OrdinaryFileVo> fileVos = fileHandleService.findByIds(fileCodes);
        Map<String, OrdinaryFileVo> fileVoMap = fileVos.stream().collect(Collectors.toMap(OrdinaryFileVo::getId, item -> item));
        Collection<TbAttachmentEntity> entityList = nebulaToolkitService.copyCollectionByBlankList(files,
                TbAttachmentVo.class, TbAttachmentEntity.class, HashSet.class, ArrayList.class);
        entityList.forEach(file -> {
            OrdinaryFileVo fileVo = fileVoMap.get(file.getFileCode());
            if (Objects.nonNull(fileVo)) {
                String attachmentTitle = fileVo.getOriginalFileName();
                if (StringUtils.isNotBlank(attachmentTitle)) {
                    attachmentTitle = attachmentTitle.replace(".".concat(fileVo.getPrefix()), "");
                }
                file.setAttachmentTitle(attachmentTitle);
                file.setRealPath(fileVo.getRelativeLocal());
                file.setExtend(fileVo.getPrefix());
            }
            file.setUpdateInfoToNull();
        });
        tbAttachmentRepository.saveOrUpdateBatch(entityList);
        return (List<TbAttachmentVo>) nebulaToolkitService.copyCollectionByBlankList(entityList, TbAttachmentEntity.class,
                TbAttachmentVo.class, HashSet.class, ArrayList.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByBusinessKey(String businessKey) {
        Assert.hasLength(businessKey, "业务id不能为空");
        this.delByBusinessKey(Collections.singletonList(businessKey));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByBusinessKey(List<String> businessKeys) {
        Assert.notEmpty(businessKeys, "业务id不能为空");
        List<TbAttachmentEntity> files = tbAttachmentRepository.findByBusinessKeys(businessKeys);
        if (CollectionUtils.isEmpty(files)) {
            return;
        }
        tbAttachmentRepository.removeByIds(files.stream().map(UuidCrmOpEntity::getId).collect(Collectors.toList()));
        fileHandleService.deleteFile(files.stream().map(TbAttachmentEntity::getFileCode).filter(StringUtils::isNotEmpty).collect(Collectors.toList()));
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByFileCode(String fileCode) {
        Assert.hasLength(fileCode, "文件id不能为空");
        this.delByFileCode(Collections.singletonList(fileCode));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByFileCode(List<String> fileCodes) {
        Assert.notEmpty(fileCodes, "文件id不能为空");
        LambdaUpdateWrapper<TbAttachmentEntity> wrapper = new LambdaUpdateWrapper<>();
        wrapper.in(TbAttachmentEntity::getFileCode, fileCodes);
        tbAttachmentRepository.remove(wrapper);
        fileHandleService.deleteFile(fileCodes);
    }


}
