package com.biz.crm.kms.business.supermarket.parameter.local.service.internal;

import com.biz.crm.kms.business.supermarket.parameter.local.entity.SupermarketParameterLabelEntity;
import com.biz.crm.kms.business.supermarket.parameter.sdk.dto.SupermarketParameterLabelDto;
import com.biz.crm.kms.business.supermarket.parameter.sdk.event.SupermarketParameterLabelEventListener;
import com.biz.crm.kms.business.supermarket.parameter.sdk.service.SupermarketParameterLabelVoService;
import com.biz.crm.kms.business.supermarket.parameter.local.repository.SupermarketParameterLabelRepository;
import com.biz.crm.kms.business.supermarket.parameter.sdk.vo.SupermarketParameterLabelVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

/**
 * 商超抓单参数表附属标签表(supermarketParameterLabel)表服务实现类
 *
 * @author xi.peng
 * @since 2022-09-23 10:49:13
 */
@Slf4j
@Service
public class SupermarketParameterLabelVoServiceImpl implements SupermarketParameterLabelVoService {

  @Autowired(required = false)
  private SupermarketParameterLabelRepository supermarketParameterLabelRepository;

  @Autowired(required = false)
  private List<SupermarketParameterLabelEventListener> supermarketParameterLabelEventListeners;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public SupermarketParameterLabelVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    SupermarketParameterLabelEntity entity = this.supermarketParameterLabelRepository.findById(id);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, SupermarketParameterLabelVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<SupermarketParameterLabelVo> findParameterIds(List<String> parameterIds) {
    if (CollectionUtils.isEmpty(parameterIds)) {
      return null;
    }
    List<SupermarketParameterLabelEntity> labelEntities = this.supermarketParameterLabelRepository.findByParameterIds(parameterIds);
    if (CollectionUtils.isEmpty(labelEntities)) {
      return null;
    }
    return  (List<SupermarketParameterLabelVo>) this.nebulaToolkitService.copyCollectionByWhiteList(labelEntities, SupermarketParameterLabelEntity.class, SupermarketParameterLabelVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void createBatch(List<SupermarketParameterLabelDto> dtoList) {
    this.validation(dtoList);
    List<SupermarketParameterLabelEntity> optionEntities = (List<SupermarketParameterLabelEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(dtoList, SupermarketParameterLabelDto.class, SupermarketParameterLabelEntity.class, HashSet.class, ArrayList.class);
    this.supermarketParameterLabelRepository.saveBatch(optionEntities);
    // 发送批量创建通知
    if (CollectionUtils.isNotEmpty(supermarketParameterLabelEventListeners)) {
      List<SupermarketParameterLabelVo> voList = (List<SupermarketParameterLabelVo>) this.nebulaToolkitService.copyCollectionByWhiteList(optionEntities, SupermarketParameterLabelEntity.class, SupermarketParameterLabelVo.class, HashSet.class, ArrayList.class);
      this.supermarketParameterLabelEventListeners.forEach(event -> event.onCreateBatch(voList));
    }
  }

  @Override
  @Transactional
  public void deleteByParameterId(String parameterId) {
    Validate.notBlank(parameterId, "接口参数ID不能为空！");
    List<SupermarketParameterLabelEntity> optionEntities = this.supermarketParameterLabelRepository.findByParameterId(parameterId);
    if (CollectionUtils.isEmpty(optionEntities)) {
      // 如果已删除或不存在就不继续向下执行
      return;
    }
    this.supermarketParameterLabelRepository.deleteByParameterId(parameterId);
    // 根据parameterId删除时发送通知
    if (CollectionUtils.isNotEmpty(supermarketParameterLabelEventListeners)) {
      this.supermarketParameterLabelEventListeners.forEach(event -> event.onDeleteByParameterId(parameterId));
    }
  }

  private void validation(List<SupermarketParameterLabelDto> dtoList) {
    Validate.notEmpty(dtoList, "标签集合对象不能为空！");
    for (int i = 0; i < dtoList.size(); i++) {
      SupermarketParameterLabelDto option = dtoList.get(i);
      Validate.notBlank(option.getParameterId(), "参数表ID不能为空！");
      Validate.notBlank(option.getNameLabel(), "标签名称不能为空！");
      Validate.notBlank(option.getValueLabel(), "标签值不能为空！");
      if (null == option.getSortIndex()) {
        // 前端未设置排序值默认按传入顺序排序
        option.setSortIndex(i);
      }
      //长度校验
      Validate.isTrue(option.getNameLabel().length()<=64, "标签名称最多64位字符串");
      Validate.isTrue(option.getValueLabel().length()<=64, "标签值最多64位字符串");
    }
  }
}
