package com.biz.crm.mdm.business.cost.center.local.controller;

import java.util.List;

import com.biz.crm.mdm.business.cost.center.sdk.service.CostCenterVoService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.cost.center.sdk.dto.CostCenterDto;
import com.biz.crm.mdm.business.cost.center.sdk.dto.CostCenterProfitsSelectDto;
import com.biz.crm.mdm.business.cost.center.sdk.dto.CostCenterSelectDto;
import com.biz.crm.mdm.business.cost.center.sdk.service.CostCenterService;
import com.biz.crm.mdm.business.cost.center.sdk.vo.CostCenterVo;
import com.biz.crm.mdm.business.cost.center.sdk.vo.UserCostCenterVo;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.biz.crm.mn.third.system.sap.fi.sdk.dto.SapFiCostCenterDto;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;

/**
 * @author: chenlong
 * @date: 2022/11/22 14:42
 * @description: 成本中心管理(CostCenter)表相关的http接口
 */
@RestController
@RequestMapping("/v1/costCenter")
@Slf4j
@Api(tags = "MDM-成本中心")
public class CostCenterController {

    @Autowired
    private CostCenterService costCenterService;

    @Autowired
    private CostCenterVoService costCenterVoService;

    /**
     * 分页查询所有数据
     *
     * @param pageable      分页对象
     * @param costCenterDto 查询dto
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<CostCenterVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                       @ApiParam(name = "costCenterDto", value = "成本中心dto") CostCenterDto costCenterDto) {
        try {
            Page<CostCenterVo> page = this.costCenterService.findByConditions(pageable, costCenterDto);
            return Result.ok(page);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("{id}")
    public Result<CostCenterVo> findById(@PathVariable @ApiParam(name = "id", value = "主键id") String id) {
        try {
            CostCenterVo centerVo = this.costCenterService.findById(id);
            return Result.ok(centerVo);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     * @return 删除结果
     */
    @ApiOperation(value = "删除数据")
    @DeleteMapping("delete")
    public Result<?> delete(@RequestParam("ids") List<String> ids) {
        try {
            this.costCenterService.delete(ids);
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 启用
     *
     * @param ids 主键列表
     * @return 启用结果
     */
    @ApiOperation(value = "启用", httpMethod = "PATCH")
    @PatchMapping("/enable")
    public Result<?> enable(@RequestBody List<String> ids) {
        try {
            this.costCenterService.enableBatch(ids);
            return Result.ok("启用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 禁用
     *
     * @param ids 主键列表
     * @return 禁用结果
     */
    @ApiOperation(value = "禁用", httpMethod = "PATCH")
    @PatchMapping("/disable")
    public Result<?> disable(@RequestBody List<String> ids) {
        try {
            this.costCenterService.disableBatch(ids);
            return Result.ok("禁用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 成本中心下拉
     *
     * @param dto 成本中心查询参数
     * @return 下拉数据
     */
    @ApiOperation(value = "成本中心下拉")
    @PostMapping("findFormSelectList")
    public Result<List<CommonSelectVo>> findFormSelectList(@ApiParam(value = "成本中心下拉参数dto") @RequestBody CostCenterSelectDto dto) {
        try {
            List<CommonSelectVo> list = costCenterService.findFormSelectList(dto);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 利润中心下拉
     *
     * @param dto 利润中心查询参数
     * @return 下拉数据
     */
    @ApiOperation(value = "利润中心下拉")
    @PostMapping("findProfitsFormSelectList")
    public Result<List<CommonSelectVo>> findProfitsFormSelectList(@ApiParam(value = "利润中心下拉参数dto") @RequestBody CostCenterProfitsSelectDto dto) {
        try {
            List<CommonSelectVo> list = costCenterService.findProfitsFormSelectList(dto);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


//    /**
//     * 通过当前登录人，获取关联的成本中心和业态业务单元
//     *
//     * @return UserCostCenterVo
//     */
//    @ApiOperation(value = "通过当前登录人，获取关联的成本中心和业态业务单元")
//    @GetMapping("/getUserCostCenter")
//    public Result<UserCostCenterVo> getUserCostCenter(@PathVariable @ApiParam(name = "userCode", value = "用户编码") String userCode) {
//        try {
//            UserCostCenterVo centerVo = this.costCenterService.getUserCostCenter(userCode);
//            return Result.ok(centerVo);
//        } catch (RuntimeException e) {
//            log.error(e.getMessage(), e);
//            return Result.error(e.getMessage());
//        }
//    }

    /**
     * 批量拉取 MDG 成本中心数据
     *
     * @param dto 请求参数
     * @return 成本中心列表
     */
    @ApiOperation(value = "批量拉取成本中心数据")
    @PostMapping("/pullCostCenterList")
    public Result<?> pullCostCenterList(@RequestBody SapFiCostCenterDto dto) {
        try {
            this.costCenterService.pullCostCenterList(dto);
            return Result.ok("数据拉取成功！");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "通过成本中心编码批量查询成本中心数据")
    @GetMapping("/findByCodes")
    public Result<List<CostCenterVo>> findByCodes(@RequestParam("costCenterCodes") List<String> costCenterCodes,
                                                  @RequestParam("effective") String effective) {
        try {
            return Result.ok(this.costCenterVoService.findByCodes(costCenterCodes, effective));
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
