package com.biz.crm.mdm.business.cost.center.local.imports;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.cost.center.local.entity.CostCenterRelation;
import com.biz.crm.mdm.business.cost.center.local.helper.ICostCenterRelationHelper;
import com.biz.crm.mdm.business.cost.center.local.repository.CostCenterRepository;
import com.biz.crm.mdm.business.cost.center.local.service.ICostCenterRelationService;
import com.biz.crm.mdm.business.cost.center.sdk.constant.CostCenterConstant;
import com.biz.crm.mdm.business.cost.center.sdk.dto.CostCenterRelationDto;
import com.biz.crm.mdm.business.cost.center.sdk.vo.CostCenterRelationImportVo;
import com.biz.crm.mdm.business.cost.center.sdk.vo.CostCenterVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.retailer.sdk.service.CustomerRetailerVoService;
import com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgPaginationDto;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.ExceptionStackMsgUtil;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @Description 成本中心映射关系导入
 * @Author YangWei
 * @Date 2023/2/2 下午4:17
 */
@Slf4j
@Component
public class ICostCenterRelationImportProcess implements ImportProcess<CostCenterRelationImportVo> {

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private DictDataVoService dictDataVoService;

    @Resource
    private ICostCenterRelationService iCostCenterRelationService;

    @Resource
    private ICostCenterRelationHelper iCostCenterRelationHelper;

    @Resource
    private CustomerRetailerVoService customerRetailerVoService;

    @Resource
    private CostCenterRepository costCenterRepository;

    @Resource
    private CustomerChannelVoService customerChannelVoService;

    @Resource
    private SalesOrgVoService salesOrgVoService;

    @Resource
    private CustomerVoService customerVoService;

    /**
     * 数据字典
     */
    private Map<String, List<DictDataVo>> dictMap = Maps.newHashMap();

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<Integer, String> execute(LinkedHashMap<Integer, CostCenterRelationImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        final Optional<CostCenterRelationImportVo> first = data.values().stream().findFirst();
        if (!first.isPresent()) {
            return null;
        }
        try {
            //初始数据字典
            this.initDict();
            //校验
            CostCenterRelationDto dto = this.validate(first.get());
            //保存
            this.iCostCenterRelationService.create(dto);
        } catch (IllegalArgumentException e) {
            throw e;
        } catch (Exception e) {
            String error = ExceptionStackMsgUtil.stackMsg(e);
            log.error("成本中心映射关系导入异常:{}", error);
            throw new IllegalArgumentException("导入异常[" + error + "]");
        }
        return null;
    }

    /**
     * 初始数据字典
     * @param
     */
    private void initDict() {
        this.dictMap = this.dictDataVoService.findByDictTypeCodeList(
            Lists.newArrayList(CostCenterConstant.MDM_BUSINESS_FORMAT
                , CostCenterConstant.MDM_BUSINESS_UNIT, CostCenterConstant.TPM_EXPENSE_CENTER
                , CostCenterConstant.MDM_CUSTOMIZE_ORG));
    }

    /**
     * 校验
     * @param vo
     * @return {@link MonthBudgetDto}
     */
    private CostCenterRelationDto validate(CostCenterRelationImportVo vo) {
        //数据拷贝
        CostCenterRelationDto dto =
            this.nebulaToolkitService.copyObjectByBlankList(
                vo, CostCenterRelationDto.class, HashSet.class, ArrayList.class);
        //校验数据字典
        this.verifyDictCode(dto);
        //零售商
        //判断是否为垂直
        vo.setCustomerRetailerCode(StringUtils.hasText(vo.getCustomerRetailerCode()) ? vo.getCustomerRetailerCode() : vo.getCustomerRetailerName());
        if(BusinessUnitEnum.VERTICAL.getCode().equals(vo.getBusinessUnitCode())){
            Assert.hasText(vo.getCustomerRetailerCode(), "零售商必填");
        }
        //查询数据
        if(StringUtils.hasText(vo.getCustomerRetailerCode())){
            List<CustomerRetailerVo> customerRetailerVos = this.customerRetailerVoService
                .findByCodeOrName(Lists.newArrayList(vo.getCustomerRetailerCode()));
            Assert.notEmpty(customerRetailerVos, "未查询到零售商信息");
            dto.setCustomerRetailerCode(customerRetailerVos.get(0).getCustomerRetailerCode());
            dto.setCustomerRetailerName(customerRetailerVos.get(0).getCustomerRetailerName());
        }

        //成本中心
        vo.setCostCenterCode(StringUtils.hasText(vo.getCostCenterCode()) ? vo.getCostCenterCode() : vo.getCostCenterName());
        Assert.hasText(dto.getCostCenterCode(), "成本中心必填");
        List<CostCenterVo> costCenterVos = this.costCenterRepository
            .findByCodeOrName(Lists.newArrayList(dto.getCostCenterCode()));
        Assert.notEmpty(costCenterVos, "未查询到成本中心信息");
        dto.setCostCenterCode(costCenterVos.get(0).getCostCenterCode());
        dto.setCostCenterName(costCenterVos.get(0).getCostCenterName());

        //客户渠道
        vo.setCustomerChannelCode(StringUtils.hasText(vo.getCustomerChannelCode()) ? vo.getCustomerChannelCode() : vo.getCustomerChannelName());
        if(StringUtils.hasText(dto.getCustomerChannelCode())){
            List<CustomerChannelVo> customerChannelVos = this.customerChannelVoService
                .findByCodeOrName(Lists.newArrayList(dto.getCustomerChannelCode()));
            Assert.notEmpty(customerChannelVos, "未查询到客户渠道信息");
            dto.setCustomerChannelCode(customerChannelVos.get(0).getCustomerChannelCode());
            dto.setCustomerChannelName(customerChannelVos.get(0).getCustomerChannelName());
        }
        String salesInstitutionCode = dto.getSalesInstitutionCode();
        String salesGroupCode = dto.getSalesGroupCode();
        String salesOrgCode = dto.getSalesOrgCode();
        if(StringUtils.hasText(salesInstitutionCode)
            || StringUtils.hasText(salesGroupCode) || StringUtils.hasText(salesOrgCode)){
            Assert.hasText(vo.getCustomerChannelCode(), "客户渠道不能为空");
            //销售机构	销售部门	销售组
            List<String> codeOrName = Lists.newArrayList();
            vo.setSalesInstitutionCode(StringUtils.hasText(vo.getSalesInstitutionCode()) ? vo.getSalesInstitutionCode() : vo.getSalesInstitutionName());
            if(StringUtils.hasText(salesInstitutionCode)){
                codeOrName.add(salesInstitutionCode);
            }
            vo.setSalesOrgCode(StringUtils.hasText(vo.getSalesOrgCode()) ? vo.getSalesOrgCode() : vo.getSalesOrgName());
            if(StringUtils.hasText(salesOrgCode)){
                codeOrName.add(salesOrgCode);
            }
            vo.setSalesGroupCode(StringUtils.hasText(vo.getSalesGroupCode()) ? vo.getSalesGroupCode() : vo.getSalesGroupName());
            if(StringUtils.hasText(salesGroupCode)){
                codeOrName.add(salesGroupCode);
            }
            List<SalesOrgVo> salesOrgAll = this.salesOrgVoService.findByCodeOrName(codeOrName);
            Assert.notEmpty(salesOrgAll, "未查询到销售组织信息");
            //渠道+业态+erp编码
            Assert.hasText(dto.getCustomerChannelName(), "渠道信息错误");
            Assert.hasText(dto.getBusinessFormatCode(), "业态信息错误");
            List<SalesOrgVo> salesOrgVos = salesOrgAll.stream().filter(o -> dto.getCustomerChannelCode().equals(o.getChannelCode())
                && dto.getBusinessFormatCode().equals(o.getBusinessFormatCode())).collect(Collectors.toList());
            Assert.notEmpty(salesOrgVos, "销售机构业态渠道不匹配");
            if(StringUtils.hasText(salesInstitutionCode)){
                List<SalesOrgVo> collect = salesOrgVos.stream().filter(o -> o.getSalesOrgCode().equals(salesInstitutionCode)
                    || o.getSalesOrgName().equals(salesInstitutionCode) || o.getErpCode().equals(salesInstitutionCode))
                    .collect(Collectors.toList());
                Assert.isTrue(collect.size() == 1, "未找到销售机构或者找到多个");
                SalesOrgVo salesOrgVo = collect.get(0);
                dto.setSalesInstitutionCode(salesOrgVo.getSalesOrgCode());
                dto.setSalesInstitutionName(salesOrgVo.getSalesOrgName());
            }
            if(StringUtils.hasText(salesOrgCode)){
                Assert.hasText(dto.getSalesInstitutionCode(), "销售机构不能为空");
                List<SalesOrgVo> collect = salesOrgVos.stream().filter(o -> (o.getSalesOrgCode().equals(salesOrgCode)
                    || o.getSalesOrgName().equals(salesOrgCode) || o.getErpCode().equals(salesOrgCode))
                    && o.getParentCode().equals(dto.getSalesInstitutionCode()))
                    .collect(Collectors.toList());
                Assert.isTrue(collect.size() == 1, "未找到销售部门或者找到多个");
                SalesOrgVo salesOrgVo = collect.get(0);
                dto.setSalesOrgCode(salesOrgVo.getSalesOrgCode());
                dto.setSalesOrgName(salesOrgVo.getSalesOrgName());
            }
            if(StringUtils.hasText(salesGroupCode)){
                Assert.hasText(dto.getSalesOrgCode(), "销售部门不能为空");
                List<SalesOrgVo> collect = salesOrgVos.stream().filter(o -> (o.getSalesOrgCode().equals(salesGroupCode)
                    || o.getSalesOrgName().equals(salesGroupCode) || o.getErpCode().equals(salesGroupCode))
                    && o.getParentCode().equals(dto.getSalesOrgCode()) )
                    .collect(Collectors.toList());
                Assert.isTrue(collect.size() == 1, "未找到销售组或者找到多个");
                SalesOrgVo salesOrgVo = collect.get(0);
                dto.setSalesGroupCode(salesOrgVo.getSalesOrgCode());
                dto.setSalesGroupName(salesOrgVo.getSalesOrgName());
            }
        }
        vo.setCustomerCode(StringUtils.hasText(vo.getCustomerCode()) ? vo.getCustomerCode() : vo.getCustomerName());
        if(StringUtils.hasText(dto.getCustomerCode())){
            Assert.hasText(dto.getSalesInstitutionName(), "销售机构信息错误");
            Assert.hasText(dto.getCustomerChannelName(), "渠道信息错误");
            Assert.hasText(dto.getBusinessFormatCode(), "业态信息错误");
            //客户
            List<CustomerVo> customerVos = this.customerVoService.findByCodeOrName(Lists.newArrayList(dto.getCustomerCode()));
            Assert.notEmpty(customerVos, "未找到客户信息");
            List<CustomerVo> collect = customerVos.stream().filter(o -> o.getSalesInstitutionCode()
                .equals(dto.getSalesInstitutionCode()) && o.getCustomerChannelCode().equals(dto.getCustomerChannelCode())
                && o.getBusinessFormatCode().equals(dto.getBusinessFormatCode())).collect(Collectors.toList());
            Assert.isTrue(collect.size() == 1, "未找到客户或者找到多个");
            dto.setCustomerCode(collect.get(0).getCustomerCode());
            dto.setCustomerName(collect.get(0).getCustomerName());
        }

        return dto;
    }

    /**
     * 校验数据字典
     * @param vo
     */
    private void verifyDictCode(CostCenterRelationDto vo) {
        //业态
        vo.setBusinessFormatCode(this.findDictValueOrCode(vo.getBusinessFormatCode(), CostCenterConstant.MDM_BUSINESS_FORMAT));
        Assert.hasText(vo.getBusinessFormatCode(), "未配置该业态");
        //业务单元
        vo.setBusinessUnitCode(this.findDictValueOrCode(vo.getBusinessUnitCode(), CostCenterConstant.MDM_BUSINESS_UNIT));
        Assert.hasText(vo.getBusinessUnitCode(), "未配置该业务单元");
        //判断是否为垂直
        if(BusinessUnitEnum.VERTICAL.getCode().equals(vo.getBusinessUnitCode())){
            Assert.hasText(vo.getExpenseCenter(), "费用中心必填");
            Assert.hasText(vo.getCustomerRetailerRegion(), "区域必填");
        }
        if(StringUtils.hasText(vo.getExpenseCenter())){
            //费用归口
            vo.setExpenseCenter(this.findDictValueOrCode(vo.getExpenseCenter(), CostCenterConstant.TPM_EXPENSE_CENTER));
            Assert.hasText(vo.getExpenseCenter(), "未配置该费用中心");
        }
        if(StringUtils.hasText(vo.getCustomerRetailerRegion())){
            //区域
            if (!org.apache.commons.lang3.StringUtils.isEmpty(vo.getCustomerRetailerRegion())) {
                vo.setCustomerRetailerRegion(this.findDictValueOrCode(vo.getCustomerRetailerRegion(), CostCenterConstant.MDM_CUSTOMIZE_ORG));
                Assert.hasText(vo.getCustomerRetailerRegion(), "未配置该区域");
            }
        }
    }

    /**
     * 获取字典对应的value值
     *
     * @param dictValue
     * @param typeCode
     * @return
     */
    private String findDictValueOrCode(String dictValue, String typeCode) {
        DictDataVo dictDataVo = this.dictMap.get(typeCode).stream()
            .filter(a -> a.getDictValue().equals(dictValue) || a.getDictCode().equals(dictValue)).findFirst().orElse(null);
        if (!ObjectUtils.isEmpty(dictDataVo)) {
            return dictDataVo.getDictCode();
        }
        return null;
    }

    @Override
    public Integer getBatchCount() {
        //一次处理1条
        return 1;
    }

    @Override
    public Class<CostCenterRelationImportVo> findCrmExcelVoClass() {
        return CostCenterRelationImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "MDM_COST_CENTER_RELATION_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM-成本中心-映射关系导入";
    }

    @Override
    public String getBusinessCode() {
        return "MDM_COST_CENTER_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM-成本中心导入";
    }

}