package com.biz.crm.mdm.business.login.config.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.login.config.dto.IndexConfigDto;
import com.biz.crm.mdm.business.login.config.entity.IndexConfig;
import com.biz.crm.mdm.business.login.config.entity.IndexConfigExpand;
import com.biz.crm.mdm.business.login.config.repository.IndexConfigExpandRepository;
import com.biz.crm.mdm.business.login.config.repository.IndexConfigRepository;
import com.biz.crm.mdm.business.login.config.service.IndexConfigService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author jerry7
 * 登录配置service实现
 */
@Service
public class IndexConfigServiceImpl implements IndexConfigService {

  @Autowired(required = false)
  private IndexConfigRepository indexConfigRepository;

  @Autowired(required = false)
  private IndexConfigExpandRepository indexConfigExpandRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Override
  public Page<IndexConfig> findByConditions(Pageable pageable, IndexConfigDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new IndexConfigDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<IndexConfig> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return indexConfigRepository.findByConditions(page, dto);
  }

  @Override
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    indexConfigRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
  }

  @Override
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    indexConfigRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
  }

  @Override
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    indexConfigRepository.updateDelStatusByIdIn(DelFlagStatusEnum.DELETE, ids);
  }

  @Override
  public IndexConfig findById(String id) {
    if(StringUtils.isBlank(id)){
      return null;
    }
    return indexConfigRepository.getById(id);
  }

  @Override
  @Transactional
  public IndexConfig create(IndexConfig indexConfig) {
    Validate.notNull(indexConfig, "创建登录配置实体不能为空！");
    Validate.isTrue(StringUtils.isBlank(indexConfig.getId()), "创建登录配置ID不能有值！");
    this.createValidate(indexConfig);
    indexConfig.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    indexConfig.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    indexConfig.setCreateTime(new Date());
    indexConfig.setCreateName(loginUserService.getLoginAccountName());
    this.indexConfigRepository.save(indexConfig);
    //如果扩展配置不为空，添加配置扩展
    if (!CollectionUtils.isEmpty(indexConfig.getItems())) {
      for (IndexConfigExpand expand : indexConfig.getItems()) {
        expand.setConfigId(indexConfig.getId());
        expand.setTenantCode(TenantUtils.getTenantCode());
      }
      this.indexConfigExpandRepository.saveBatch(indexConfig.getItems());
    }
    return indexConfig;
  }

  @Override
  @Transactional
  public IndexConfig update(IndexConfig indexConfig) {
    Validate.notNull(indexConfig, "更新登录配置实体不能为空！");
    Validate.notBlank(indexConfig.getId(), "登录登录配置ID不能为空！");
    this.createValidate(indexConfig);
    this.indexConfigRepository.saveOrUpdate(indexConfig);
    //删除之前的配置扩展
    this.indexConfigExpandRepository.deleteExpandByConfId(indexConfig.getId());
    //如果扩展配置不为空，添加配置扩展
    if (!CollectionUtils.isEmpty(indexConfig.getItems())) {
      for (IndexConfigExpand expand : indexConfig.getItems()) {
        expand.setConfigId(indexConfig.getId());
        expand.setTenantCode(TenantUtils.getTenantCode());
      }
      this.indexConfigExpandRepository.saveBatch(indexConfig.getItems());
    }
    return indexConfig;
  }

  /**
   * 创建/编辑 数据校验
   *
   * @param indexConfig 登录配置实体
   */
  private void createValidate(IndexConfig indexConfig) {
    indexConfig.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(indexConfig.getDataType(), "登录配置类型不能为空");
    //唯一校验
    List<IndexConfig> existList = this.indexConfigRepository.findByDataType(indexConfig.getDataType(),indexConfig.getTenantCode());
    Map<String, IndexConfig> existMap = existList.stream()
        .filter(o -> !o.getId().equals(indexConfig.getId()))
        .collect(Collectors.toMap(IndexConfig::getDataType, Function.identity()));
    IndexConfig exist = existMap.get(indexConfig.getDataType());
    Validate.isTrue(ObjectUtils.isEmpty(exist), "该类型配置已存在");

    Validate.notBlank(indexConfig.getTemplateName(), "登录配置模板名称不能为空");
    Validate.notBlank(indexConfig.getIndexTitle(), "登录配置标题不能为空");
    Validate.notBlank(indexConfig.getCompanyName(), "登录配置公司名称不能为空");
    Validate.notBlank(indexConfig.getPcFirstUrl(), "登录配置首页URL地址不能为空");
    indexConfig.getItems().forEach(o -> {
      Validate.notBlank(o.getDetailUrl(), "二维码图片不能为空");
    });
  }
}
