package com.biz.crm.mdm.business.login.config.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.login.config.dto.IndexConfigDto;
import com.biz.crm.mdm.business.login.config.dto.PersonalizedThemeDto;
import com.biz.crm.mdm.business.login.config.entity.IndexConfig;
import com.biz.crm.mdm.business.login.config.entity.IndexConfigExpand;
import com.biz.crm.mdm.business.login.config.event.IndexConfigEventListener;
import com.biz.crm.mdm.business.login.config.repository.IndexConfigExpandRepository;
import com.biz.crm.mdm.business.login.config.repository.IndexConfigRepository;
import com.biz.crm.mdm.business.login.config.service.IndexConfigService;
import com.biz.crm.mdm.business.login.config.service.PersonalizedThemeService;
import com.biz.crm.mdm.business.login.config.vo.IndexConfigExpandVo;
import com.biz.crm.mdm.business.login.config.vo.IndexConfigVo;
import com.biz.crm.mdm.business.login.config.vo.PersonalizedThemeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author jerry7
 * 登录配置service实现
 */
@Service
public class IndexConfigServiceImpl implements IndexConfigService {

  @Autowired(required = false)
  private IndexConfigRepository indexConfigRepository;

  @Autowired(required = false)
  private IndexConfigExpandRepository indexConfigExpandRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  @Lazy
  private List<IndexConfigEventListener> eventListeners;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private PersonalizedThemeService personalizedThemeService;

  @Override
  public Page<IndexConfig> findByConditions(Pageable pageable, IndexConfigDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new IndexConfigDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<IndexConfig> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<IndexConfig> byConditions = indexConfigRepository.findByConditions(page, dto);
    List<IndexConfig> records = byConditions.getRecords();
    if (CollectionUtils.isEmpty(records)) {
      return byConditions;
    }
    List<String> ids = records.stream().map(IndexConfig::getId).collect(Collectors.toList());
    List<IndexConfigExpand> detailByIds = indexConfigExpandRepository.findDetailByIds(ids, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(detailByIds)) {
      return byConditions;
    }
    Map<String, List<IndexConfigExpand>> expandMap = detailByIds.stream().collect(Collectors.groupingBy(IndexConfigExpand::getConfigId));
    for (IndexConfig record : records) {
      String id = record.getId();
      List<IndexConfigExpand> indexConfigExpands = expandMap.get(id);
      record.setItems(indexConfigExpands);
    }
    //根据当前选择的主题配置查询详情
    for (IndexConfig record : records) {
      if(StringUtils.isNotEmpty(record.getCurrentThemeCode())){
        List<PersonalizedThemeVo> list = new ArrayList<>();
        PersonalizedThemeVo themeVo = this.personalizedThemeService.findDetailByCode(record.getCurrentThemeCode());
        Validate.isTrue(ObjectUtils.isNotEmpty(themeVo), "当前主题配色不存在!");
        list.add(themeVo);
        record.setThemeVoList(list);
      }
    }
    return byConditions;
  }

  @Override
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    indexConfigRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
  }

  @Override
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    indexConfigRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
  }

  @Override
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    List<IndexConfig> indexConfigs =  indexConfigRepository.findListByIds(ids);
    indexConfigRepository.updateDelStatusByIdIn(DelFlagStatusEnum.DELETE, ids);
    if (CollectionUtils.isEmpty(eventListeners)) {
      return;
    }
    //类型转换
    List<IndexConfigVo> oldObj = (List<IndexConfigVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            indexConfigs,
            IndexConfig.class,
            IndexConfigVo.class,
            HashSet.class, ArrayList.class);
    eventListeners.forEach(indexConfigEventListener -> {
      //编写更新日志
      indexConfigEventListener.onDelete(oldObj);
    });
  }

  @Override
  public IndexConfig findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    //重构查询方法
    return indexConfigRepository.findByIdAndTenantCode(id, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public IndexConfig create(IndexConfig indexConfig) {
    Validate.notNull(indexConfig, "创建登录配置实体不能为空！");
    Validate.isTrue(StringUtils.isBlank(indexConfig.getId()), "创建登录配置ID不能有值！");
    this.createValidate(indexConfig);
    indexConfig.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    indexConfig.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    indexConfig.setCreateTime(new Date());
    indexConfig.setCreateName(loginUserService.getLoginAccountName());
    //新增租户编号
    indexConfig.setTenantCode(TenantUtils.getTenantCode());
    this.indexConfigRepository.save(indexConfig);
    //如果扩展配置不为空，添加配置扩展
    if (!CollectionUtils.isEmpty(indexConfig.getItems())) {
      for (IndexConfigExpand expand : indexConfig.getItems()) {
        expand.setConfigId(indexConfig.getId());
        expand.setTenantCode(TenantUtils.getTenantCode());
      }
      this.indexConfigExpandRepository.saveBatch(indexConfig.getItems());
    }
    if (CollectionUtils.isEmpty(eventListeners)) {
      return indexConfig;
    }
    //类型转换
    IndexConfigVo newObj =
            this.nebulaToolkitService.copyObjectByWhiteList(
                    indexConfig, IndexConfigVo.class, HashSet.class, ArrayList.class);
    newObj.setItems((List<IndexConfigExpandVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            indexConfig.getItems(),
            IndexConfigExpand.class,
            IndexConfigExpandVo.class,
            HashSet.class, ArrayList.class));
    eventListeners.forEach(indexConfigEventListener -> {
      //编写更新日志
      indexConfigEventListener.onCreate(newObj);
    });
    return indexConfig;
  }

  @Override
  @Transactional
  public IndexConfig update(IndexConfig indexConfig) {
    Validate.notNull(indexConfig, "更新登录配置实体不能为空！");
    Validate.notBlank(indexConfig.getId(), "登录登录配置ID不能为空！");
    this.createValidate(indexConfig);
    //新增租户编号
    indexConfig.setTenantCode(TenantUtils.getTenantCode());
    //查询出旧数据
    IndexConfig entity = this.indexConfigRepository.getById(indexConfig.getId());
    //类型转换
    IndexConfigVo oldObj =
            this.nebulaToolkitService.copyObjectByWhiteList(
                    entity, IndexConfigVo.class, HashSet.class, ArrayList.class);
    List<IndexConfigExpand> indexConfigExpands =
            this.indexConfigExpandRepository.findDetailByIds(Collections.singletonList(indexConfig.getId()), indexConfig.getTenantCode());
    List<IndexConfigExpandVo> indexConfigExpandVos = (List<IndexConfigExpandVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            indexConfigExpands,
            IndexConfigExpand.class,
            IndexConfigExpandVo.class,
            HashSet.class, ArrayList.class);
    oldObj.setItems(indexConfigExpandVos);
    this.indexConfigRepository.saveOrUpdate(indexConfig);
    //删除之前的配置扩展
    this.indexConfigExpandRepository.deleteExpandByConfId(indexConfig.getId());
    //如果扩展配置不为空，添加配置扩展
    if (!CollectionUtils.isEmpty(indexConfig.getItems())) {
      for (IndexConfigExpand expand : indexConfig.getItems()) {
        expand.setConfigId(indexConfig.getId());
        expand.setTenantCode(TenantUtils.getTenantCode());
      }
      this.indexConfigExpandRepository.saveBatch(indexConfig.getItems());
    }
    if (CollectionUtils.isEmpty(eventListeners)) {
      return indexConfig;
    }
    //类型转换
    IndexConfigVo newObj =
            this.nebulaToolkitService.copyObjectByWhiteList(
                    indexConfig, IndexConfigVo.class, HashSet.class, ArrayList.class);
    newObj.setItems((List<IndexConfigExpandVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            indexConfig.getItems(),
            IndexConfigExpand.class,
            IndexConfigExpandVo.class,
            HashSet.class, ArrayList.class));
    eventListeners.forEach(indexConfigEventListener -> {
      //编写更新日志
      indexConfigEventListener.onUpdate(oldObj,newObj);
    });
    return indexConfig;
  }

  /**
   * 根据租户获取当前配置项
   *
   * @return
   */
  @Override
  public IndexConfigVo findThemeByTenantCode(IndexConfigDto dto) {
    //获取到当前租户
    String tenantCode = TenantUtils.getTenantCode();
    //获取到当前的登陆配置信息
    Pageable pageable = PageRequest.of(0, 200);
    List<IndexConfig> list = this.findByConditions(pageable, dto).getRecords();
    IndexConfig entity = new IndexConfig();
    //当前租户为找到登陆页配置时 默认空数据，但配色需携带
    if(!CollectionUtils.isEmpty(list)){
      entity = list.get(0);
      entity.setUpdateFlag(true);
    }
    //清空主题配置
    entity.setThemeVoList(null);
    //根据id获取登录页具体信息 及颜色
    List<PersonalizedThemeVo> themeVos = personalizedThemeService.findThemeByTenantCode(new PersonalizedThemeDto());
    IndexConfigVo result = this.nebulaToolkitService.copyObjectByBlankList(entity, IndexConfigVo.class, HashSet.class, ArrayList.class);
    result.setThemeVoList(themeVos);
    return result;
  }

  /**
   * 查询当前主题配置是否在使用
   *
   * @param code
   * @return
   */
  @Override
  public int findIndexByThemeCode(String code) {
    int count = indexConfigRepository.lambdaQuery()
            .eq(IndexConfig::getTenantCode, TenantUtils.getTenantCode())
            .eq(IndexConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(IndexConfig::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
            .eq(IndexConfig::getCurrentThemeCode, code)
            .count();
    return ObjectUtils.isNotEmpty(count) ? count : 0;
  }

  /**
   * 创建/编辑 数据校验
   *
   * @param indexConfig 登录配置实体
   */
  private void createValidate(IndexConfig indexConfig) {
    indexConfig.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(indexConfig.getDataType(), "登录配置类型不能为空");
    //唯一校验
    List<IndexConfig> existList = this.indexConfigRepository.findByDataType(indexConfig.getDataType(), indexConfig.getTenantCode());
    Map<String, IndexConfig> existMap = existList.stream()
        .filter(o -> !o.getId().equals(indexConfig.getId()))
        .collect(Collectors.toMap(IndexConfig::getDataType, Function.identity()));
    IndexConfig exist = existMap.get(indexConfig.getDataType());
    Validate.isTrue(ObjectUtils.isEmpty(exist), "该类型配置已存在");

//    Validate.notBlank(indexConfig.getTemplateName(), "登录配置模板名称不能为空");
    Validate.notBlank(indexConfig.getIndexTitle(), "登录配置标题不能为空");
    Validate.notBlank(indexConfig.getCompanyName(), "登录配置公司名称不能为空");
    indexConfig.getItems().forEach(o -> {
      Validate.notBlank(o.getFileCode(), "轮播图片不能为空");
    });
  }
}
