package com.biz.crm.mdm.business.login.config.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.login.config.constant.PersonalizedThemeConstant;
import com.biz.crm.mdm.business.login.config.dto.PersonalizedThemeDto;
import com.biz.crm.mdm.business.login.config.entity.PersonalizedTheme;
import com.biz.crm.mdm.business.login.config.repository.PersonalizedThemeRepository;
import com.biz.crm.mdm.business.login.config.service.IndexConfigService;
import com.biz.crm.mdm.business.login.config.service.PersonalizedThemeService;
import com.biz.crm.mdm.business.login.config.vo.PersonalizedThemeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

/**
 * @author hejun
 * @date 2023/2/13 14:54
 * @description personalized_theme 个性化主题PersonalizedTheme表服务实现类
 */
@Service("personalizedThemeService")
public class PersonalizedThemeServiceImpl implements PersonalizedThemeService {

  @Autowired(required = false)
  private PersonalizedThemeRepository personalizedThemeRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private IndexConfigService indexConfigService;

  /**
   * 根据租户获取主体配置信息
   *
   * @param dto
   * @return
   */
  @Override
  public List<PersonalizedThemeVo> findThemeByTenantCode(PersonalizedThemeDto dto) {
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setTenantCode(TenantUtils.getTenantCode());

    // 查询默认主题配色
    List<PersonalizedTheme> defaultList = this.personalizedThemeRepository.findDefaultTheme(dto);
    //查询当前用户的主题配色
    List<PersonalizedTheme> userList = this.personalizedThemeRepository.findByTenantCode(dto);
    if(!ObjectUtils.isEmpty(userList)){
      defaultList.addAll(userList);
    }
    return (List<PersonalizedThemeVo>) this.nebulaToolkitService.copyCollectionByBlankList(defaultList, PersonalizedTheme.class, PersonalizedThemeVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public PersonalizedThemeVo create(PersonalizedThemeDto dto) {
    this.validation(dto);
    String userName = this.loginUserService.getLoginAccountName();
    String tenantCode = TenantUtils.getTenantCode();
    PersonalizedTheme entity = this.nebulaToolkitService.copyObjectByBlankList(dto, PersonalizedTheme.class, HashSet.class, ArrayList.class);
    if (StringUtils.isEmpty(entity.getThemeCode())){
      String themeCode =  this.generateCodeService.generateCode(PersonalizedThemeConstant.THEME_CODE,1).get(0);
      entity.setThemeCode(themeCode);
    }else{
      Integer count = this.personalizedThemeRepository.CountByThemeCode(entity.getThemeCode());
      Validate.isTrue(null == count || 1 > count, entity.getThemeCode() + "系统主题编码已存在");
    }
    //自定义的主题配置
    entity.setDefaultTheme(false);
    entity.setCreateName(userName);
    entity.setCreateTime(new Date());
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setTenantCode(tenantCode);
    personalizedThemeRepository.saveOrUpdate(entity);
    return this.nebulaToolkitService.copyObjectByBlankList(entity, PersonalizedThemeVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public PersonalizedThemeVo update(PersonalizedThemeDto dto) {
    this.validation(dto);
    String userName = this.loginUserService.getLoginAccountName();
    String tenantCode = TenantUtils.getTenantCode();
    PersonalizedTheme entity = this.nebulaToolkitService.copyObjectByBlankList(dto, PersonalizedTheme.class, HashSet.class, ArrayList.class);
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setModifyName(userName);
    entity.setModifyTime(new Date());
    entity.setDefaultTheme(false);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setTenantCode(tenantCode);
    PersonalizedTheme theme = this.personalizedThemeRepository.findById(entity.getId(), entity.getTenantCode());
    Validate.isTrue(!ObjectUtils.isEmpty(theme), "当前系统主题不存在");
    Validate.isTrue(!theme.getDefaultTheme(), "系统默认主题禁止修改!");
    Validate.isTrue(theme.getThemeCode().equals(entity.getThemeCode()),"系统主题编码不能修改");
    this.personalizedThemeRepository.saveOrUpdate(entity);
    return this.nebulaToolkitService.copyObjectByBlankList(entity, PersonalizedThemeVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public PersonalizedThemeVo findDetailById(String id) {
    PersonalizedTheme entity = this.personalizedThemeRepository.findById(id, TenantUtils.getTenantCode());
    return this.nebulaToolkitService.copyObjectByBlankList(entity, PersonalizedThemeVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 根据code 查询详情
   *
   * @param code
   * @return
   */
  @Override
  public PersonalizedThemeVo findDetailByCode(String code) {
    PersonalizedTheme entity = this.personalizedThemeRepository.findByCode(code, TenantUtils.getTenantCode());
    return this.nebulaToolkitService.copyObjectByBlankList(entity, PersonalizedThemeVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<PersonalizedTheme> themes = this.personalizedThemeRepository.findThemesByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(themes), "请选择删除的主题配置!");
    String tenantCode = TenantUtils.getTenantCode();
    //物理删除
    for(PersonalizedTheme item : themes){
      Validate.isTrue(!item.getDefaultTheme(), "系统默认主题配置禁止删除!");
      Validate.isTrue(tenantCode.equals(item.getTenantCode()), "禁止删除非当前租户的主题配置!");
      //排查是否该主题在使用了
      int count = indexConfigService.findIndexByThemeCode(item.getThemeCode());
      Validate.isTrue(count <= 0, "当前配色正在被使用, 请勿删除");
      this.personalizedThemeRepository.removeById(item.getId());
    }
  }

  private void validation(PersonalizedThemeDto dto) {
    Validate.notNull(dto, "主题信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(dto.getThemeName()), "系统主题名称不能为空");
  }
}
