package com.biz.crm.mdm.business.login.log.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.login.log.local.entity.LoginLog;
import com.biz.crm.mdm.business.login.log.local.repository.LoginLogRepository;
import com.biz.crm.mdm.business.login.log.sdk.dto.LoginLogDto;
import com.biz.crm.mdm.business.login.log.sdk.service.LoginLogVoService;
import com.biz.crm.mdm.business.login.log.sdk.vo.LoginLogVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Optional;

/**
 * 登录日志服务实现
 * @author pengxi
 **/
@Service
public class LoginLogVoServiceImpl implements LoginLogVoService {

  @Autowired(required = false)
  private LoginLogRepository loginLogRepository;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
   */
  private static volatile Cache<String, LoginLogVo> cache = null;

  public LoginLogVoServiceImpl(){
    if(cache == null) {
      synchronized (LoginLogVoServiceImpl.class) {
        while(cache == null) {
          cache = CacheBuilder.newBuilder()
              .initialCapacity(10000)
              .expireAfterWrite(500, TimeUnit.MILLISECONDS)
              .maximumSize(100000)
              .build();
        }
      }
    }
  }

  @Override
  public Page<LoginLogVo> findByConditions(Pageable pageable, LoginLogDto loginLogDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
    loginLogDto = Optional.ofNullable(loginLogDto).orElse(new LoginLogDto());
    loginLogDto.setTenantCode(TenantUtils.getTenantCode());
    return this.loginLogRepository.findByConditions(pageable, loginLogDto);
  }

  @Override
  public LoginLogVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    //重构查询方法
    LoginLog loginLog = this.loginLogRepository.findByIdAndTenantCode(id,TenantUtils.getTenantCode());
    if (loginLog == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(loginLog, LoginLogVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public LoginLogVo findByAccount(String account) {
    if (StringUtils.isBlank(account)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();

    LoginLogVo loginLogVo;
    String cacheKey = StringUtils.join(tenantCode, account);
    if ((loginLogVo = cache.getIfPresent(cacheKey)) == null) {
      synchronized (this){
        if ((loginLogVo = cache.getIfPresent(cacheKey)) == null) {
          LoginLog loginLog = this.loginLogRepository.findByAccount(account,tenantCode);
          if (loginLog == null) {
            return null;
          }
          loginLogVo = this.nebulaToolkitService.copyObjectByWhiteList(loginLog, LoginLogVo.class, HashSet.class, ArrayList.class);
          cache.put(cacheKey, loginLogVo);
        }
      }
    }
    return loginLogVo;
  }

  @Override
  @Transactional
  public void create(LoginLogDto loginLogDto) {
    // TODO 前端改完登录接口后打开校验
    //Validate.notNull(loginLogDto.getAppType(), "业务系统类型不能为空！");
    Validate.notNull(loginLogDto.getLoginType(), "登录方式不能为空！");
    Validate.notBlank(loginLogDto.getUsertype(), "用户类别不能为空！");
    Validate.notBlank(loginLogDto.getAccount(), "账号不能为空！");
    Validate.notBlank(loginLogDto.getFullName(), "用户名称不能为空！");
    LoginLog loginLog = this.nebulaToolkitService.copyObjectByWhiteList(loginLogDto, LoginLog.class, HashSet.class, ArrayList.class);
    loginLog.setTenantCode(TenantUtils.getTenantCode());
    loginLog.setLoginTime(new Date());
    this.loginLogRepository.save(loginLog);
  }

}
