package com.biz.crm.mdm.business.material.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.material.sdk.dto.MaterialDto;
import com.biz.crm.mdm.business.material.sdk.dto.MaterialPageDto;
import com.biz.crm.mdm.business.material.sdk.service.MaterialVoService;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Set;

/**
 * 物料管理
 *
 * @author pengxi
 * @date 2021-09-27 13:55:17
 */
@Slf4j
@RequestMapping("/v1/material/material")
@RestController
@Api(tags = "物料管理：MaterialVo：物料管理")
public class MaterialVoController {

  @Autowired(required = false)
  private MaterialVoService materialVoService;

  /**
   * 查询分页列表
   * @param pageable
   * @param materialDto
   * @return
   */
  @ApiOperation(value = "查询分页列表", httpMethod = "GET")
  @GetMapping("/findByConditions")
  public Result<Page<MaterialVo>> findByConditions(@PageableDefault(50) Pageable pageable, MaterialDto materialDto) {
    try {
      Page<MaterialVo> result = this.materialVoService.findByConditions(pageable, materialDto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据ID查询详情
   * @param id
   * @return
   */
  @ApiOperation(value = "根据ID查询详情", httpMethod = "GET")
  @GetMapping("/findDetailById")
  public Result<MaterialVo> findDetailById(@RequestParam(value = "id") String id) {
    try {
      return Result.ok(this.materialVoService.findDetailById(id));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据物料编码查询详情
   * @param materialCode
   * @return
   */
  @ApiOperation(value = "根据物料编码查询详情", httpMethod = "GET")
  @GetMapping("/findDetailByMaterialCode")
  public Result<MaterialVo> findDetailByMaterialCode(@RequestParam(value = "materialCode") String materialCode) {
    try {
      return Result.ok(this.materialVoService.findDetailByMaterialCode(materialCode));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据物料编码集合查询详情
   * @param materialCodes
   * @return
   */
  @ApiOperation(value = "根据物料编码集合查询详情", httpMethod = "GET")
  @GetMapping("/findDetailByMaterialCodes")
  public Result<List<MaterialVo>> findDetailByMaterialCodes(@RequestParam(value = "materialCodes") Set<String> materialCodes) {
    try {
      return Result.ok(this.materialVoService.findDetailByMaterialCodes(materialCodes));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 新增
   * @param materialDto
   * @return
   */
  @ApiOperation(value = "新增", httpMethod = "POST")
  @PostMapping
  public Result<?> create(@RequestBody MaterialDto materialDto) {
    try {
      this.materialVoService.save(materialDto);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 更新
   * @param materialDto
   * @return
   */
  @ApiOperation(value = "更新", httpMethod = "PATCH")
  @PatchMapping
  public Result<?> update(@RequestBody MaterialDto materialDto) {
    try {
      this.materialVoService.update(materialDto);
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 删除
   * @param ids
   * @return
   */
  @ApiOperation(value = "删除", httpMethod = "DELETE")
  @DeleteMapping
  public Result<?> delete(@RequestParam(value = "ids") List<String> ids) {
    try {
      this.materialVoService.deleteBatch(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 启用
   * @param ids
   * @return
   */
  @ApiOperation(value = "启用", httpMethod = "PATCH")
  @PatchMapping("/enable")
  public Result<?> enable(@RequestBody List<String> ids) {
    try {
      this.materialVoService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 禁用
   * @param ids
   * @return
   */
  @ApiOperation(value = "禁用", httpMethod = "PATCH")
  @PatchMapping("/disable")
  public Result<?> disable(@RequestBody List<String> ids) {
    try {
      this.materialVoService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 获取物料数据分页信息
   *
   * @param pageable 分页信息
   * @param dto 请求参数
   * @return 物料数据分页信息
   */
  @ApiOperation(value = "获取物料数据分页信息", httpMethod = "GET")
  @GetMapping("/findByMaterialPageDto")
  public Result<Page<MaterialVo>> findByMaterialPageDto(@PageableDefault(50) Pageable pageable, MaterialPageDto dto) {
    try {
      Page<MaterialVo> result = this.materialVoService.findByMaterialPageDto(pageable, dto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据产品层级获取启用的未删除的物料编码集合
   *
   * @param productLevelCodeSet
   * @return
   */
  @ApiOperation(value = "根据产品层级获取启用的未删除的物料编码集合")
  @PostMapping("/findCodeByProductLevelCodes")
  public Result<Set<String>> findCodeByProductLevelCodes(@RequestBody Set<String> productLevelCodeSet) {
    try {
      return Result.ok(this.materialVoService.findCodeByProductLevelCodes(productLevelCodeSet));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
