package com.biz.crm.mdm.business.org.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.org.local.entity.Org;
import com.biz.crm.mdm.business.org.local.service.OrgService;
import com.biz.crm.mdm.business.org.sdk.dto.OrgPaginationDto;
import com.biz.crm.mdm.business.org.sdk.dto.OrgQueryIdsDto;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author hecheng
 * @description: 组织信息 http 接口
 * @date 2021/9/27 上午10:36
 */
@Api(tags = "组织信息")
@Slf4j
@RestController
@RequestMapping(value = {"/v1/org/org"})
public class OrgController {

  @Autowired(required = false)
  private OrgService orgService;

  @ApiOperation(value = "查询分页列表")
  @GetMapping(value = {"/findByConditions"})
  public Result<Page<Org>> findByConditions(@PageableDefault(50) Pageable pageable, OrgPaginationDto paginationDto) {
    try {
      Page<Org> result = this.orgService.findByConditions(pageable, paginationDto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "通过id/orgCode 查询详情")
  @GetMapping(value = {"/findDetailsByIdOrOrgCode"})
  public Result<Org> findDetailsByIdOrOrgCode(String id, String orgCode) {
    try {
      Org org = null;
      if (StringUtils.isNotBlank(id)) {
        org = this.orgService.findDetailsById(id);
      } else if (StringUtils.isNotBlank(orgCode)) {
        org = this.orgService.findByOrgCode(orgCode);
      }
      return Result.ok(org);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "按ids查询详情")
  @GetMapping(value = {"/findDetailsByIds"})
  public Result<List<Org>> findDetailsByIds(@RequestParam("ids") List<String> ids) {
    try {
      List<Org> orgs = this.orgService.findDetailsByIds(ids);
      return Result.ok(orgs);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据组织编码或组织名称查询
   *
   * @param
   * @return
   */
  @ApiOperation(value = "根据组织编码或组织名称查询")
  @GetMapping("/findByOrgCodeLikeOrOrgNameLike")
  public Result<List<Org>> findByOrgCodeLikeOrOrgNameLike(@RequestParam(value = "orgCodeLikeorNameLike") String orgCodeLikeorNameLike){
    try {
      return Result.ok(orgService.findByOrgCodeLikeOrOrgNameLike(orgCodeLikeorNameLike));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "逻辑删除", httpMethod = "POST")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
  })
  @DeleteMapping("/delete")
  public Result<?> delete(@RequestParam("ids") List<String> ids) {
    try {
      this.orgService.deleteByIds(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "启用", httpMethod = "PATCH")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
  })
  @PatchMapping("/enable")
  public Result<?> enable(@RequestBody List<String> ids) {
    try {
      this.orgService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "禁用", httpMethod = "PATCH")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
  })
  @PatchMapping("/disable")
  public Result<?> disable(@RequestBody List<String> ids) {
    try {
      this.orgService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 相关的创建过程，http接口。请注意该创建过程除了可以创建org中的基本信息以外，还可以对org中属于OneToMany关联的明细信息一同进行创建注意：基于（org）模型的创建操作传入的orgJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   */
  @ApiOperation(value = "相关的创建过程，http接口。请注意该创建过程除了可以创建org中的基本信息以外，还可以对org中属于OneToMany关联的明细信息一同进行创建注意：基于（org）模型的创建操作传入的orgJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PostMapping(value = "")
  public Result<Org> create(
      @RequestBody @ApiParam(name = "org", value = "相关的创建过程，http接口。请注意该创建过程除了可以创建org中的基本信息以外，还可以对org中属于OneToMany关联的明细信息一同进行创建注意：基于（org）模型的创建操作传入的orgJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") Org org) {
    try {
      Org current = this.orgService.create(org);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（org）的修改操作传入的orgJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   */
  @ApiOperation(value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（org）的修改操作传入的orgJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PatchMapping(value = "")
  public Result<Org> update(
      @RequestBody @ApiParam(name = "org", value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（org）的修改操作传入的orgJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") Org org) {
    try {
      Org current = this.orgService.update(org);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation("根据 组织id/组织编码 查询全部下级（含当前）组织列表")
  @GetMapping("/findAllChildrenByIdOrOrgCode")
  public Result<List<Org>> findAllChildrenByIdOrOrgCode(String id, String orgCode) {
    try {
      List<Org> list = null;
      if (StringUtils.isNotBlank(id)) {
        list = this.orgService.findAllChildrenById(id);
      } else if (StringUtils.isNotBlank(orgCode)) {
        list = this.orgService.findAllChildrenByOrgCode(orgCode);
      }
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }


  @ApiOperation("根据 组织id/组织编码 查询当前下级组织列表")
  @GetMapping("/findChildrenByIdOrOrgCode")
  public Result<List<Org>> findChildrenByIdOrOrgCode(String id, String orgCode) {
    try {
      List<Org> list = null;
      if (StringUtils.isNotBlank(id)) {
        list = this.orgService.findChildrenById(id);
      } else if (StringUtils.isNotBlank(orgCode)) {
        list = this.orgService.findChildrenByOrgCode(orgCode);
      }
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }


  @ApiOperation("根据 组织id或/编码查询 全部上级（含当前）组织列表")
  @GetMapping("/findAllParentByIdOrOrgCode")
  public Result<List<Org>> findAllParentByIdOrOrgCode(String id, String orgCode) {
    try {
      List<Org> list = null;
      if (StringUtils.isNotBlank(id)) {
        list = this.orgService.findAllParentById(id);
      } else if (StringUtils.isNotBlank(orgCode)) {
        list = this.orgService.findAllParentByOrgCode(orgCode);
      }
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation("根据组织id/组织orgCode 查询当前组织的直接上级组织")
  @GetMapping("/findParentByIdOrOrgCode")
  public Result<Org> findParentByIdOrOrgCode(String id, String orgCode) {
    try {
      Org parent = null;
      if (StringUtils.isNotBlank(id)) {
        parent = this.orgService.findParentById(id);
      } else if (StringUtils.isNotBlank(orgCode)) {
        parent = this.orgService.findParentByOrgCode(orgCode);
      }
      return Result.ok(parent);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation("查询组织及下级组织，只查询指定条数")
  @GetMapping("/findAllChildrenByOrgCodes")
  public Result<List<Org>> findAllChildrenByOrgCodes(@PageableDefault(50) Pageable pageable, @RequestParam("orgCodes") List<String> orgCodes) {
    try {
      List<Org> list = this.orgService.findAllChildrenByOrgCodes(pageable, orgCodes);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation("查询所有组织（不分页，数据量大，给后台用）(启用状态、组织类型、上级编码精确查询，组织编码、名称模糊查询)")
  @GetMapping("/findListByConditions")
  public Result<List<Org>> findListByConditions(OrgPaginationDto dao) {
    try {
      List<Org> list = this.orgService.findByConditions(dao);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation("查询所有组织")
  @GetMapping("/findAll")
  public Result<List<Org>> findAll() {
    try {
      OrgPaginationDto dao = new OrgPaginationDto();
      dao.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      List<Org> list = this.orgService.findByConditions(dao);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation("重置降维编码")
  @PatchMapping("/updateRuleCode")
  public Result<?> updateRuleCode() {
    try {
      this.orgService.updateRuleCode();
      return Result.ok("重置降维编码成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation("根据组织编码集合查询这些组织（不含自己）的上级组织编码，如果参数本身有上下级关系，则会返回处于上级的组织编码")
  @GetMapping("/findAllParentOrgCodeExcludeSelf")
  public Result<List<String>> findAllParentOrgCodeExcludeSelf(@RequestParam("orgCodes") List<String> orgCodes) {
    try {
      List<Org> list = this.orgService.findAllParentByOrgCodesExcludeSelf(orgCodes);
      List<String> parentOrgCodes = null;
      if (CollectionUtils.isNotEmpty(list)) {
        parentOrgCodes = list.stream().map(Org::getOrgCode).collect(Collectors.toList());
      }
      return Result.ok(parentOrgCodes);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
  /**
   *启用组织配送权限
   */
  @ApiOperation("启用组织配送权限")
  @PatchMapping("enableAuthority")
  public Result<?> enableAuthority(@RequestBody List<String> ids){
    try {
      this.orgService.enableAuthority(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(),e);
      return Result.error(e.getMessage());
    }
  }

  /**
   *禁用组织配送权限
   */
  @ApiOperation("禁用组织配送权限")
  @PatchMapping("disableAuthority")
  public Result<?> disableAuthority(@RequestBody List<String> ids){
    try {
      this.orgService.disableAuthority(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(),e);
      return Result.error(e.getMessage());
    }
  }


  /**
   * 根据组织id集合查询组织
   */
  @ApiOperation("根据组织id集合查询组织")
  @PostMapping("findByIds")
  public Result<List<OrgVo>> findByIds(@RequestBody OrgQueryIdsDto dto){
    try {
      List<OrgVo> list =this.orgService.findByIds(dto);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(),e);
      return Result.error(e.getMessage());
    }
  }

}
