package com.biz.crm.mdm.business.product.spu.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuRelateSku;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuTag;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuTagMapping;
import com.biz.crm.mdm.business.product.spu.local.repository.ProductSpuTagMappingRepository;
import com.biz.crm.mdm.business.product.spu.local.service.ProductSpuRelateSkuService;
import com.biz.crm.mdm.business.product.spu.local.service.ProductSpuTagMappingService;
import com.biz.crm.mdm.business.product.spu.sdk.dto.BindTagDto;
import com.biz.crm.mdm.business.product.spu.sdk.dto.ProductSpuTagMappingPaginationDto;
import com.biz.crm.mdm.business.product.spu.sdk.dto.ProductSpuTagMappingQueryDto;
import com.biz.crm.mdm.business.product.spu.sdk.dto.RebindTagDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 商品关联标签管理service实现
 *
 * @author sunx
 * @date 2022/6/14
 */
@Service
public class ProductSpuTagMappingServiceImpl implements ProductSpuTagMappingService {

  @Autowired private ProductSpuTagMappingRepository productSpuTagMappingRepository;

  @Autowired private ProductSpuRelateSkuService productSpuRelateSkuService;

  @Override
  public Page<ProductSpuTagMapping> findByConditions(
      Pageable pageable, ProductSpuTagMappingPaginationDto paginationDto) {
    paginationDto =
        Optional.ofNullable(paginationDto).orElse(new ProductSpuTagMappingPaginationDto());
    paginationDto.setTenantCode(TenantUtils.getTenantCode());
    paginationDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<ProductSpuTagMapping> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    if (StringUtils.isBlank(paginationDto.getTagCode())) {
      return new Page<>();
    }
    final Page<ProductSpuTagMapping> pageResult =
        this.productSpuTagMappingRepository.findByConditions(page, paginationDto);
    if (Objects.isNull(pageResult) || CollectionUtils.isEmpty(pageResult.getRecords())) {
      return pageResult;
    }
    final Map<String, String> map = this.findSpuRelateTagNameMap(pageResult.getRecords());
    final Map<String, Long> map1 = this.findSpuRelateSkuNum(pageResult.getRecords());
    for (ProductSpuTagMapping item : pageResult.getRecords()) {
      item.setTagName(map.getOrDefault(item.getSpuCode(), StringUtils.EMPTY));
      item.setProductQuantity(map1.getOrDefault(item.getSpuCode(), 0L));
    }
    return pageResult;
  }

  @Override
  public Page<ProductSpuTagMapping> findExcludeByConditions(
      Pageable pageable, ProductSpuTagMappingPaginationDto paginationDto) {
    paginationDto =
        Optional.ofNullable(paginationDto).orElse(new ProductSpuTagMappingPaginationDto());
    paginationDto.setTenantCode(TenantUtils.getTenantCode());
    paginationDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<ProductSpuTagMapping> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    if (StringUtils.isBlank(paginationDto.getTagCode())) {
      return new Page<>();
    }
    final Page<ProductSpuTagMapping> pageResult =
        this.productSpuTagMappingRepository.findExcludeByConditions(page, paginationDto);
    if (Objects.isNull(pageResult) || CollectionUtils.isEmpty(pageResult.getRecords())) {
      return pageResult;
    }
    final Map<String, String> map = this.findSpuRelateTagNameMap(pageResult.getRecords());
    final Map<String, Long> map1 = this.findSpuRelateSkuNum(pageResult.getRecords());
    for (ProductSpuTagMapping item : pageResult.getRecords()) {
      item.setTagName(map.getOrDefault(item.getSpuCode(), StringUtils.EMPTY));
      item.setProductQuantity(map1.getOrDefault(item.getSpuCode(), 0L));
    }
    return pageResult;
  }

  /**
   * 批量设置标签
   *
   * @param tagList
   * @param spuCode
   */
  @Override
  @Transactional
  public void saveBatch(List<ProductSpuTag> tagList, String spuCode) {
    Validate.notBlank(spuCode, "spuCode不能为空");
    this.productSpuTagMappingRepository.deleteBySupCodes(Sets.newHashSet(spuCode));
    if (CollectionUtils.isEmpty(tagList)) {
      return;
    }
    List<ProductSpuTagMapping> mappings = new ArrayList<>();
    for (ProductSpuTag tag : tagList) {
      ProductSpuTagMapping mapping = new ProductSpuTagMapping();
      mapping.setTagCode(tag.getTagCode());
      mapping.setSpuCode(spuCode);
      mapping.setTenantCode(TenantUtils.getTenantCode());
      mapping.setItemKey(
          StringUtils.joinWith(
              ":", mapping.getTenantCode(), mapping.getTagCode(), mapping.getSpuCode()));
      mappings.add(mapping);
    }
    this.deleteMapping(mappings);
    this.productSpuTagMappingRepository.saveBatch(mappings);
  }

  @Override
  @Transactional
  public void bindTag(BindTagDto dto) {
    boolean f =
        Objects.nonNull(dto)
            && StringUtils.isNotBlank(dto.getTagCode())
            && CollectionUtils.isNotEmpty(dto.getSpuCodeSet());
    Validate.isTrue(f, "参数不能为空");
    this.productSpuTagMappingRepository.deleteByTagCodeAndSupCodes(
        dto.getTagCode(), dto.getSpuCodeSet());
    List<ProductSpuTagMapping> list = Lists.newLinkedList();
    for (String item : dto.getSpuCodeSet()) {
      ProductSpuTagMapping mapping = new ProductSpuTagMapping();
      mapping.setTagCode(dto.getTagCode());
      mapping.setSpuCode(item);
      mapping.setTenantCode(TenantUtils.getTenantCode());
      mapping.setItemKey(
          StringUtils.joinWith(
              ":", mapping.getTenantCode(), mapping.getTagCode(), mapping.getSpuCode()));
      list.add(mapping);
    }
    this.deleteMapping(list);
    this.productSpuTagMappingRepository.saveBatch(list);
  }

  @Override
  @Transactional
  public void rebindTag(RebindTagDto dto) {
    boolean f =
        Objects.nonNull(dto)
            && StringUtils.isNoneBlank(dto.getOldTagCode(), dto.getTagCode())
            && CollectionUtils.isNotEmpty(dto.getSpuCodeSet());
    Validate.isTrue(f, "参数不能为空");
    this.productSpuTagMappingRepository.deleteByTagCodeAndSupCodes(
        dto.getOldTagCode(), dto.getSpuCodeSet());
    List<ProductSpuTagMapping> list = Lists.newLinkedList();
    for (String item : dto.getSpuCodeSet()) {
      ProductSpuTagMapping mapping = new ProductSpuTagMapping();
      mapping.setTagCode(dto.getTagCode());
      mapping.setSpuCode(item);
      mapping.setTenantCode(TenantUtils.getTenantCode());
      mapping.setItemKey(
          StringUtils.joinWith(
              ":", mapping.getTenantCode(), mapping.getTagCode(), mapping.getSpuCode()));
      list.add(mapping);
    }
    this.deleteMapping(list);
    this.productSpuTagMappingRepository.saveBatch(list);
  }

  @Override
  @Transactional
  public void unbindTag(BindTagDto dto) {
    boolean f =
        Objects.nonNull(dto)
            && StringUtils.isNotBlank(dto.getTagCode())
            && CollectionUtils.isNotEmpty(dto.getSpuCodeSet());
    Validate.isTrue(f, "参数不能为空");
    this.productSpuTagMappingRepository.deleteByTagCodeAndSupCodes(
        dto.getTagCode(), dto.getSpuCodeSet());
  }

  /**
   * 删除已关联的数据
   *
   * @param list
   */
  private void deleteMapping(List<ProductSpuTagMapping> list) {
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    Set<String> itemKeys =
        list.stream().map(ProductSpuTagMapping::getItemKey).collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(itemKeys)) {
      return;
    }
    this.productSpuTagMappingRepository.deleteByItemKeys(itemKeys);
  }

  /**
   * 获取spu关联标签信息
   *
   * @param list
   * @return
   */
  private Map<String, String> findSpuRelateTagNameMap(List<ProductSpuTagMapping> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    final Set<String> spuCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
            .map(ProductSpuTagMapping::getSpuCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(spuCodeSet)) {
      return Maps.newHashMap();
    }
    final ProductSpuTagMappingQueryDto queryDto = new ProductSpuTagMappingQueryDto();
    queryDto.setTenantCode(TenantUtils.getTenantCode());
    queryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    queryDto.setSpuCodes(spuCodeSet);
    final List<ProductSpuTagMapping> tagMappingList =
        this.productSpuTagMappingRepository.findByProductSpuTagMappingQueryDto(queryDto);
    if (CollectionUtils.isEmpty(tagMappingList)) {
      return Maps.newHashMap();
    }
    return tagMappingList.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getSpuCode(), a.getTagName()))
        .collect(
            Collectors.groupingBy(
                ProductSpuTagMapping::getSpuCode,
                Collectors.mapping(ProductSpuTagMapping::getTagName, Collectors.joining(","))));
  }

  /**
   * 获取spu关联sku数量map
   *
   * @param list
   * @return
   */
  private Map<String, Long> findSpuRelateSkuNum(List<ProductSpuTagMapping> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    final Set<String> spuCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
            .map(ProductSpuTagMapping::getSpuCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(spuCodeSet)) {
      return Maps.newHashMap();
    }
    List<ProductSpuRelateSku> productSpuRelateSkuList =
        this.productSpuRelateSkuService.findBySpuCodes(Lists.newArrayList(spuCodeSet));
    if (CollectionUtils.isEmpty(productSpuRelateSkuList)) {
      return Maps.newHashMap();
    }
    return productSpuRelateSkuList.stream()
        .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
        .collect(Collectors.groupingBy(ProductSpuRelateSku::getSpuCode, Collectors.counting()));
  }
}
