package com.biz.crm.mdm.business.user.feign.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.user.feign.feign.UserRelDingTalkVoFeign;
import com.biz.crm.mdm.business.user.sdk.dto.UserRelDingTalkDto;
import com.biz.crm.mdm.business.user.sdk.service.UserRelDingTalkVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 用户与钉钉关联表(UserRelDingTalk)表服务接口实现类
 *
 * @return
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.12.3 23:44
 */
@Service("FeignUserRelDingTalkVoServiceImpl")
public class UserRelDingTalkVoServiceImpl implements UserRelDingTalkVoService {

    @Autowired(required = false)
    private UserRelDingTalkVoFeign userRelDingTalkVoFeign;

    /**
     * 新增(新增前会删除账号已关联信息)
     *
     * @param dtoList 新增信息
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:47
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<UserRelDingTalkVo> createBatch(List<UserRelDingTalkDto> dtoList) {
        this.createValidation(dtoList);
        Result<List<UserRelDingTalkVo>> result = userRelDingTalkVoFeign.createBatch(dtoList);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        return result.getResult();
    }

    /**
     * 新增验证
     *
     * @param dtoList 参数集合
     */
    private void createValidation(List<UserRelDingTalkDto> dtoList) {
        Assert.notEmpty(dtoList, "信息不能为空!");
        dtoList.forEach(dto -> {
            Assert.hasLength(dto.getUserName(), "账号信息不能为空!");
            Assert.hasLength(dto.getDingTalkUserId(), "钉钉ID不能为空!");
        });
    }

    /**
     * 根据ID集合解除账号关联钉钉信息
     *
     * @param ids ID集合
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<String> ids) {
        if (CollectionUtil.isEmpty(ids)) {
            return;
        }
        Result<?> result = userRelDingTalkVoFeign.deleteByIds(ids);
        Assert.isTrue(result.isSuccess(), result.getMessage());
    }

    /**
     * 根据账号集合解除账号关联钉钉信息
     *
     * @param userNameList 账号集合
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return;
        }
        Result<?> result = userRelDingTalkVoFeign.deleteByUserNames(userNameList);
        Assert.isTrue(result.isSuccess(), result.getMessage());
    }

    /**
     * 根据钉钉ID集合解除账号关联钉钉信息
     *
     * @param dingTalkUserIds 钉钉ID集合
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByDingTalkUserIds(List<String> dingTalkUserIds) {
        if (CollectionUtil.isEmpty(dingTalkUserIds)) {
            return;
        }
        Result<?> result = userRelDingTalkVoFeign.deleteByDingTalkUserIds(dingTalkUserIds);
        Assert.isTrue(result.isSuccess(), result.getMessage());
    }

    /**
     * 根据ID获取用户关联钉钉信息
     *
     * @param id ID
     * @return com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public UserRelDingTalkVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        Result<UserRelDingTalkVo> result = userRelDingTalkVoFeign.findById(id);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        return result.getResult();
    }

    /**
     * 根据钉钉ID获取用户关联钉钉信息
     *
     * @param dingTalkUserId 钉钉ID
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public List<UserRelDingTalkVo> findByDingTalkUserId(String dingTalkUserId) {
        if (StringUtils.isBlank(dingTalkUserId)) {
            return null;
        }
        Result<List<UserRelDingTalkVo>> result = userRelDingTalkVoFeign.findByDingTalkUserId(dingTalkUserId);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        return result.getResult();
    }

    /**
     * 根据钉钉ID获取用户关联钉钉信息
     *
     * @param dingTalkUserIds 钉钉ID
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public List<UserRelDingTalkVo> findByDingTalkUserIds(List<String> dingTalkUserIds) {
        if (CollectionUtil.isEmpty(dingTalkUserIds)) {
            return null;
        }
        Result<List<UserRelDingTalkVo>> result = userRelDingTalkVoFeign.findByDingTalkUserIds(dingTalkUserIds);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        return result.getResult();
    }

    /**
     * 根据账号获取用户关联钉钉信息
     *
     * @param userName 账号
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public List<UserRelDingTalkVo> findByUserName(String userName) {
        if (StringUtils.isBlank(userName)) {
            return Collections.emptyList();
        }
        Result<List<UserRelDingTalkVo>> result = userRelDingTalkVoFeign.findByUserName(userName);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        return result.getResult();
    }

    /**
     * 根据账号集合获取用户关联钉钉信息
     *
     * @param userNameList 账号集合
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public List<UserRelDingTalkVo> findByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return Collections.emptyList();
        }
        Result<List<UserRelDingTalkVo>> result = userRelDingTalkVoFeign.findByUserNames(userNameList);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        return result.getResult();
    }

    /**
     * 根据账号集合获取用户关联钉钉信息
     * key userName
     * value 钉钉用户ID Set
     *
     * @param userNameList 账号集合
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public Map<String, Set<String>> findMapByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return Collections.emptyMap();
        }
        Result<Map<String, Set<String>>> result = userRelDingTalkVoFeign.findMapByUserNames(userNameList);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        return result.getResult();
    }


    /**
     * 根据账号获取用户关联钉钉用户ID
     *
     * @param userName 账号
     * @return java.util.Set<java.lang.String>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.5 11:27
     */
    @Override
    public Set<String> findDingTalkUserIdByUserName(String userName) {
        if (StringUtils.isEmpty(userName)) {
            return Collections.emptySet();
        }
        Result<Set<String>> result = userRelDingTalkVoFeign.findDingTalkUserIdByUserName(userName);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        return result.getResult();
    }


    /**
     * 根据账号集合获取用户关联钉钉用户ID
     *
     * @param userNameList 账号集合
     * @return java.util.Set<java.lang.String>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.5 11:27
     */
    @Override
    public Set<String> findDingTalkUserIdByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return Collections.emptySet();
        }
        Result<Set<String>> result = userRelDingTalkVoFeign.findDingTalkUserIdByUserNames(userNameList);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        return result.getResult();
    }
}
