package com.biz.crm.mn.third.system.invoice.identify.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.auth.sdk.service.UrlApiService;
import com.biz.crm.mn.common.auth.sdk.vo.UrlAddressVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.base.util.HttpCrmUtil;
import com.biz.crm.mn.common.log.sdk.dto.ForeignLogDetailDto;
import com.biz.crm.mn.common.log.sdk.service.ForeignLogVoService;
import com.biz.crm.mn.common.log.sdk.util.ForeignLogUtil;
import com.biz.crm.mn.third.system.invoice.identify.local.utils.InvoiceIdentifyutil;
import com.biz.crm.mn.third.system.invoice.identify.sdk.constants.InvoiceIdentifyConstant;
import com.biz.crm.mn.third.system.invoice.identify.sdk.dto.InvoiceIdentifyCheckDto;
import com.biz.crm.mn.third.system.invoice.identify.sdk.dto.InvoiceIdentifyCommonDto;
import com.biz.crm.mn.third.system.invoice.identify.sdk.dto.InvoiceIdentifyIdentifyDto;
import com.biz.crm.mn.third.system.invoice.identify.sdk.enums.InvoiceIdentifyResponseTypeEnum;
import com.biz.crm.mn.third.system.invoice.identify.sdk.enums.InvoiceIdentifySignTypeEnum;
import com.biz.crm.mn.third.system.invoice.identify.sdk.service.InvoiceIdentifyService;
import com.biz.crm.mn.third.system.invoice.identify.sdk.vo.InvoiceIdentifyCheckVo;
import com.biz.crm.mn.third.system.invoice.identify.sdk.vo.InvoiceIdentifyIdentifyListVo;
import com.biz.crm.mn.third.system.invoice.identify.sdk.vo.InvoiceIdentifyIdentifyVo;
import com.biz.crm.mn.third.system.invoice.identify.sdk.vo.InvoiceVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

/**
 * @author: chenlong
 * @date: 2022/12/2 17:51
 * @description: 发票识别与验真服务接口实现
 */
@Slf4j
@Service("InvoiceIdentifyService")
public class InvoiceIdentifyServiceImpl implements InvoiceIdentifyService {

    @Autowired(required = false)
    private UrlApiService urlApiService;
    @Autowired(required = false)
    private ForeignLogVoService foreignLogVoService;


    /**
     * 发票验真
     *
     * @param dto 请求参数
     * @return InvoiceIdentifyCheckVo
     */
    @Override
    public InvoiceIdentifyCheckVo checkInvoice(InvoiceIdentifyCheckDto dto) {

        this.validateCheck(dto);
        UrlAddressVo addressVo = urlApiService.getUrlAddressByAccount(InvoiceIdentifyConstant.ACCOUNT);
        Validate.notNull(addressVo, "根据账号未获取到接口配置");
//        dto.setFplx(InvoiceIdentifyConstant.INVOICE_TYPE_NO);
        InvoiceIdentifyCommonDto<InvoiceIdentifyCheckDto> commonDto = new InvoiceIdentifyCommonDto<>();
        commonDto.setData(dto);
        this.packageCommon(addressVo, commonDto, InvoiceIdentifyConstant.SERVICE_CHECK);

        InvoiceVo<String> invoiceVo = this.getInvoiceInfo(addressVo, "发票验真", commonDto);
        if (!invoiceVo.getSuccess()) {
            log.error("发票验真返回信息错误信息!返回数据[{}]", JSON.toJSONString(invoiceVo));
            Validate.isTrue(invoiceVo.getSuccess(), "OCR发票验真异常:" + invoiceVo.getMessage());
        }
        InvoiceIdentifyCheckVo checkVo = JSON.parseObject(JSON.toJSONString(invoiceVo.getResult()), InvoiceIdentifyCheckVo.class);
        return checkVo;
    }

    @Override
    public List<InvoiceIdentifyIdentifyVo> identifyInvoice(InvoiceIdentifyIdentifyDto dto) {
        this.validateIdentify(dto);
        UrlAddressVo addressVo = urlApiService.getUrlAddressByAccount(InvoiceIdentifyConstant.ACCOUNT);
        Validate.notNull(addressVo, "根据账号未获取到接口配置");
        InvoiceIdentifyCommonDto<InvoiceIdentifyIdentifyDto> commonDto = new InvoiceIdentifyCommonDto<>();
        commonDto.setData(dto);
        this.packageCommon(addressVo, commonDto, InvoiceIdentifyConstant.SERVICE_IDENTIFY);

        InvoiceVo<String> invoiceVo = this.getInvoiceInfo(addressVo, "发票识别", commonDto);
        if (!invoiceVo.getSuccess()) {
            log.error("发票识别返回信息错误信息!返回数据[{}]", JSON.toJSONString(invoiceVo));
            Validate.isTrue(invoiceVo.getSuccess(), "OCR发票识别异常：" + invoiceVo.getMessage());
        }
        InvoiceIdentifyIdentifyListVo listVo = JSONObject.parseObject(JSON.toJSONString(invoiceVo.getResult()), InvoiceIdentifyIdentifyListVo.class);
        List<InvoiceIdentifyIdentifyVo> list = listVo.getList();
        return list;
    }

    /**
     * 发票验真数据校验
     *
     * @param dto 请求参数
     */
    public void validateCheck(InvoiceIdentifyCheckDto dto) {
//        Validate.notBlank(dto.getFpdm(), "发票代码为空");
        Validate.notBlank(dto.getFphm(), "发票号码为空");
        Validate.notBlank(dto.getFplx(), "发票类型为空");
        Validate.notBlank(dto.getKprqStr(), "开票日期为空");
//        Validate.notBlank(dto.getJym(), "校验码为空");
        Validate.notBlank(dto.getHjje(), "合计金额（不含税）为空");
    }

    /**
     * 发票识别数据校验
     *
     * @param dto 请求参数
     */
    public void validateIdentify(InvoiceIdentifyIdentifyDto dto) {
//        Validate.notBlank(dto.getFpdm(), "发票代码为空");
//        Validate.notBlank(dto.getFphm(), "发票号码为空");
//        Validate.notBlank(dto.getKprqStr(), "开票日期为空");
//        Validate.notBlank(dto.getJym(), "校验码为空");
//        Validate.notBlank(dto.getHjje(), "合计金额（不含税）为空");
    }

    /**
     * 公共参数组装
     *
     * @param addressVo  密钥配置信息vo
     * @param commonDto  参数dto
     * @param serviceStr 接口名称
     */
    public void packageCommon(UrlAddressVo addressVo, InvoiceIdentifyCommonDto<?> commonDto, String serviceStr) {
        commonDto.setAccount(addressVo.getAccessId());
        commonDto.setCompid(addressVo.getAccessKey());
        commonDto.setService(serviceStr);
        commonDto.setSignType(InvoiceIdentifySignTypeEnum.MD5.getCode());
        commonDto.setResponseType(InvoiceIdentifyResponseTypeEnum.JSON.getCode());
        commonDto.setTimestamp(DateUtil.dateStrNowAll());
        String builder = commonDto.getAccount() +
                commonDto.getTimestamp() +
                addressVo.getSecretKey();
        commonDto.setSign(InvoiceIdentifyutil.getMD5(builder));
    }


    /**
     * 去发票识别验证系统查询数据
     *
     * @param urlAddressVo 密钥配置信息
     * @param commonDto    接口参数
     * @return InvoiceVo<String
     */
    private InvoiceVo<String> getInvoiceInfo(UrlAddressVo urlAddressVo, String methodMsg, InvoiceIdentifyCommonDto<?> commonDto) {
        String json = JSON.toJSONString(commonDto);
        String url = urlAddressVo.getUrl() + InvoiceIdentifyConstant.URL;
        ForeignLogDetailDto detailDto = ForeignLogUtil.buildLogSaveInfo(json, urlAddressVo);
        detailDto.setMethod(commonDto.getService());
        detailDto.setMethodMsg(methodMsg);
        foreignLogVoService.addOrUpdateLog(detailDto, true);
        Result<String> result = HttpCrmUtil.post(url, json);
        ForeignLogUtil.buildLogResult(detailDto, result);
        foreignLogVoService.addOrUpdateLog(detailDto, false);
        InvoiceVo<String> invoiceVo = new InvoiceVo<>();
        if (!result.isSuccess()) {
            log.error("请求发票识别验真失败!错误信息:[{}]", result.getMessage());
            invoiceVo.setSuccess(false);
            return invoiceVo;
        }
        if (result.getResult() == null) {
            log.error("发票识别验真无返回信息!请求返回信息[{}]", JSON.toJSONString(result));
            invoiceVo.setSuccess(false);
            return invoiceVo;
        }
        return JSON.parseObject(result.getResult(), InvoiceVo.class);
    }
}
