package com.biz.crm.sfa.business.sign.in.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.sfa.business.sign.in.local.entity.SignInGroupEntity;
import com.biz.crm.sfa.business.sign.in.local.repository.SignInGroupRepository;
import com.biz.crm.sfa.business.sign.in.local.service.SignInGroupDateService;
import com.biz.crm.sfa.business.sign.in.local.service.SignInGroupPlaceService;
import com.biz.crm.sfa.business.sign.in.local.service.SignInGroupScopeService;
import com.biz.crm.sfa.business.sign.in.local.service.SignInGroupService;
import com.biz.crm.sfa.business.sign.in.local.service.SignInGroupTypeService;
import com.biz.crm.sfa.business.sign.in.sdk.dto.SignInGroupDto;
import com.biz.crm.sfa.business.sign.in.sdk.dto.SignInGroupPageDto;
import com.biz.crm.sfa.business.sign.in.sdk.enums.SignInElectronFenceType;
import com.biz.crm.sfa.business.sign.in.sdk.event.SignInGroupEventListener;
import com.biz.crm.sfa.business.sign.in.sdk.vo.SignInGroupVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

/**
 * 签到组表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-09 11:42:29
 */
@Slf4j
@Service("signInGroupService")
public class SignInGroupServiceImpl implements SignInGroupService {

  @Autowired
  private SignInGroupRepository signInGroupRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private SignInGroupTypeService signInGroupTypeService;
  @Autowired
  private SignInGroupDateService signInGroupDateService;
  @Autowired
  private SignInGroupPlaceService signInGroupPlaceService;
  @Autowired
  private SignInGroupScopeService signInGroupScopeService;
  @Autowired
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private List<SignInGroupEventListener> listeners;

  @Override
  @Transactional
  public SignInGroupEntity create(SignInGroupDto dto) {
    this.createValidation(dto);
    SignInGroupEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, SignInGroupEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setGroupCode(this.generateCodeService.generateCode("SIG", 1).get(0));
    this.signInGroupRepository.save(entity);
    //绑定关联数据
    dto.setId(entity.getId());
    this.bindRelationData(dto);
    //新增签到组事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      SignInGroupVo vo = this.nebulaToolkitService.copyObjectByBlankList(entity, SignInGroupVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onCreate(vo);
      });
    }
    return entity;
  }

  @Override
  @Transactional
  public SignInGroupEntity update(SignInGroupDto dto) {
    this.updateValidation(dto);
    List<SignInGroupEntity> entities = this.signInGroupRepository.findByIds(Lists.newArrayList(dto.getId()));
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "签到组信息不存在");
    SignInGroupEntity updateEntity = this.nebulaToolkitService.copyObjectByBlankList(dto, SignInGroupEntity.class, HashSet.class, ArrayList.class);
    this.signInGroupRepository.updateById(updateEntity);
    //绑定关联数据
    this.bindRelationData(dto);
    //更新签到组事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      SignInGroupVo oldVo = this.nebulaToolkitService.copyObjectByBlankList(entities.get(0), SignInGroupVo.class, HashSet.class, ArrayList.class);
      SignInGroupVo newVo = this.nebulaToolkitService.copyObjectByBlankList(updateEntity, SignInGroupVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onUpdate(oldVo, newVo);
      });
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<SignInGroupEntity> entities = this.signInGroupRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.signInGroupRepository.updateDelFlagByIds(ids);
    //删除签到组事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<SignInGroupVo> voList = (List<SignInGroupVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SignInGroupEntity.class
          , SignInGroupVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onDelete(voList);
      });
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<SignInGroupEntity> entities = this.signInGroupRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.signInGroupRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //启用签到组事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<SignInGroupVo> voList = (List<SignInGroupVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SignInGroupEntity.class
          , SignInGroupVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onEnable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<SignInGroupEntity> entities = this.signInGroupRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.signInGroupRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //启用签到组事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<SignInGroupVo> voList = (List<SignInGroupVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SignInGroupEntity.class
          , SignInGroupVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onDisable(voList);
      });
    }
  }

  @Override
  public Page<SignInGroupEntity> findByConditions(Pageable pageable, SignInGroupPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new SignInGroupPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.signInGroupRepository.findByConditions(pageable, dto);
  }

  /**
   * 绑定关联数据
   *
   * @param dto 签到组信息dto
   */
  private void bindRelationData(SignInGroupDto dto) {
    this.signInGroupTypeService.update(dto);
    this.signInGroupDateService.update(dto);
    this.signInGroupPlaceService.update(dto);
    this.signInGroupScopeService.update(dto);
  }

  /**
   * 在创建signInGroup模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(SignInGroupDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getGroupName(), "缺失签到组名称");
    Validate.notBlank(dto.getTimeType(), "缺失有效时间类型");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getScopeList()), "缺失适用范围");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getTypeList()), "缺失签到类型");
    if (!CollectionUtils.isEmpty(dto.getPlaceList())) {
      Validate.notBlank(dto.getElectronFenceType(), "缺失打卡范围类型");
    } else {
      dto.setElectronFenceType(SignInElectronFenceType.NONE.getDictCode());
    }
  }

  /**
   * 在修改signInGroup模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(SignInGroupDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getGroupName(), "缺失签到组名称");
    Validate.notBlank(dto.getTimeType(), "缺失有效时间类型");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getScopeList()), "缺失适用范围");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getTypeList()), "缺失签到类型");
    if (!CollectionUtils.isEmpty(dto.getPlaceList())) {
      Validate.notBlank(dto.getElectronFenceType(), "缺失打卡范围类型");
    }  else {
      dto.setElectronFenceType(SignInElectronFenceType.NONE.getDictCode());
    }
  }
}
