package com.biz.crm.sfa.business.sign.in.local.service.internal;

import com.biz.crm.sfa.business.sign.in.local.entity.SignInGroupDateEntity;
import com.biz.crm.sfa.business.sign.in.local.repository.SignInGroupDateRepository;
import com.biz.crm.sfa.business.sign.in.local.service.SignInGroupDateService;
import com.biz.crm.sfa.business.sign.in.sdk.dto.SignInGroupDateDto;
import com.biz.crm.sfa.business.sign.in.sdk.dto.SignInGroupDto;
import com.biz.crm.sfa.business.sign.in.sdk.enums.SignInTimeType;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 签到组日期表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-09 11:42:44
 */
@Slf4j
@Service("signInGroupDateService")
public class SignInGroupDateServiceImpl implements SignInGroupDateService {

  @Autowired
  private SignInGroupDateRepository signInGroupDateRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public void update(SignInGroupDto dto) {
    Validate.notBlank(dto.getId(), "签到组ID不能为空");
    this.signInGroupDateRepository.deleteByGroupId(dto.getId());
    this.updateValidation(dto);
    List<SignInGroupDateEntity> entities = dto.getDateList().stream().map(dateDto -> {
      SignInGroupDateEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dateDto, SignInGroupDateEntity.class, HashSet.class, ArrayList.class);
      entity.setGroupId(dto.getId());
      return entity;
    }).collect(Collectors.toList());
    this.signInGroupDateRepository.saveBatch(entities);
  }

  /**
   * 在修改signInGroupDate模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(SignInGroupDto dto) {
    if (SignInTimeType.DIY_TIME.getDictCode().equals(dto.getTimeType())) {
      Validate.isTrue(!CollectionUtils.isEmpty(dto.getDateList()), "缺失打卡日期");
    } else {
      List<SignInGroupDateDto> dateList = Lists.newArrayList();
      SignInGroupDateDto dateDto = new SignInGroupDateDto();
      dateDto.setStartTime(LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
      dateList.add(dateDto);
      dto.setDateList(dateList);
    }
    dto.getDateList().forEach(dateDto -> {
      int timeNo = dto.getDateList().indexOf(dateDto) + 1;
      dateDto.setTimeNo(String.valueOf(timeNo));
      Validate.notBlank(dateDto.getStartTime(), String.format("第%s段打卡日期,缺失开始日期", timeNo));
      if (SignInTimeType.DIY_TIME.getDictCode().equals(dto.getTimeType())) {
        Validate.notBlank(dateDto.getEndTime(), String.format("第%s段打卡日期,缺失结束日期", timeNo));
        Validate.isTrue(dateDto.getStartTime().compareTo(dateDto.getEndTime()) <= 0, String.format("第%s段打卡日期,开始日期需早于结束日期", timeNo));
      }
    });
  }
}
