package com.biz.crm.sfa.business.sign.in.local.service.internal;

import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.utils.DistanceUtil;
import com.biz.crm.mdm.business.user.sdk.service.UserFeignVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.biz.crm.sfa.business.sign.in.local.entity.SignInGroupDateEntity;
import com.biz.crm.sfa.business.sign.in.local.entity.SignInGroupEntity;
import com.biz.crm.sfa.business.sign.in.local.entity.SignInGroupPlaceEntity;
import com.biz.crm.sfa.business.sign.in.local.entity.SignInGroupTypeEntity;
import com.biz.crm.sfa.business.sign.in.local.entity.SignInRecordEntity;
import com.biz.crm.sfa.business.sign.in.local.entity.SignInTypeEntity;
import com.biz.crm.sfa.business.sign.in.local.model.SignInConditionModel;
import com.biz.crm.sfa.business.sign.in.local.repository.SignInGroupRepository;
import com.biz.crm.sfa.business.sign.in.local.repository.SignInRecordRepository;
import com.biz.crm.sfa.business.sign.in.local.repository.SignInTypeRepository;
import com.biz.crm.sfa.business.sign.in.local.service.SignInRecordPictureService;
import com.biz.crm.sfa.business.sign.in.local.service.SignInRecordService;
import com.biz.crm.sfa.business.sign.in.sdk.constant.SignInConstant;
import com.biz.crm.sfa.business.sign.in.sdk.dto.SignInRecordDto;
import com.biz.crm.sfa.business.sign.in.sdk.enums.SignInElectronFenceType;
import com.biz.crm.sfa.business.sign.in.sdk.enums.SignInPlaceStatus;
import com.biz.crm.sfa.business.sign.in.sdk.enums.SignInStatus;
import com.biz.crm.sfa.business.sign.in.sdk.enums.SignInTimeType;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 签到记录表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-23 10:18:40
 */
@Slf4j
@Service("signInRecordService")
public class SignInRecordServiceImpl implements SignInRecordService {

  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private SignInRecordRepository signInRecordRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private SignInGroupRepository signInGroupRepository;
  @Autowired
  private SignInTypeRepository signInTypeRepository;
  @Autowired
  private SignInRecordPictureService signInRecordPictureService;
  @Autowired
  private UserFeignVoService userFeignVoService;

  @Override
  @Transactional
  public SignInRecordEntity create(SignInRecordDto dto) {
    this.createValidation(dto);
    this.validateSignInGroup(dto);
    SignInRecordEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, SignInRecordEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setSignInStatus(SignInStatus.OK.getDictCode());
    List<UserVo> userVoList = this.userFeignVoService.findByUserNames(Collections.singletonList(dto.getUserName()));
    if (!CollectionUtils.isEmpty(userVoList)) {
      Map<String, UserVo> userMap = userVoList.stream().collect(Collectors.toMap(UserVo::getUserName, t -> t, (a, b) -> b));
      UserVo userVo = userMap.get(dto.getUserName());
      entity.setUserRealName(userVo.getFullName());
      entity.setOrgCode(userVo.getOrgCode());
      entity.setOrgName(userVo.getOrgName());
      entity.setPostCode(userVo.getPositionCode());
      entity.setPostName(userVo.getPositionName());
    }
    this.signInRecordRepository.save(entity);
    dto.setId(entity.getId());
    this.signInRecordPictureService.update(dto);
    return entity;
  }

  /**
   * 检验签到组信息
   *
   * @param dto 参数dto
   */
  private void validateSignInGroup(SignInRecordDto dto) {
    if (StringUtils.isBlank(dto.getGroupId())) {
      return;
    }
    List<SignInGroupEntity> groupEntities = this.signInGroupRepository.findByIds(Lists.newArrayList(dto.getGroupId()));
    Validate.isTrue(!CollectionUtils.isEmpty(groupEntities), "签到组不存在!");
    SignInGroupEntity groupEntity = groupEntities.get(0);
    if (SignInTimeType.DIY_TIME.getDictCode().equals(groupEntity.getTimeType())) {
      //校验是否在签到规则时间段中
      List<SignInGroupDateEntity> dateEntities = groupEntity.getDateList().stream().filter(dateEntity
          -> dateEntity.getStartTime().compareTo(dto.getSignInDate()) <= 0 && dateEntity.getEndTime().compareTo(dto.getSignInDate()) >= 0).collect(Collectors.toList());
      Validate.isTrue(!CollectionUtils.isEmpty(dateEntities), "当前日期不能签到!");
    }
    // 签到类型
    List<SignInGroupTypeEntity> typeEntities = groupEntity.getTypeList().stream().filter(typeEntity
        -> typeEntity.getTypeId().equals(dto.getSignInTypeId())).collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(typeEntities), "签到类型与签到组不匹配!");
    SignInTypeEntity typeEntity = this.signInTypeRepository.getById(dto.getSignInTypeId());
    Validate.notNull(typeEntity, "签到类型不存在!");
    // 签到照片
    if (BooleanEnum.TRUE.getCapital().equals(typeEntity.getPhotograph())) {
      Validate.isTrue(!CollectionUtils.isEmpty(dto.getPictureList()), "签到照片不能为空!");
    }
    // 签到时间
    if (BooleanEnum.TRUE.getCapital().equals(typeEntity.getAppointTime())) {
      Validate.isTrue(typeEntity.getBeginTime().compareTo(dto.getSignInTime()) <= 0 && typeEntity.getEndTime().compareTo(dto.getSignInTime()) >= 0
          , "当前时间不能签到");
    }
    // 打卡范围 签到地址 校验
    String signInPlaceStatus = this.validatePlace(dto, groupEntity);
    dto.setSignInPlaceStatus(signInPlaceStatus);
  }

  /**
   * 校验签到地点
   *
   * @param dto         参数dto
   * @param groupEntity 签到组信息
   * @return 签到地点状态
   */
  private String validatePlace(SignInRecordDto dto, SignInGroupEntity groupEntity) {
    String electronFenceType = groupEntity.getElectronFenceType();
    // 默认签到地点异常
    String signInPlaceStatus = SignInPlaceStatus.EX.getDictCode();
    // 无打卡范围信息默认为正常状态
    if (StringUtils.isBlank(electronFenceType) || SignInElectronFenceType.NONE.getDictCode().equals(electronFenceType)) {
      signInPlaceStatus = SignInPlaceStatus.OK.getDictCode();
      return signInPlaceStatus;
    }
    // 允许范围外签到，地点记录为正常
    if (SignInElectronFenceType.OUT_SIGN_OK.getDictCode().equals(electronFenceType)) {
      signInPlaceStatus = SignInPlaceStatus.OK.getDictCode();
      return signInPlaceStatus;
    }
    // 签到地点列表
    List<SignInGroupPlaceEntity> placeList = groupEntity.getPlaceList();
    Validate.isTrue(!CollectionUtils.isEmpty(placeList), "不允许签到范围外签到");
    // 判断是否范围外
    for (SignInGroupPlaceEntity placeEntity : placeList) {
      //计算两个坐标之前的距离
      double distance = DistanceUtil.calculatePointDistance(dto.getSignInLatitude().doubleValue(), dto.getSignInLongitude().doubleValue(),
          placeEntity.getPlaceLatitude().doubleValue(), placeEntity.getPlaceLongitude().doubleValue());
      if (BigDecimal.valueOf(placeEntity.getPlaceRange()).compareTo(BigDecimal.valueOf(distance)) >= 0) {
        signInPlaceStatus = SignInPlaceStatus.OK.getDictCode();
      }
      if (SignInPlaceStatus.OK.getDictCode().equals(signInPlaceStatus)) {
        break;
      }
    }
    Validate.isTrue(!(SignInPlaceStatus.EX.getDictCode().equals(signInPlaceStatus) && SignInElectronFenceType.NO_OUT_SIGN.getDictCode().equals(electronFenceType))
        , "不允许签到范围外签到");
    return signInPlaceStatus;
  }

  /**
   * 在创建signInRecord模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(SignInRecordDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    dto.setUserName(loginDetails.getAccount());
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setSignInDate(LocalDateTime.now().format(SignInConstant.YYYY_MM_DD));
    dto.setSignInTime(LocalDateTime.now().format(SignInConstant.HH_MM_SS));
    Validate.notNull(dto.getGroupId(), "缺失签到组");
    Validate.notBlank(dto.getSignInTypeId(), "缺失签到类型");
    SignInConditionModel model = new SignInConditionModel();
    model.setGroupId(dto.getGroupId());
    model.setSignInDate(dto.getSignInDate());
    model.setTenantCode(dto.getTenantCode());
    model.setSignInTypeId(dto.getSignInTypeId());
    model.setUserName(dto.getUserName());
    List<SignInRecordEntity> recordEntities = this.signInRecordRepository.findBySignInConditionModel(model);
    Validate.isTrue(CollectionUtils.isEmpty(recordEntities), "当前签到类型已经签到，请勿重复签到！");
    Validate.notNull(dto.getSignInLatitude(), "缺失定位信息");
    Validate.notNull(dto.getSignInLatitude(), "缺失定位信息");
  }
}
