package com.biz.crm.sfa.business.sign.in.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.sfa.business.sign.in.local.entity.SignInTypeEntity;
import com.biz.crm.sfa.business.sign.in.local.repository.SignInTypeRepository;
import com.biz.crm.sfa.business.sign.in.local.service.SignInTypeService;
import com.biz.crm.sfa.business.sign.in.sdk.dto.SignInTypeDto;
import com.biz.crm.sfa.business.sign.in.sdk.dto.SignInTypePageDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

/**
 * 签到类型表服务实现类
 *
 * @author ning.zhang
 * @date 2022-05-09 11:43:32
 */
@Slf4j
@Service("signInTypeService")
public class SignInTypeServiceImpl implements SignInTypeService {

  @Autowired
  private SignInTypeRepository signInTypeRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public SignInTypeEntity create(SignInTypeDto dto) {
    this.createValidation(dto);
    SignInTypeEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, SignInTypeEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.signInTypeRepository.save(entity);
    return entity;
  }

  @Override
  @Transactional
  public SignInTypeEntity update(SignInTypeDto dto) {
    this.updateValidation(dto);
    SignInTypeEntity entity = this.signInTypeRepository.getById(dto.getId());
    Validate.notNull(entity, "签到类型不存在");
    SignInTypeEntity updateEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, SignInTypeEntity.class, HashSet.class, ArrayList.class);
    this.signInTypeRepository.updateById(updateEntity);
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<SignInTypeEntity> entities = this.signInTypeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.signInTypeRepository.updateDelFlagByIds(ids);
  }

  @Override
  public Page<SignInTypeEntity> findByConditions(Pageable pageable, SignInTypePageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new SignInTypePageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.signInTypeRepository.findByConditions(pageable, dto);
  }

  @Override
  public SignInTypeEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.signInTypeRepository.getById(id);
  }

  /**
   * 在创建signInType模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(SignInTypeDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getTypeName(), "缺失类型名称");
    Validate.notBlank(dto.getAppointTime(), "缺失是否指定时间");
    Validate.notBlank(dto.getPhotograph(), "缺失是否拍照");
    if (BooleanEnum.TRUE.getCapital().equals(dto.getAppointTime())) {
      Validate.notBlank(dto.getBeginTime(), "缺失打卡开始时间");
      Validate.notBlank(dto.getEndTime(), "缺失打卡结束时间");
      Validate.isTrue(this.validateTimeFormat(dto.getBeginTime()), "非法的打卡开始时间格式,可用的格式:[HH:mm:ss]");
      Validate.isTrue(this.validateTimeFormat(dto.getEndTime()), "非法的打卡结束时间格式,可用的格式:[HH:mm:ss]");
      Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) < 0, "打卡开始时间需早于打开结束时间");
    } else {
      dto.setBeginTime(null);
      dto.setEndTime(null);
    }
  }

  /**
   * 在修改signInType模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(SignInTypeDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getTypeName(), "缺失类型名称");
    Validate.notBlank(dto.getAppointTime(), "缺失是否指定时间");
    Validate.notBlank(dto.getPhotograph(), "缺失是否拍照");
    if (BooleanEnum.TRUE.getCapital().equals(dto.getAppointTime())) {
      Validate.notBlank(dto.getEndTime(), "缺失打卡结束时间");
      Validate.notBlank(dto.getBeginTime(), "缺失打卡开始时间");
      Validate.isTrue(this.validateTimeFormat(dto.getBeginTime()), "非法的打卡开始时间格式,可用的格式:[HH:mm:ss]");
      Validate.isTrue(this.validateTimeFormat(dto.getEndTime()), "非法的打卡结束时间格式,可用的格式:[HH:mm:ss]");
      Validate.isTrue(dto.getBeginTime().compareTo(dto.getEndTime()) < 0, "打卡开始时间需早于打开结束时间");
    } else {
      dto.setBeginTime(null);
      dto.setEndTime(null);
    }
  }

  /**
   * 验证时间格式
   *
   * @param timeStr 待检查时间字符串
   * @return 验证结果
   */
  private Boolean validateTimeFormat(String timeStr) {
    Boolean result;
    try {
      LocalTime.parse(timeStr, DateTimeFormatter.ofPattern("HH:mm:ss"));
      result = Boolean.TRUE;
    } catch (Exception e) {
      result = Boolean.FALSE;
    }
    return result;
  }
}
