package com.biz.crm.common.sms.local.service.internal;

import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.common.sms.sdk.constant.SmsConstant;
import com.biz.crm.common.sms.sdk.service.ValiditySmsCodeService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 短信验证码有效期管理服务实现
 *
 * @author pengxi
 */
@Service
public class ValiditySmsCodeServiceImpl implements ValiditySmsCodeService {

    @Autowired(required = false)
    private RedisService redisService;

    @Override
    public void setVerificationCodeValidityPeriod(String templateId, String phone, String verificationCode) {
        this.setVerificationCodeValidityPeriod(templateId, phone, verificationCode, 600L);
    }

    @Override
    public void setVerificationCodeValidityPeriod(String templateId, String phone, String verificationCode, long timeout) {
        // 默认同一个电话号码1分钟内最多1条短信
        this.setVerificationCodeValidityPeriod(templateId, phone, verificationCode, timeout, 1, 1);
    }

    @Override
    public void setVerificationCodeValidityPeriod(String templateId, String phone, String verificationCode, long timeout, int minutes, int times) {
        Validate.notNull(phone, "手机号不能为空！");
        Validate.notNull(templateId, "短信类型不能为空！");
        if (timeout <= 0) {
            //单位秒，默认10分钟
            timeout = 600L;
        }
        // 检查当前手机规定时间内发送短信验证码次数
        String redisKey = StringUtils.join(SmsConstant.SMS_VERIFICATION_CODE_TIMES, templateId, ":", phone);
        Long incrTimes = this.redisService.incr(redisKey, 1);
        //单位秒，默认x分钟
        this.redisService.expire(redisKey, minutes * 60L);
        Validate.isTrue(incrTimes <= times, String.format("由于您操作频繁，请%d分钟后重新尝试", minutes));
        Validate.notNull(verificationCode, "短信验证码不能为空！");
        // 根据短信模板+手机号获取60s有效期的redis信息，用于控制频繁发送短信
        this.redisService.hSet(
                StringUtils.join(SmsConstant.SMS_VERIFICATION_CODE, phone, ":", templateId),
                StringUtils.join(phone, verificationCode),
                verificationCode,
                // 有效期10分钟
                timeout);
    }

    @Override
    public Boolean isAvailableVerificationCode(String phone, String templateId, String verificationCode) {
        Validate.notNull(phone, "手机号不能为空！");
        Validate.notNull(templateId, "短信类型不能为空！");
        Validate.notNull(verificationCode, "短信验证码不能为空！");
        // 根据短信模板+手机号获取60s有效期的redis信息，用于控制频繁发送短信
        Object obj = this.redisService.hGet(
                StringUtils.join(SmsConstant.SMS_VERIFICATION_CODE, phone, ":", templateId),
                StringUtils.join(phone, verificationCode));
        if (obj != null) {
            this.redisService.hDel(
                    StringUtils.join(SmsConstant.SMS_VERIFICATION_CODE, phone, ":", templateId),
                    StringUtils.join(phone, verificationCode));
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }
}
