package com.biz.crm.common.sms.local.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.biz.crm.common.sms.local.config.TencentSmsConfiguration;
import com.bizunited.nebula.common.service.sms.SmsServiceAdapter;
import com.bizunited.nebula.common.service.sms.SmsTypeEnums;
import com.tencentcloudapi.common.Credential;
import com.tencentcloudapi.common.profile.ClientProfile;
import com.tencentcloudapi.sms.v20190711.SmsClient;
import com.tencentcloudapi.sms.v20190711.models.SendSmsRequest;
import com.tencentcloudapi.sms.v20190711.models.SendSmsResponse;
import com.tencentcloudapi.sms.v20190711.models.SendStatus;
import java.util.Objects;
import javax.annotation.PostConstruct;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Service;

/**
 * @author ning.zhang
 * @description 腾讯云短信发送实现类
 * @date 2024/08/16
 */
@Slf4j
@Service
@ConditionalOnProperty(prefix = "sms", name = "supplier", havingValue = "tencent")
public class TencentSmsServiceImpl extends SmsServiceAdapter {

  private SmsClient smsClient;

  @Autowired
  private TencentSmsConfiguration tencentSmsConfiguration;

  @PostConstruct
  public void initClient() {
    // 实例化一个认证对象，入参需要传入腾讯云账户secretId，secretKey，见《创建secretId和secretKey》小节
    Credential cred = new Credential(
        tencentSmsConfiguration.getAccessKeyId(), tencentSmsConfiguration.getAccessKeySecret());
    // 实例化要请求产品(以cvm为例)的client对象
    ClientProfile clientProfile = new ClientProfile();
    clientProfile.setSignMethod(ClientProfile.SIGN_TC3_256);
    //第二个ap-chongqing 填产品所在的区
    this.smsClient = new SmsClient(cred, tencentSmsConfiguration.getRegion());
  }

  public void aliyunSendSms(SendSmsRequest sendSmsRequest) {
    try {
      SendSmsResponse resp = this.smsClient.SendSms(sendSmsRequest);
      Validate.isTrue(Objects.nonNull(resp) && resp.getSendStatusSet() != null
              && resp.getSendStatusSet().length > 0 && Objects.nonNull(resp.getSendStatusSet()[0])
          , "短信发送异常");
      SendStatus sendStatus = resp.getSendStatusSet()[0];
      Validate.isTrue(StringUtils.equals(sendStatus.getCode(), "Ok"), sendStatus.getMessage());
    } catch (Exception e) {
      log.error("短信发送异常：{}",e.getMessage(),e);
      Validate.isTrue(Boolean.FALSE, "短信发送异常：" + e.getMessage());
    }
  }

  @Override
  public void sendSms(String phone, String templateId, String content, SmsTypeEnums smsType) {
    // content 内容为模板的参数,内容应该是List转换为JSON字符串,如["张三","13555555555"]这样的JSON字符串
    Validate.notNull(phone, "手机号不能为空！");
    Validate.notNull(content, "短信内容不能为空！");
    Validate.notNull(smsType, "短信类型不能为空！");

    SendSmsRequest sendSmsRequest = new SendSmsRequest();
    //appId ,见《创建应用》小节
    sendSmsRequest.setSmsSdkAppid(tencentSmsConfiguration.getAppId());
    //发送短信的目标手机号，可填多个。
    String[] phones={phone};
    sendSmsRequest.setPhoneNumberSet(phones);
    //模版id,见《创建短信签名和模版》小节
    sendSmsRequest.setTemplateID(templateId);
    //模版参数，从前往后对应的是模版的{1}、{2}等,见《创建短信签名和模版》小节
    String [] templateParam= JSONArray.parseArray(content, String.class).toArray(new String[0]);
    sendSmsRequest.setTemplateParamSet(templateParam);
    //签名内容，不是填签名id,见《创建短信签名和模版》小节
    sendSmsRequest.setSign(tencentSmsConfiguration.getSignName());
    this.aliyunSendSms(sendSmsRequest);
  }
}
