package com.biz.crm.taxpay.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpResponse;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.biz.crm.taxpay.core.Config;
import com.biz.crm.taxpay.core.cipher.Signer;
import com.biz.crm.taxpay.core.cipher.Validator;
import com.biz.crm.taxpay.core.enums.ApiInfoEnum;
import com.biz.crm.taxpay.exception.ServiceException;
import com.biz.crm.taxpay.model.*;
import com.biz.crm.taxpay.service.TaxRaiseService;
import com.biz.crm.taxpay.uitil.HttpServletUtil;

import javax.servlet.http.HttpServletRequest;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 税筹平台服务接口
 */
public class TaxRaiseServiceImpl implements TaxRaiseService {

    private static final String SPLITER = "|";
    private final Signer signer;
    private final Validator validator;
    private final String basePath;

    private TaxRaiseServiceImpl(Builder builder) {
        this.signer = builder.signer;
        this.basePath = builder.basePath;
        this.validator= builder.validator;
    }
    /**
     * TaxRaiseService构造器
     */
    public static class Builder {

        private Signer signer;
        private Validator validator;
        private String basePath;

        public Builder config(Config config) {
            this.signer = config.createSigner();
            this.basePath = config.getBasePath();
            this.validator= config.createValidator();
            return this;
        }

        public TaxRaiseService build() {
            return new TaxRaiseServiceImpl(this);
        }
    }

    @Override
    public TaxBaseResponse taxCash(TaxCashRequest request) {
        //1.组装参数
        Objects.requireNonNull(request);
        String realPath=String.format(ApiInfoEnum.TAX_CASH.getUrl(),basePath);
        JSONObject data = new JSONObject();
        //提现红包唯一流水编码
        data.put("cashRecordNo",request.getCashRecordNo());
        //终端门店编码
        data.put("terminalCode",request.getTerminalCode());
        //金额(字符串)
        data.put("amount",String.valueOf(request.getAmount()));
        //1.1.组装请求头
        Map<String, String> headMap = signer.createHeadMap(data);
        //1.2组装请求体
        List<Map<String,String>> details = new ArrayList<>();
        for(TaxCashDetailRequest detailRequest:request.getDetails()){
            Map<String,String> poolMap = new HashMap<>();
            poolMap.put("amount",String.valueOf(detailRequest.getAmount()));
            poolMap.put("poolAccountCode",detailRequest.getPoolAccountCode());
            details.add(poolMap);
        }
        data.put("details",details);
        data.put("notifyUrl",request.getNotifyUrl());

        //2.发起请求
        HttpResponse execute = HttpRequest.post(realPath)
                .headerMap(headMap,true)
                .body(JSON.toJSONString(data))
                .execute();
        //3.校验结果
        validator.validate(execute.getStatus(),execute.body());
        //4.处理结果
       return JSONObject.parseObject(execute.body(), TaxBaseResponse.class);
    }

    @Override
    public TaxBaseResponse<TaxSignResponse>  verifySign(String terminalCode) {
        Objects.requireNonNull(terminalCode);
        String realPath=String.format(ApiInfoEnum.TAX_QUERY_SIGN_STATUS.getUrl(),basePath);
        JSONObject data = new JSONObject();
        data.put("terminalCode", terminalCode);
        //1.1.组装请求头
        //2.发起请求
        HttpResponse execute = HttpRequest.post(realPath)
                .headerMap(signer.createHeadMap(data), true)
                .body(JSON.toJSONString(data))
                .execute();
        //3.校验并返回结果
        validator.validate(execute.getStatus(), execute.body());
        return JSONObject.parseObject(execute.body(),   new TypeReference<TaxBaseResponse<TaxSignResponse>>(){});
    }

    @Override
    public TaxBaseResponse<TaxCashResultResponse> findCashResultByNo(String cashRecordNo) {
        Objects.requireNonNull(cashRecordNo);
        String realPath=String.format(ApiInfoEnum.TAX_FIND_CASH_INFO_BY_CODE.getUrl(),basePath);
        JSONObject data = new JSONObject();
        data.put("cashRecordNo", cashRecordNo);
        //1.1.组装请求头
        //2.发起请求
        HttpResponse execute = HttpRequest.post(realPath)
                .headerMap(signer.createHeadMap(data), true)
                .body(JSON.toJSONString(data))
                .execute();
        //3.校验并返回结果
        validator.validate(execute.getStatus(), execute.body());
        return JSONObject.parseObject(execute.body(),   new TypeReference<TaxBaseResponse<TaxCashResultResponse>>(){});
    }

    @Override
    public TaxBaseResponse cleanSign(String terminalCode) {
        Objects.requireNonNull(terminalCode);
        String realPath=String.format(ApiInfoEnum.TAX_CLEAN_SING.getUrl(),basePath);
        JSONObject data = new JSONObject();
        data.put("terminalCode", terminalCode);
        //1.1.组装请求头
        //2.发起请求
        HttpResponse execute = HttpRequest.post(realPath)
                .headerMap(signer.createHeadMap(data), true)
                .body(JSON.toJSONString(data))
                .execute();
        //3.校验并返回结果
        validator.validate(execute.getStatus(), execute.body());
        return JSONObject.parseObject(execute.body(),   TaxBaseResponse.class);
    }

    @Override
    public TaxCashResultResponse cashCallBack(HttpServletRequest request) {
        String requestBody = HttpServletUtil.getBodyString(request);
        //1.验证并转换对象
        Objects.requireNonNull(requestBody);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        TaxBaseCallBackResponse<TaxCashResultResponse> callBackResponse =
                JSONObject.parseObject(requestBody,
                        new TypeReference<TaxBaseCallBackResponse<TaxCashResultResponse>>() {});
        Objects.requireNonNull(callBackResponse);
        TaxCashResultResponse cashData = callBackResponse.getData();
        Objects.requireNonNull(cashData);
        //2. 校验签名
        JSONObject body = new JSONObject();
        body.put("cashRecordNo", cashData.getCashRecordNo());
        body.put("terminalCode", cashData.getTerminalCode());
        body.put("amount", String.valueOf(cashData.getAmount()));
        body.put("cashStatus", cashData.getCashStatus());
        if(cashData.getPayDate()!=null){
            body.put("payDate", sdf.format(cashData.getPayDate()));
        }
        body.put("createTime", sdf.format(cashData.getCreateTime()));
        body.put("payeeName", cashData.getPayeeName());
        body.put("idCard", cashData.getIdCard());
        body.put("payeePhone", cashData.getPayeePhone());
        body.put("bankCardNo", cashData.getBankCardNo());
        if(cashData.getReason()!=null&&!"".equals(cashData.getReason())){
            body.put("reason", cashData.getReason());
        }
        String sign = signer.callBackSign(body, callBackResponse.getTimestamp());
        if(sign==null||!sign.equals(callBackResponse.getSign())){
            throw new ServiceException("签名校验失败");
        }
        return cashData;
    }

    @Override
    public TaxSignCallBackResponse signCallBack(HttpServletRequest request) {
        String requestBody = HttpServletUtil.getBodyString(request);
        //1.验证并转换对象
        Objects.requireNonNull(requestBody);
        TaxBaseCallBackResponse<TaxSignCallBackResponse> callBackResponse =
                JSONObject.parseObject(requestBody,
                        new TypeReference<TaxBaseCallBackResponse<TaxSignCallBackResponse>>() {});
        Objects.requireNonNull(callBackResponse);
        TaxSignCallBackResponse signData = callBackResponse.getData();
        Objects.requireNonNull(signData);
        //2. 校验签名
        JSONObject body = new JSONObject();
        body.put("terminalCode", signData.getTerminalCode());
        body.put("signStatus", signData.getSignStatus());
        body.put("signDesc", signData.getSignDesc());
        body.put("idCard",signData.getIdCard());
        body.put("phone", signData.getPhone());
        body.put("name", signData.getName());
        body.put("bankCardNo", signData.getBankCardNo());
        String sign = signer.callBackSign(body, callBackResponse.getTimestamp());
        if(sign==null||!sign.equals(callBackResponse.getSign())){
            throw new ServiceException("签名校验失败");
        }
        return signData;
    }

    @Override
    public TaxBaseResponse<List<TaxCashStatAccountResponse>> statCashAccount(String startDate, String endDate) {
        Objects.requireNonNull(startDate);Objects.requireNonNull(endDate);
        String realPath=String.format(ApiInfoEnum.TAX_FIND_CASH_LIST.getUrl(),basePath);
        JSONObject data = new JSONObject();
        data.put("startDate", startDate);
        data.put("endDate", endDate);
        //1.1.组装请求头
        //2.发起请求
        HttpResponse execute = HttpRequest.post(realPath)
                .headerMap(signer.createHeadMap(data), true)
                .body(JSON.toJSONString(data))
                .execute();
        //3.校验并返回结果
        validator.validate(execute.getStatus(), execute.body());
        return JSONObject.parseObject(execute.body(),   new TypeReference<TaxBaseResponse<List<TaxCashStatAccountResponse>>>(){});
    }

    public static void main(String[] args) {

    }

    @Override
    public TaxBaseResponse<TaxCashAmountResponse> findCashResultByNo(String userCode, Integer flag) {
        Objects.requireNonNull(userCode);
        Objects.requireNonNull(flag);
        String realPath=String.format(ApiInfoEnum.TAX_CHECK_CASH_AMOUNT.getUrl(),basePath);
        JSONObject data = new JSONObject();
        data.put("flag", flag);
        data.put("terminalCode", userCode);
        HttpResponse execute = HttpRequest.post(realPath)
                .headerMap(signer.createHeadMap(data), true)
                .body(JSON.toJSONString(data))
                .execute();
        //3.校验并返回结果
        validator.validate(execute.getStatus(), execute.body());
        return JSONObject.parseObject(execute.body(),   new TypeReference<TaxBaseResponse<TaxCashAmountResponse>>(){});
    }

    @Override
    public TaxBaseResponse<List<TaxGovRateConfigResponse>> queryGovRateConfig() {
        String realPath=String.format(ApiInfoEnum.TAX_GOV_RATE_CONFIG.getUrl(),basePath);
        JSONObject data = new JSONObject();
        HttpResponse execute = HttpRequest.post(realPath)
                .headerMap(signer.createHeadMap(data), true)
                .body(JSON.toJSONString(data))
                .execute();
        //3.校验并返回结果
        validator.validate(execute.getStatus(), execute.body());
        return JSONObject.parseObject(execute.body(),   new TypeReference<TaxBaseResponse<List<TaxGovRateConfigResponse>>>(){});
    }

    @Override
    public TaxBaseResponse terminalSendRecord(TaxCashRecordRequest request) {
        //1.组装参数
        Objects.requireNonNull(request);
        String realPath=String.format(ApiInfoEnum.TAX_CASH_RECORD.getUrl(),basePath);
        JSONObject tempData = (JSONObject)JSONObject.toJSON(request);
        // 排序字段
        Map<String, Object> sortedMap = new LinkedHashMap<>();
        List<String> keySet = new ArrayList<>(tempData.keySet());
        keySet.sort(String::compareTo); // 按字母排序
        for (String key : keySet) {
            sortedMap.put(key, tempData.get(key));
        }
        // 3. 构造新的有序 JSONObject
        JSONObject data = new JSONObject(sortedMap);
        //1.1.组装请求头
        Map<String, String> headMap = signer.createHeadMap(data);
        //2.发起请求
        HttpResponse execute = HttpRequest.post(realPath)
                .headerMap(headMap,true)
                .body(JSON.toJSONString(data))
                .execute();
        //3.校验结果
        validator.validate(execute.getStatus(),execute.body());
        //4.处理结果
        return JSONObject.parseObject(execute.body(), TaxBaseResponse.class);
    }

    @Override
    public TaxBaseResponse syncConsumerCashRecord(List<TaxConsumerCashRecordRequest> list) {
        //1.组装参数
        Assert.notEmpty(list,"传入参数不能为空");
        String realPath=String.format(ApiInfoEnum.TAX_SYNC_CONSUMER_CASH_RECORD.getUrl(),basePath);

        JSONObject data = new JSONObject();
        for(TaxConsumerCashRecordRequest recordVo : list){
            data.put(recordVo.getCashCode(), recordVo.getPayeeAccountNo().concat(SPLITER).concat(recordVo.getAmount().toString()));
        }
        //1.1.组装请求头
        Map<String, String> headMap = signer.createHeadMap(data);
        //2.发起请求
        HttpResponse execute = HttpRequest.post(realPath)
                .headerMap(headMap,true)
                .body(JSON.toJSONString(list))
                .execute();
        //3.校验结果
        validator.validate(execute.getStatus(),execute.body());
        //4.处理结果
        return JSONObject.parseObject(execute.body(), TaxBaseResponse.class);
    }
}