package com.biz.crm.tpm.business.third.system.local.job;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.third.system.local.repository.Ce1MnLockRepository;
import com.biz.crm.tpm.business.third.system.local.service.Ce1MnLockService;
import com.biz.crm.tpm.business.third.system.local.service.PullCeSalesDataService;
import com.biz.crm.tpm.business.third.system.sdk.constants.CeConstant;
import com.biz.crm.tpm.business.third.system.sdk.dto.Ce1MnLockDto;
import com.biz.crm.tpm.business.third.system.sdk.vo.Ce1MnLockVo;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.Optional;


/**
 * 会计科目自动更新定时任务     job
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.12.15 13:39
 */
@Slf4j
@Service
public class PullCeSalesDataAutoSyncXxlJob {


    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private PullCeSalesDataService pullCeSalesDataService;

    @Autowired(required = false)
    private Ce1MnLockRepository ce1MnLockRepository;

    @Autowired(required = false)
    private Ce1MnLockService ce1MnLockService;

    @DynamicTaskService(cornExpression = "0 20 6 * * ?", taskDesc = "电商日销售报表SAP")
    public void pullSalesDataDay() {
        loginUserService.refreshAuthentication(null);
        String ds = DateUtil.format(DateUtil.dateAddDay(new Date(), -1), "yyyyMMdd");
        log.info("=====>    电商日销售报表执行定时任务ds[{}] <=====", ds);
        Ce1MnLockVo vo = new Ce1MnLockVo();
        vo.setDs(ds);
        vo.setCurrentPageNumber(0);
        pullCeSalesDataService.pullSalesDataReport(vo);
        log.info("=====>    电商日销售报表定时任务执行完毕ds[{}] <=====", ds);
    }

    @DynamicTaskService(cornExpression = "0 10 0 2-4 * ?", taskDesc = "电商日销售报表拉取整月数据")
    public void pullSalesDataMonth() {
        loginUserService.refreshAuthentication(null);
        String ds = DateUtil.format(DateUtil.getLastDayOfLastMonth(), "yyyyMMdd");
        log.info("=====>    电商日销售报表拉取整月数据，执行定时任务ds[{}]    <=====", ds);
        Ce1MnLockVo vo = new Ce1MnLockVo();
        vo.setDs(ds);
        vo.setCurrentPageNumber(0);
        pullCeSalesDataService.pullSalesDataReport(vo);
        log.info("=====>    电商日销售报表拉取整月数据，定时任务执行完毕ds[{}]    <=====", ds);
    }

    @DynamicTaskService(cornExpression = "0 20 0/1 * * ?", taskDesc = "电商日销售报表拉取数据补偿机制")
    public void pullSalesDataRetry() {
        loginUserService.refreshAuthentication(null);
        this.removeExpireTimeData();
        log.info("=====>    电商日销售报表拉取数据补偿机制，执行定时任务 start    <=====");
        Pageable pageable = PageRequest.of(1, 15);
        Page<Ce1MnLockVo> mnLockVoPage = null;
        do {
            mnLockVoPage = ce1MnLockService.findByConditions(pageable, new Ce1MnLockDto());
            pageable = pageable.next();
            if (CollectionUtil.isEmpty(mnLockVoPage.getRecords())) {
                log.info("=====>    电商日销售报表拉取数据补偿机制，无需补偿    <=====");
                return;
            }
            mnLockVoPage.getRecords().forEach(item -> {
                String ds = item.getDs();
                item.setRetryTime(Optional.ofNullable(item.getRetryTime()).orElse(0));
                log.info("=====>    电商日销售报表拉取数据补偿机制，执行定时任务ds[{}] [{}]  start  <=====", ds, JSON.toJSONString(item));
                if (CeConstant.PULL_SALES_DATA_MAX_RETRY_TIME < item.getRetryTime()) {
                    this.removeRetrySuccess(item.getId());
                    log.info("=====>    电商日销售报表拉取数据补偿机制，执行定时任务ds[{}] 已超过最大重试次数[{}]; [{}]  end  <=====",
                            ds, CeConstant.PULL_SALES_DATA_MAX_RETRY_TIME, JSON.toJSONString(item));
                    return;
                }
                try {
                    pullCeSalesDataService.pullSalesDataReport(item);
                    this.removeRetrySuccess(item.getId());
                } catch (Exception e) {
                    log.info("=====>    电商日销售报表拉取数据补偿机制，执行定时任务ds[{}] [{}]  异常  <=====", ds, JSON.toJSONString(item));
                    log.error("", e);
                }
                log.info("=====>    电商日销售报表拉取数据补偿机制，执行定时任务ds[{}] [{}]  end  <=====", ds, JSON.toJSONString(item));
            });
        } while (mnLockVoPage.hasNext() && pageable.getPageNumber() < 100);
        log.info("=====>    电商日销售报表拉取数据补偿机制，定时任务执行完毕 end    <=====");
    }

    /**
     * 删除锁定超时的单据
     */
    @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
    public void removeExpireTimeData() {
        ce1MnLockRepository.removeExpireTime();
    }

    /**
     * 根据ID删除
     */
    @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
    public void removeRetrySuccess(String id) {
        if (StringUtil.isEmpty(id)) {
            return;
        }
        ce1MnLockRepository.removeById(id);
    }


}
