package com.biz.crm.tpm.business.third.system.local.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.aliyun.openservices.shade.com.google.common.collect.Sets;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.biz.crm.mn.third.system.dataphin.sdk.service.DataPhinCe1MnjtService;
import com.biz.crm.mn.third.system.dataphin.sdk.vo.Ce1MnjtFullVo;
import com.biz.crm.tpm.business.third.system.local.service.Ce1MnLockService;
import com.biz.crm.tpm.business.third.system.local.service.Ce1MnjtService;
import com.biz.crm.tpm.business.third.system.local.service.PullCeSalesDataService;
import com.biz.crm.tpm.business.third.system.sdk.constants.CeConstant;
import com.biz.crm.tpm.business.third.system.sdk.vo.Ce1MnLockVo;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 电商日销售数据拉取
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2023-08-16 17:17
 */
@Service
@Slf4j
public class PullCeSalesDataServiceImpl implements PullCeSalesDataService {

    @Autowired
    private DataPhinCe1MnjtService dataPhinCe1MnjtService;

    @Autowired
    private Ce1MnjtService ce1MnjtService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private Ce1MnLockService ce1MnLockService;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    private static final Integer SIZE = 400;

    /**
     * 异步拉取电商日销售数据拉取
     *
     * @param ds
     * @param createDate
     * @param userIdentity
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-08-16 17:20
     */
    @Override
    @Async
    public void pullSalesDataReportAsync(String ds, String createDate, AbstractCrmUserIdentity userIdentity) {
        loginUserService.refreshAuthentication(userIdentity);
        Ce1MnLockVo vo = new Ce1MnLockVo();
        vo.setDs(ds);
        vo.setCurrentPageNumber(0);
        this.pullSalesDataReport(vo);
    }

    /**
     * 电商日销售数据拉取
     *
     * @param vo
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-08-16 17:20
     */
    @Override
    @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
    public void pullSalesDataReport(Ce1MnLockVo vo) {
        vo = Optional.ofNullable(vo).orElse(new Ce1MnLockVo());
        String ds = vo.getDs();
        Assert.hasLength(ds, "ds时间[yyyyMMdd]不能为空!");
        try {
            Date date = DateUtil.date_yyyyMMdd.parse(ds);
            Assert.isTrue(ds.equals(DateUtil.date_yyyyMMdd.format(date)), "年月格式错误【yyyyMMdd】");
        } catch (Exception e) {
            throw new IllegalArgumentException("年月格式错误【yyyyMMdd】");
        }
        String createDate = vo.getCeCreateDate();
        Integer pageNumber = Optional.ofNullable(vo.getCurrentPageNumber()).orElse(0);
        log.info("=====>    电商日销售报表ds[{}], [{}]  start  <=====", ds, JSON.toJSONString(vo));
        String lockKey = CeConstant.PULL_SALES_DATA_LOCK + ds;
        boolean hasLock = redisLockService.tryLock(lockKey, TimeUnit.HOURS, 6);
        Assert.isTrue(hasLock, "正在拉取[" + ds + "]数据，请稍后");
        try {
            //获取总数
            Integer total = dataPhinCe1MnjtService.getCe1MnjtDataTotal(ds, createDate);
            log.info("=====>    电商日销售报表ds[{}],createDate[{}] total[{}]   <=====", ds, createDate, total);
            if (ObjectUtil.isNull(total) || total.compareTo(0) == 0) {
                log.error("=====>    电商日销售报表ds[{}],createDate[{}] total[{}]   <=====", ds, createDate, total);
                return;
            }
            vo.setTotal(total);
            vo.setRetryTime(Optional.ofNullable(vo.getRetryTime()).orElse(0) + 1);
            //调用总次数
            int page = total / SIZE;
            page = total % SIZE == 0 ? page : page + 1;
            log.info("=====>    电商日销售报表ds[{}],createDate[{}] total[{}] 需调用[{}]次, 每次获取[{}]条数据    <=====",
                    ds, createDate, total, page, SIZE);
            //分批查询
            for (int i = pageNumber; i < page; i++) {
                vo.setCurrentPageNumber(i);
                List<Ce1MnjtFullVo> ce1MajDataList = dataPhinCe1MnjtService.getCe1MnjtData(ds, createDate, i, SIZE);
                ce1MajDataList.forEach(item -> item.setProductCode(takeOutFrontZero(item.getProductCode())));
                ce1MnjtService.saveEntityBatch(ce1MajDataList, ds);
                log.info("=====>    电商日销售报表第[{}/{}]页数据,数据总数[{}]条保存成功    <=====", i + 1, page, ce1MajDataList.size());
            }
            sendMqMsg(ds);
        } catch (Exception e) {
            log.error("电商日销售报表拉取时报错{}", e.getMessage());
            log.error("", e);
            try {
                ce1MnLockService.saveOrUpdateNewTransactional(vo);
            } catch (Exception lockE) {
                log.error("电商日销售报表拉取异常生成重复请求时报错{}", e.getMessage());
                log.error("", lockE);
            }
            throw e;
        } finally {
            log.info("=====>    电商日销售报表ds[{}], [{}]  end  <=====", ds, JSON.toJSONString(vo));
            redisLockService.unlock(lockKey);
        }
    }

    /**
     * 发送MQ消息
     *
     * @param ds
     */
    private void sendMqMsg(String ds) {
        if (StringUtil.isEmpty(ds)) {
            return;
        }

        MqMessageVo msgCodeList = new MqMessageVo();
        msgCodeList.setMsgBody(ds);
        msgCodeList.setTag(CeConstant.SEND_PLATFORM_SALE_TAG);
        rocketMqProducer.sendMqOrderMsg(msgCodeList, ds, 10);

        //月销售报表
        MqMessageVo mqMessageVo2 = new MqMessageVo();
        mqMessageVo2.setMsgBody(ds);
        mqMessageVo2.setTag(CeConstant.SEND_DISTRIBUTION_CUSTOMER_MONTH_SALE_TAG);
        rocketMqProducer.sendMqOrderMsg(mqMessageVo2, ds, 10);
    }

    /**
     * 去除前导零
     *
     * @param str
     * @return
     */
    private static String takeOutFrontZero(String str) {
        int arrayLength = 0;
        char[] array = str.toCharArray();
        arrayLength = array.length;
        int firstNonZeroAt = 0;
        for (int i = 0; i < array.length; i++) {
            if (!"0".equalsIgnoreCase(String.valueOf(array[i]))) {
                firstNonZeroAt = i;
                break;
            }
        }
        char[] newArray = Arrays.copyOfRange(array, firstNonZeroAt, arrayLength);
        return new String(newArray);
    }

}
