package com.biz.crm.tpm.business.third.system.local.service.impl;

import com.aliyun.openservices.shade.org.apache.commons.lang3.Validate;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.util.RocketMqUtil;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.biz.crm.mn.third.system.dataphin.sdk.service.DataPhinZmsd121Service;
import com.biz.crm.mn.third.system.dataphin.sdk.service.DataPhinZmsd187Service;
import com.biz.crm.mn.third.system.dataphin.sdk.vo.Zmsd121Vo;
import com.biz.crm.mn.third.system.dataphin.sdk.vo.Zmsd187Vo;
import com.biz.crm.tpm.business.third.system.local.service.PullFeePoolDataService;
import com.biz.crm.tpm.business.third.system.sdk.constants.FeePoolConstant;
import com.biz.crm.tpm.business.third.system.sdk.service.Zmsd121Service;
import com.biz.crm.tpm.business.third.system.sdk.service.Zmsd187Service;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 拉取费用池信息
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2023-08-16 17:52
 */
@Service
@Slf4j
public class PullFeePoolDataServiceImpl implements PullFeePoolDataService {

    @Autowired(required = false)
    private DataPhinZmsd187Service dataPhinZmsd187Service;

    @Autowired(required = false)
    private DataPhinZmsd121Service dataPhinZmsd121Service;

    @Autowired(required = false)
    private Zmsd187Service zmsd187Service;

    @Autowired(required = false)
    private Zmsd121Service zmsd121Service;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 异步拉取费用池余额
     *
     * @param ds
     * @param accountingPeriod
     * @param userIdentity
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-08-16 17:45
     */
    @Override
    @Async
    public void pullFeePoolDataAsync(String ds, String accountingPeriod, AbstractCrmUserIdentity userIdentity) {
        loginUserService.refreshAuthentication(userIdentity);
        this.pullFeePoolData(ds, accountingPeriod);
    }

    /**
     * 拉取费用池余额
     *
     * @param ds
     * @param accountingPeriod
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-08-16 17:45
     */
    @Override
    public void pullFeePoolData(String ds, String accountingPeriod) {
        String lockKey = FeePoolConstant.PULL_FEE_POOL_DATA_LOCK;
        boolean hasLock = redisLockService.tryLock(lockKey, TimeUnit.HOURS, 6);
        Validate.isTrue(hasLock, "正在拉取数据，请稍后");
        try {
            Integer size = FeePoolConstant.PAGE_SIZE;
            //取ZMSD187数据
            //获取总数
            Integer total = dataPhinZmsd187Service.getZmsd187DataTotal(ds, accountingPeriod);
            //调用总次数
            int page = total / size;
            page = total % size == 0 ? page : page + 1;
            log.info("费用池余额自动同步数据187 共{}条数据, 需调用{}次, 每次获取{}条数据", total, page, size);
            int i = 0;
            //根据ds清除已有数据
            zmsd187Service.deleteByDs(ds);
            //分批查询
            for (; i < page; i++) {
                try {
                    List<Zmsd187Vo> zmsd187VoList = dataPhinZmsd187Service.getZmsd187Data(ds, accountingPeriod, i, size);
                    zmsd187Service.saveEntityBatch(ds, zmsd187VoList);
                    log.info("费用池余额自动同步数据187 第{} 批数据保存成功！", i);
                } catch (Exception e) {
                    log.error("费用池余额自动同步数据187 第{} 批数据保存失败！" + e.getMessage(), i);
                }
            }
            //取ZMSD121数据
            //获取总数
            total = dataPhinZmsd121Service.getZmsd121DataTotal(ds, accountingPeriod);
            //调用总次数
            page = total / size;
            page = total % size == 0 ? page : page + 1;
            log.info("费用池余额自动同步数据121 共{}条数据, 需调用{}次, 每次获取{}条数据", total, page, size);
            //分批查询
            int j = 0;
            //根据ds清除已有数据
            zmsd121Service.deleteByDs(ds);
            for (; j < page; j++) {
                try {
                    List<Zmsd121Vo> zmsd121VoList = dataPhinZmsd121Service.getZmsd121Data(ds, accountingPeriod, j, size);
                    zmsd121Service.saveEntityBatch(ds, zmsd121VoList);
                    log.info("费用池余额自动同步数据121 第{} 批数据保存成功！", i + j);
                } catch (Exception e) {
                    log.error("费用池余额自动同步数据121 第{} 批数据保存失败！" + e.getMessage(), j);
                }
            }

            // 数据处理完成后，通过消息通知预提明细执行后续处理
            // 封装信息，传递到mq
            // 发送MQ消息
            MqMessageVo mqMessage = new MqMessageVo();
            mqMessage.setTopic(FeePoolConstant.TPM_WITHHOLDING_DETAIL_TOPIC + RocketMqUtil.mqEnvironment());
            //数据
            mqMessage.setMsgBody(ds);
            //MQ标签
            mqMessage.setTag(FeePoolConstant.WITHHOLDING_DETAIL_FEE_POOL_TAG);
            rocketMqProducer.sendMqOrderMsg(mqMessage, ds, 10);
        } catch (Exception e) {
            log.error("费用池拉取时报错{}", e.getMessage());
            log.error("", e);
            throw e;
        } finally {
            redisLockService.unlock(lockKey);
        }
    }

}
