package com.biz.crm.tpm.business.third.system.local.service.impl;


import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.aliyun.openservices.shade.com.google.common.collect.Maps;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgSubComOrgService;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgSubComOrgVo;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.eunm.BusinessFormatEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.base.util.UuidCrmUtil;
import com.biz.crm.mn.third.system.dataphin.sdk.DataphinService;
import com.biz.crm.mn.third.system.dataphin.sdk.vo.ProfitabilityAnalysisVo;
import com.biz.crm.tpm.business.third.system.local.entity.TpmProfitabilityAnalysis;
import com.biz.crm.tpm.business.third.system.local.repository.TpmProfitabilityAnalysisRepository;
import com.biz.crm.tpm.business.third.system.sdk.constants.CeConstant;
import com.biz.crm.tpm.business.third.system.sdk.dto.TpmProfitabilityAnalysisDto;
import com.biz.crm.tpm.business.third.system.sdk.dto.TpmZtfi099ForEctQueryDto;
import com.biz.crm.tpm.business.third.system.sdk.service.TpmProfitabilityAnalysisService;
import com.biz.crm.tpm.business.third.system.sdk.vo.TpmProfitabilityAnalysisVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Sap099获利能力分析数据表(TpmProfitabilityAnalysis)表服务实现类
 *
 * @author huanglong
 * @date 2023-01-13 12:01:42
 */
@Slf4j
@Service("tpmProfitabilityAnalysisService")
public class TpmProfitabilityAnalysisServiceImpl implements TpmProfitabilityAnalysisService {

    @Autowired(required = false)
    private TpmProfitabilityAnalysisRepository tpmProfitabilityAnalysisRepository;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DataphinService dataphinService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private SalesOrgSubComOrgService salesOrgSubComOrgService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private RedisService redisService;

    /**
     * 根据年月执行099同步
     *
     * @param yearMonth   年月  yyyyMM
     * @param companyCode 公司代码
     * @param accountDate 过账日期
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2024/3/29 11:12
     */
    @Override
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.NOT_SUPPORTED)
    public void pull099ByYearMonth(String yearMonth, String companyCode, String accountDate) {
        Assert.hasLength(yearMonth, "年月不能为空");
        if (yearMonth.contains("-")) {
            yearMonth = yearMonth.replace("-", "");
        }
        if (StringUtils.isNotEmpty(accountDate)
                && accountDate.contains("-")) {
            accountDate = accountDate.replace("-", "");
        }
        Assert.isTrue(yearMonth.length() == 6, "年月格式不合法![yyyyMM]");
        //拉取ds  为 02的时间分区 2024年3月28日  确认
        String ds = yearMonth + "02";
        String lockKey = CeConstant.BJX099_PULL_LOCK_PREFIX + yearMonth;
        String successKey = CeConstant.BJX099_PULL_FLAG_PREFIX + yearMonth;
        Object successFlag = redisService.get(successKey);
        if (Objects.nonNull(successFlag)
                && BooleanEnum.TRUE.getCapital().equals(successFlag.toString())) {
            log.info("099报表拉取数据ds分区[{}]yearMonth[{}]companyCode[{}]accountDate[{}]本次忽略,上次已成功!", ds, yearMonth, companyCode, accountDate);
            return;
        }
        boolean lock = this.redisLockService.tryLock(lockKey, TimeUnit.HOURS, 5);
        Assert.isTrue(lock, "上次拉取【" + yearMonth + "】月尚未完成");
        try {
            Integer total = doExecuteZtfi099ByMonth(ds, yearMonth, companyCode, accountDate);
            if (Objects.nonNull(total)
                    && total > 0
                    && StringUtil.isEmpty(companyCode)
                    && StringUtil.isEmpty(accountDate)) {
                //当月拉取成功标记!
                redisService.set(successKey, BooleanEnum.TRUE.getCapital(), 7 * 24 * 60 * 60);
            }
            log.info("099报表拉取数据ds分区[{}]yearMonth[{}]companyCode[{}]accountDate[{}]成功!", ds, yearMonth, companyCode, accountDate);
        } finally {
            this.redisLockService.unlock(lockKey);
        }
    }

    /**
     * 根据年月执行099同步
     *
     * @param yearMonth   年月  yyyyMM
     * @param companyCode 公司代码
     * @param accountDate 过账日期
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2024/3/29 11:12
     */
    @Override
    @Async
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.NOT_SUPPORTED)
    public void handPull099ByYearMonthAsync(String yearMonth, String companyCode, String accountDate) {
        Assert.hasLength(yearMonth, "年月不能为空");
        loginUserService.refreshAuthentication(null);
        this.pull099ByYearMonth(yearMonth, companyCode, accountDate);
    }

    private Integer doExecuteZtfi099ByMonth(String ds, String yearMonth, String companyCode, String accountDate) {
        String lastYearMonth = DateUtil.format(DateUtil.formatAddMonth(DateUtil.parse(yearMonth, DateUtil.DEFAULT_YEAR_MONTH_NO_CH), -1),
                DateUtil.DEFAULT_YEAR_MONTH_NO_CH);
        log.info("099报表拉取数据ds分区删除数据[{}]yearMonthLy[{}]companyCode[{}]accountDate[{}]", ds, lastYearMonth, companyCode, accountDate);
        tpmProfitabilityAnalysisRepository.removeByQuery(ds, lastYearMonth, companyCode, accountDate);
        Integer total = dataphinService.getZtfi099Total(ds, companyCode, accountDate);
        log.info("099报表拉取数据ds分区[{}]总数[{}]", ds, total);
        String tenantCode = TenantUtils.getTenantCode();
        Date nowTime = new Date();
        if (total > 0) {
            AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
            int pageSize = 1000;
            int pageNumber = total / pageSize;
            int startPageNum = 0;
            int endPageNum = 19;
            do {
                if (startPageNum > endPageNum) {
                    return total;
                }
                try {
                    log.info("099报表拉取数据ds分区[{}][开始页[{}]结束页[{}]总页[{}]剩余页[{}]", ds, startPageNum, endPageNum, pageNumber, pageNumber - endPageNum - 1);
                    List<ProfitabilityAnalysisVo> data = dataphinService.getZtfi099Data(ds, companyCode, accountDate, startPageNum, endPageNum, pageSize);
                    Collection<TpmProfitabilityAnalysis> list = nebulaToolkitService.copyCollectionByWhiteList(data
                            , ProfitabilityAnalysisVo.class, TpmProfitabilityAnalysis.class, HashSet.class, ArrayList.class);

                    List<String> materialCodes = list.stream()
                            .filter(item -> CharSequenceUtil.isNotEmpty(item.getMaterialCode()))
                            .map(o -> o.getMaterialCode().substring(6))
                            .distinct().collect(Collectors.toList());

                    List<ProductVo> productVoList = productVoService.findByProductCodes(materialCodes);
                    Map<String, ProductVo> productMap = productVoList
                            .stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity(), (newValue, oldValue) -> newValue));

                    Map<String, SalesOrgSubComOrgVo> salesOrgSubComOrgMap = Maps.newHashMap();
                    List<String> salesOrgCodes = list.stream().map(TpmProfitabilityAnalysis::getSalesOrgCode).distinct().collect(Collectors.toList());
                    List<SalesOrgSubComOrgVo> salesOrgList = salesOrgSubComOrgService.listBySalesOrgCodeList(salesOrgCodes, BusinessFormatEnum.NORMAL.getCode());
                    if (CollectionUtils.isNotEmpty(salesOrgList)) {
                        salesOrgSubComOrgMap = salesOrgList.stream().collect(Collectors.toMap(SalesOrgSubComOrgVo::getSalesOrgCode, Function.identity(), (v1, v2) -> v2));
                    }
                    for (TpmProfitabilityAnalysis pro : list) {
                        if (CollectionUtil.isNotEmpty(salesOrgSubComOrgMap)) {
                            SalesOrgSubComOrgVo salesOrgSubComOrgVo = salesOrgSubComOrgMap.get(pro.getSalesOrgCode());
                            if (Objects.nonNull(salesOrgSubComOrgVo)) {
                                pro.setOrgCode(salesOrgSubComOrgVo.getOrgCode());
                                pro.setOrgName(salesOrgSubComOrgVo.getOrgName());
                            }
                        }
                    }

                    list.forEach(o -> {
                        o.setId(UuidCrmUtil.general());
                        o.setTenantCode(tenantCode);
                        o.setYearMonthLy(o.getAccountDate().substring(0, 6));
                        o.setCreateName(loginUser.getRealName());
                        o.setCreateAccount(loginUser.getAccount());
                        o.setCreateTime(nowTime);
                        o.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                        o.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                        o.setCustomerCode(o.getDealerCode() + o.getSalesOrgCode() + o.getChannelCode() + o.getSpartCode());
                        o.setSalesInstitutionErpCode(o.getSalesOrgCode());
                        o.setSalesInstitutionCode(o.getChannelCode() + o.getSpartCode() + o.getSalesOrgCode());
                        o.setSalesOrgCode("");
                        if (o.getMaterialCode().startsWith("000000")) {
                            o.setMaterialCode(o.getMaterialCode().substring(6));
                        }
                        String materialCode = o.getMaterialCode();
                        ProductVo productVo = productMap.get(materialCode);
                        if (productVo != null) {
                            o.setCategoryCode(productVo.getProductCategoryCode());
                            o.setCategoryName(productVo.getProductCategoryName());
                            o.setItemCode(productVo.getProductLevelCode());
                            o.setItemName(productVo.getProductLevelName());
                        }
                    });
                    setExtandInfo(list);
                    this.tpmProfitabilityAnalysisRepository.batchSave(list);
                    //已经拉取到最后一页,退出循环
                    if (pageNumber <= endPageNum) {
                        return total;
                    }
                    startPageNum = endPageNum + 1;
                    endPageNum = endPageNum + 10;
                    //重置最后一页
                    if (pageNumber < endPageNum) {
                        endPageNum = pageNumber;
                    }
                } catch (Exception e) {
                    log.info("099报表拉取数据ds分区[{}] 异常\n{}", ds, e.getMessage());
                    log.error("", e);
                    throw e;
                }
            } while (endPageNum <= pageNumber);
        }
        return total;
    }


    private void setExtandInfo(Collection<TpmProfitabilityAnalysis> list) {
        if (CollectionUtil.isEmpty(list)) {
            return;
        }
        List<String> salesInstitutionCodeList = list.stream()
                .filter(item -> StringUtils.isNotEmpty(item.getSalesInstitutionCode()))
                .map(TpmProfitabilityAnalysis::getSalesInstitutionCode)
                .distinct().collect(Collectors.toList());

        List<String> customerCodeList = list.stream()
                .filter(item -> StringUtils.isNotEmpty(item.getCustomerCode()))
                .map(TpmProfitabilityAnalysis::getCustomerCode)
                .distinct().collect(Collectors.toList());

        Map<String, CustomerVo> customerVoMap = Maps.newHashMap();
        if (CollectionUtil.isNotEmpty(customerCodeList)) {
            List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(customerCodeList);
            if (CollectionUtil.isNotEmpty(customerVos)) {
                customerVoMap.putAll(customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, v -> v, (n, o) -> n)));
            }

        }
        Map<String, SalesOrgVo> salesOrgVoMap = Maps.newHashMap();
        if (CollectionUtil.isNotEmpty(salesInstitutionCodeList)) {
            Map<String, SalesOrgVo> salesOrgMap = salesOrgVoService.getSalesMapByTpmCodes(salesInstitutionCodeList);
            if (CollectionUtil.isNotEmpty(salesOrgMap)) {
                salesOrgVoMap.putAll(salesOrgMap);
            }
        }

        for (TpmProfitabilityAnalysis entity : list) {
            if (StringUtils.isNotEmpty(entity.getSalesInstitutionCode())) {
                SalesOrgVo salesOrgVo = salesOrgVoMap.get(entity.getSalesInstitutionCode());
                if (null != salesOrgVo) {
                    entity.setSalesInstitutionName(salesOrgVo.getSalesOrgName());
                }
            }
            if (StringUtils.isNotEmpty(entity.getCustomerCode())) {
                CustomerVo customerVo = customerVoMap.get(entity.getCustomerCode());
                if (null != customerVo) {
                    entity.setSalesRegionName(customerVo.getSalesRegionName());
                    entity.setSalesRegionCode(customerVo.getSalesRegionCode());
                    entity.setSalesRegionErpCode(customerVo.getSalesRegionErpCode());

                    entity.setSalesOrgName(customerVo.getSalesOrgName());
                    entity.setSalesOrgCode(customerVo.getSalesOrgCode());
                    entity.setSalesOrgErpCode(customerVo.getSalesOrgErpCode());
                }
            }

        }
    }

    @Override
    public List<TpmProfitabilityAnalysisVo> listForVariable(TpmProfitabilityAnalysisDto dto) {

        if (StringUtils.isNotEmpty(dto.getChannelCode())) {
            Map<String, String> map = dictToolkitService.findMapByDictTypeCode(CeConstant.MDM_CHANNEL_R_SAP);
            dto.setChannelCode(map.getOrDefault(dto.getChannelCode(), dto.getChannelCode()));
        }

        return tpmProfitabilityAnalysisRepository.listForVariable(dto);
    }

    @Override
    public Page<TpmProfitabilityAnalysisVo> findData4Ect(Pageable pageable, TpmZtfi099ForEctQueryDto ectZtfi099QueryDto) {
        return tpmProfitabilityAnalysisRepository.findData4Ect(pageable, ectZtfi099QueryDto);
    }

    /**
     * 查询099报表 实际理论毛利=实际收入（含税）-实际成本（含税） 条件；发货过账日期+公司代码
     *
     * @param analysisDtos
     * @return
     */
    @Override
    public List<TpmProfitabilityAnalysisVo> findDataSummaryByProfitMonitor(List<TpmProfitabilityAnalysisDto> analysisDtos) {
        return this.tpmProfitabilityAnalysisRepository.findDataSummaryByProfitMonitor(analysisDtos);
    }

    @Override
    public BigDecimal computeAmountForProductMergeSaleAmount(TpmProfitabilityAnalysisDto dto) {
        if(Objects.isNull(dto)){
            return BigDecimal.ZERO;
        }
        return this.tpmProfitabilityAnalysisRepository.computeAmountForProductMergeSaleAmount(dto);
    }
}
