package com.biz.crm.tpm.business.audit.fee.validation.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.mn.common.base.util.UuidCrmUtil;
import com.biz.crm.tpm.business.activity.form.sdk.dto.ActivityFormAuditDto;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.activity.form.sdk.vo.ActivityFormVo;
import com.biz.crm.tpm.business.audit.fee.validation.local.cache.AuditFeeValidationDetailCacheHelper;
import com.biz.crm.tpm.business.audit.fee.validation.local.entity.AuditFeeValidationInfoEntity;
import com.biz.crm.tpm.business.audit.fee.validation.local.repository.AuditFeeValidationInfoAttachmentRepository;
import com.biz.crm.tpm.business.audit.fee.validation.local.repository.AuditFeeValidationInfoRepository;
import com.biz.crm.tpm.business.audit.fee.validation.local.service.AuditFeeValidationInfoAttachmentService;
import com.biz.crm.tpm.business.audit.fee.validation.local.service.AuditFeeValidationInfoService;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.constant.AuditFeeValidationConstant;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.dto.AuditFeeValidationDetailDto;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.dto.AuditFeeValidationDto;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.dto.AuditFeeValidationInfoAttachmentDto;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.service.AuditFeeValidationDetailVoService;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.vo.AuditFeeValidationInfoAttachmentVo;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.vo.AuditFeeValidationInfoVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @Description
 * @Author wei·yang
 * @Date Created in 2023/9/28 13:44
 */
@Slf4j
@Service("auditFeeValidationInfoService")
public class AuditFeeValidationInfoServiceImpl implements AuditFeeValidationInfoService {

    @Resource
    private AuditFeeValidationInfoRepository auditFeeValidationInfoRepository;

    @Resource
    private AuditFeeValidationInfoAttachmentRepository auditFeeValidationInfoAttachmentRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private AuditFeeValidationInfoAttachmentService auditFeeValidationInfoAttachmentService;

    @Resource
    private RedisTemplate<String, Object> redisTemplate;

    @Resource
    private AuditFeeValidationDetailVoService auditFeeValidationDetailVoService;

    @Resource
    private ActivityFormService activityFormService;

    @Autowired(required = false)
    private AuditFeeValidationDetailCacheHelper helper;

    /**
     * 保存核销资料
     *  @param auditCode
     * @param isEdit
     * @param cacheKey
     */
    @Override
    public void createOrUpdate(String auditCode, boolean isEdit,String cacheKey) {
        //删除数据
        if(isEdit) {
            this.auditFeeValidationInfoRepository.deleteByAuditCode(auditCode);
        }
        //从缓存中获取核销资料
        List<AuditFeeValidationInfoVo> auditInfoList = getAuditInfoFromCache(cacheKey);
        log.info("保存时核销资料,{}",JSON.toJSONString(auditInfoList));
        //保存数据
        if (CollectionUtils.isEmpty(auditInfoList)) {
            return;
        }
        //核销资料附件
        List<AuditFeeValidationInfoAttachmentVo> attachmentVos = Lists.newArrayList();
        for (AuditFeeValidationInfoVo o : auditInfoList) {
            o.setAuditCode(auditCode);
            o.setId(UuidCrmUtil.randomUuid());
            o.setTenantCode(TenantUtils.getTenantCode());
            if (!CollectionUtils.isEmpty(o.getAuditInfoAttachmentList())) {
                o.getAuditInfoAttachmentList().forEach(att -> {
                    att.setAuditCode(auditCode);
                    att.setAuditInfoId(o.getId());
                    o.setTenantCode(TenantUtils.getTenantCode());
                });
                attachmentVos.addAll(o.getAuditInfoAttachmentList());
            }
        }
        List<AuditFeeValidationInfoEntity> entities = (List<AuditFeeValidationInfoEntity>) this.nebulaToolkitService
                .copyCollectionByWhiteList(auditInfoList, AuditFeeValidationInfoVo.class
                        , AuditFeeValidationInfoEntity.class, HashSet.class, ArrayList.class);
        this.auditFeeValidationInfoRepository.saveBatch(entities);
        //保存核销资料附件
        this.auditFeeValidationInfoAttachmentService.save(auditCode, attachmentVos);
    }

    /**
     * 查询核销资料
     *
     * @param auditCode
     * @return {@link List}<{@link AuditFeeValidationInfoVo}>
     */
    @Override
    public List<AuditFeeValidationInfoVo> findByAuditCode(String auditCode) {
        List<AuditFeeValidationInfoEntity> auditFeeInfos = this.auditFeeValidationInfoRepository.findByAuditCode(auditCode);
        if(CollectionUtils.isEmpty(auditFeeInfos)){
            return Lists.newArrayList();
        }
        List<AuditFeeValidationInfoAttachmentVo> attachmentVos = this.auditFeeValidationInfoAttachmentService.findByAuditCode(auditCode);
        Map<String, List<AuditFeeValidationInfoAttachmentVo>> attachmentVoMap = new HashMap<>();
        if(!CollectionUtils.isEmpty(attachmentVos)){
            attachmentVoMap = attachmentVos.stream().collect(Collectors.groupingBy(AuditFeeValidationInfoAttachmentVo::getAuditInfoId));
        }
        List<AuditFeeValidationInfoVo> auditFeeValidationInfoVos = (List<AuditFeeValidationInfoVo>)this.nebulaToolkitService.copyCollectionByWhiteList(auditFeeInfos, AuditFeeValidationInfoEntity.class, AuditFeeValidationInfoVo.class, LinkedHashSet.class, ArrayList.class);
        for (AuditFeeValidationInfoVo auditFeeValidationInfoVo : auditFeeValidationInfoVos) {
            List<AuditFeeValidationInfoAttachmentVo> attachmentVos1 = attachmentVoMap.get(auditFeeValidationInfoVo.getId());
            if(!CollectionUtils.isEmpty(attachmentVos1)){
                auditFeeValidationInfoVo.setAuditInfoAttachmentList(attachmentVos1);
            }
        }
        return auditFeeValidationInfoVos;
    }


    @Override
    public List<AuditFeeValidationInfoVo> findAuditInfo(AuditFeeValidationDto dto) {

        Validate.notNull(dto,"输入参数不能为空");
        Validate.notBlank(dto.getCacheKey(),"缓存key不能为空");
        String cacheKey = dto.getCacheKey();
        String key = this.helper.getRedisCacheIdKey(cacheKey);
        if(Boolean.TRUE.equals(redisTemplate.hasKey(key))){
            List<AuditFeeValidationDetailDto> detailDtoList = auditFeeValidationDetailVoService.findCacheList(cacheKey);
            log.info("查看核销资料所有明细,{}", JSON.toJSONString(detailDtoList));
            if(CollectionUtils.isEmpty(detailDtoList)){
                return Lists.newArrayList();
            }
            Set<String> activityFormCodes = detailDtoList.stream().map(AuditFeeValidationDetailDto::getActivityFormCode).filter(StringUtils::isNotEmpty).collect(Collectors.toSet());
            List<AuditFeeValidationInfoVo> auditInfoFromCache = getAuditInfoFromCache(cacheKey);
            log.info("查看核销资料缓存资料,{}", JSON.toJSONString(auditInfoFromCache));
            Map<String, List<AuditFeeValidationInfoVo>> auditInfoCacheMap = new HashMap<>();
            if(!CollectionUtils.isEmpty(auditInfoFromCache)){
                auditInfoCacheMap = auditInfoFromCache.stream().collect(Collectors.groupingBy(AuditFeeValidationInfoVo::getActivityFormCode));
            }
            List<AuditFeeValidationInfoVo> auditInfoVoList = new ArrayList<>();
            Set<String> addActivityFormCodes = new HashSet<>();
            if (!CollectionUtils.isEmpty(auditInfoFromCache) && auditInfoFromCache.size() > 0) {
                for (String activityFormCode : activityFormCodes) {
                    List<AuditFeeValidationInfoVo> auditFeeValidationInfoVos = auditInfoCacheMap.get(activityFormCode);
                    if(CollectionUtils.isEmpty(auditFeeValidationInfoVos)){
                        addActivityFormCodes.add(activityFormCode);
                    }else {
                        auditInfoVoList.addAll(auditFeeValidationInfoVos);
                    }
                }
            }else {
                addActivityFormCodes = activityFormCodes;
            }
            log.info("查看核销资料所有形式编码,{}", JSON.toJSONString(addActivityFormCodes));
            if(!CollectionUtils.isEmpty(addActivityFormCodes)) {
                List<ActivityFormVo> activityFormVos = activityFormService.findActivityFormByCode(addActivityFormCodes);
                log.info("查看核销资料形式带出资料,{}", JSON.toJSONString(activityFormVos));
                if (org.apache.commons.collections.CollectionUtils.isNotEmpty(activityFormVos)) {
                    for (ActivityFormVo activityFormVo : activityFormVos) {
                        List<ActivityFormAuditDto> activityFormAuditDtoList = activityFormVo.getActivityFormAuditDtoList();
                        if (!CollectionUtils.isEmpty(activityFormAuditDtoList)) {
                            for (ActivityFormAuditDto activityFormAuditDto : activityFormAuditDtoList) {
                                AuditFeeValidationInfoVo auditInfoVo = new AuditFeeValidationInfoVo();
                                auditInfoVo.setId(UUID.randomUUID().toString().replace("-", ""));
                                auditInfoVo.setActivityFormCode(activityFormVo.getActivityFormCode());
                                auditInfoVo.setActivityFormName(activityFormVo.getActivityFormName());
                                if (Objects.nonNull(activityFormAuditDto.getMnApprovalCollect())) {
                                    auditInfoVo.setDescription(activityFormAuditDto.getMnApprovalCollect().getName());
                                }
                                auditInfoVoList.add(auditInfoVo);
                            }
                        }
                    }
                }
            }
            List<AuditFeeValidationInfoVo> infoVos = auditInfoVoList.stream().collect(Collectors.collectingAndThen(
                    Collectors.toCollection(() ->
                            new TreeSet<>(Comparator.comparing(a -> a.getActivityFormCode() + a.getDescription()))), ArrayList::new));
            return infoVos;

        }else {
            if(StringUtils.isNotEmpty(dto.getAuditCode())){
                String auditCode = dto.getAuditCode();
                List<AuditFeeValidationInfoVo> auditFeeValidationInfoVos = findByAuditCode(auditCode);
                //放入缓存
                saveAuditInfoCache(cacheKey,auditFeeValidationInfoVos);
            }
        }
        return Lists.newArrayList();
    }

    @Override
    public void saveAuditInfoCache(AuditFeeValidationDto dto) {
        Validate.notNull(dto,"输入参数不能为空");
        Validate.notBlank(dto.getCacheKey()," 缓存key不能为空");
        if(!CollectionUtils.isEmpty(dto.getAuditInfoList())){
            log.info("核销资料拷贝前,{}",JSON.toJSONString(dto.getAuditInfoList()));
            //List<AuditFeeValidationInfoVo> auditFeeValidationInfoVos = (List<AuditFeeValidationInfoVo>)this.nebulaToolkitService.copyCollectionByWhiteList(dto.getAuditInfoList(), AuditFeeValidationInfoDto.class, AuditFeeValidationInfoVo.class, LinkedHashSet.class, ArrayList.class);
            List<AuditFeeValidationInfoVo> auditFeeValidationInfoVos = dto.getAuditInfoList().stream().map(e -> {
                AuditFeeValidationInfoVo vo = new AuditFeeValidationInfoVo();
                BeanUtils.copyProperties(e,vo);
                vo.setAuditInfoAttachmentList((List<AuditFeeValidationInfoAttachmentVo>)this.nebulaToolkitService.copyCollectionByWhiteList(e.getAuditInfoAttachmentList(), AuditFeeValidationInfoAttachmentDto.class, AuditFeeValidationInfoAttachmentVo.class, LinkedHashSet.class, ArrayList.class));
                return vo;
            }).collect(Collectors.toList());
            log.info("核销资料拷贝完,{}",JSON.toJSONString(auditFeeValidationInfoVos));
            saveAuditInfoCache(dto.getCacheKey(),auditFeeValidationInfoVos);
        }
    }

    @Override
    public void saveAuditInfoCache(String cacheKey, List<AuditFeeValidationInfoVo> auditFeeValidationInfoVos) {
        if(StringUtils.isEmpty(cacheKey)){
            return;
        }
        if(CollectionUtils.isEmpty(auditFeeValidationInfoVos)){
            return;
        }
        if (!CollectionUtils.isEmpty(auditFeeValidationInfoVos)) {
            cacheKey = AuditFeeValidationConstant.AUDIT_FEE_INFO_CACHE_KEY + cacheKey;
            redisTemplate.delete(cacheKey);
            redisTemplate.opsForList().rightPushAll(cacheKey, auditFeeValidationInfoVos.toArray());
            redisTemplate.expire(cacheKey, 1, TimeUnit.HOURS);
        }
    }

    private List<AuditFeeValidationInfoVo> getAuditInfoFromCache(String cacheKey) {
        Validate.notBlank(cacheKey,"缓存key不能为空");
        cacheKey = AuditFeeValidationConstant.AUDIT_FEE_INFO_CACHE_KEY + cacheKey;
        List<AuditFeeValidationInfoVo> auditInfoVoList = new ArrayList<>();
        if (Boolean.TRUE.equals(redisTemplate.hasKey(cacheKey))) {
            List<Object> range = redisTemplate.opsForList().range(cacheKey, 0, -1);
            auditInfoVoList = (List<AuditFeeValidationInfoVo>)(List) range;
        }
        redisTemplate.expire(cacheKey, 1, TimeUnit.HOURS);
        return auditInfoVoList;
    }
}
