package com.biz.crm.tpm.business.audit.fee.validation.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.mn.common.base.eunm.YesOrNoEnum;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.activity.form.sdk.vo.ActivityFormVo;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.ledger.AuditFeeDiffLedgerDeductionDto;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.track.AuditFeeDiffTrackDetailDto;
import com.biz.crm.tpm.business.audit.fee.sdk.enumeration.AuditFeeDiffLedgerOperationTypeEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.enumeration.DiffLedgerDataSourceEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.service.ledger.AuditFeeDiffLedgerLockService;
import com.biz.crm.tpm.business.audit.fee.sdk.service.ledger.AuditFeeDiffLedgerVoService;
import com.biz.crm.tpm.business.audit.fee.sdk.service.track.AuditFeeDiffTrackDetailVoService;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.track.AuditFeeDiffTrackDetailLedgerVo;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.track.AuditFeeDiffTrackDetailVo;
import com.biz.crm.tpm.business.audit.fee.validation.local.entity.AuditFeeValidationDetailEntity;
import com.biz.crm.tpm.business.audit.fee.validation.local.entity.AuditFeeValidationEntity;
import com.biz.crm.tpm.business.audit.fee.validation.local.helper.AuditFeeValidationHelper;
import com.biz.crm.tpm.business.audit.fee.validation.local.repository.AuditFeeValidationRepository;
import com.biz.crm.tpm.business.audit.fee.validation.local.service.AuditFeeValidationDetailService;
import com.biz.crm.tpm.business.audit.fee.validation.local.service.AuditFeeValidationDetailUseLedgerService;
import com.biz.crm.tpm.business.audit.fee.validation.local.service.AuditFeeValidationInfoService;
import com.biz.crm.tpm.business.audit.fee.validation.local.service.AuditFeeValidationService;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.constant.AuditFeeValidationConstant;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.dto.AuditFeeValidationDetailDto;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.dto.AuditFeeValidationDetailUseLedgerDto;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.dto.AuditFeeValidationDto;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.enums.SaveTypeEnum;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.service.AuditFeeValidationVoService;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.vo.AuditFeeValidationDetailVo;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.vo.AuditFeeValidationVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 商超费用结案核销主表 内部接口实现
 *
 * @author weiÂ·yang
 * @date 2023-09-26 17:23:21
 */
@Slf4j
@Service("AuditFeeValidationService")
public class AuditFeeValidationServiceImpl implements AuditFeeValidationService {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Resource
    private AuditFeeValidationRepository auditFeeValidationRepository;

    @Resource
    private AuditFeeValidationVoService auditFeeValidationVoService;

    @Resource
    private AuditFeeValidationHelper auditFeeValidationHelper;

    @Resource
    private AuditFeeValidationDetailService auditFeeValidationDetailService;

    @Resource
    private AuditFeeValidationInfoService auditFeeValidationInfoService;

    @Autowired(required = false)
    private ProcessBusinessService processBusinessService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private AuditFeeDiffTrackDetailVoService auditFeeDiffTrackDetailVoService;

    @Autowired(required = false)
    private AuditFeeDiffLedgerLockService auditFeeDiffLedgerLockService;

    @Autowired(required = false)
    private AuditFeeDiffLedgerVoService auditFeeDiffLedgerVoService;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private ActivityFormService activityFormService;

    @Autowired(required = false)
    private AuditFeeValidationDetailUseLedgerService auditFeeValidationDetailUseLedgerService;

    /**
     * 新增
     *
     * @param dto
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void createOrUpdate(AuditFeeValidationDto dto) {
        this.auditFeeValidationHelper.verifyParams(dto);
        boolean isEdit = YesOrNoEnum.YES.getCode().equals(dto.getIsEdit());
        if (!isEdit) {
            dto.setAuditCode(this.auditFeeValidationHelper.generateCode());
        }
        if (StringUtils.isEmpty(dto.getProcessStatus())) {
            dto.setProcessStatus(ProcessStatusEnum.PREPARE.getKey());
        }
        //主表
        AuditFeeValidationEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto,
                AuditFeeValidationEntity.class, HashSet.class, ArrayList.class);
        entity.setTenantCode(TenantUtils.getTenantCode());

        //保存明细
        List<AuditFeeValidationDetailDto> detailDtos = this.auditFeeValidationDetailService.createOrUpdate(entity.getAuditCode(), isEdit, dto.getCacheKey(),dto.getSaveType());

        //找到明细中，关联了相同差异追踪明细切状态为‘非审批通过’的数据
        List<String> detailCodes = detailDtos.stream().map(AuditFeeValidationDetailDto::getDetailCode).collect(Collectors.toList());
        List<AuditFeeValidationDetailDto> inApprovalCodes = auditFeeValidationDetailService.findInApprovalValidation(isEdit ? entity.getId() : null,entity.getAuditCode(),detailCodes,ProcessStatusEnum.PASS.getDictCode());
        if (!CollectionUtils.isEmpty(inApprovalCodes)) {
            AuditFeeValidationDetailDto msg = inApprovalCodes.get(0);
            throw new RuntimeException("差异追踪明细【"+msg.getDetailCode()+"】所在【"+msg.getAuditCode()+"】差异结案正在流程中，不允许发起新的流程，请处理！");
        }

        //保存核销资料
        this.auditFeeValidationInfoService.createOrUpdate(entity.getAuditCode(), isEdit, dto.getCacheKey());
        BigDecimal reduce = detailDtos.stream().map(AuditFeeValidationDetailDto::getThisAuditAmount).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
        entity.setThisEndCaseTotalAmount(Optional.ofNullable(reduce).orElse(BigDecimal.ZERO));
        //保存主表
        this.auditFeeValidationRepository.saveOrUpdate(entity);

        if (SaveTypeEnum.SUBMIT.getCode().equals(dto.getSaveType())) {
            submitApproval(dto);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void submitApproval(AuditFeeValidationDto dto) {

        Assert.notNull(dto, "提交的业务对象不能为空!");
        Assert.notNull(dto.getProcessBusiness(), "流程信息不能为空!");
        AuditFeeValidationEntity entity = this.auditFeeValidationRepository.findByAuditCode(dto.getAuditCode());
        Assert.notNull(entity, "单据[" + dto.getAuditCode() + "]不存在,请刷新重新选择单据!");
        Validate.isTrue(ProcessStatusEnum.PREPARE.getDictCode().equals(entity.getProcessStatus())
                        || ProcessStatusEnum.REJECT.getDictCode().equals(entity.getProcessStatus())
                        || ProcessStatusEnum.RECOVER.getDictCode().equals(entity.getProcessStatus())
                        || StringUtils.isBlank(entity.getProcessStatus()),
                "当前流程状态不能提交审批");

        //操作预算
        this.operateAmount(dto.getCurrentOperateId(), dto.getAuditCode());

        ProcessBusinessDto processBusiness = Optional.ofNullable(dto.getProcessBusiness()).orElse(new ProcessBusinessDto());

        processBusiness.setBusinessNo(dto.getAuditCode());
        processBusiness.setProcessTitle("差异费用结案[" + dto.getAuditName() + "]发起审批流");
        JSONObject jsonObject = JsonUtils.toJSONObject(dto);

        processBusiness.setBusinessFormJson(jsonObject.toJSONString());
        processBusiness.setBusinessCode(AuditFeeValidationConstant.TPM_AUDIT_FEE_PROCESS);
        ProcessBusinessVo processBusinessVo = this.processBusinessService.processStart(processBusiness);
        this.auditFeeValidationRepository.updateProcessStatusByAuditCode(dto.getAuditCode(), ProcessStatusEnum.COMMIT, processBusinessVo.getProcessNo());
    }

    //处理需要扣减的明细行
    private void operateAmount(String currentOperateId, String auditCode) {
        FacturerUserDetails loginUserDetails = (FacturerUserDetails)this.loginUserService.getLoginDetails(FacturerUserDetails.class);
        List<AuditFeeValidationDetailEntity> details = this.auditFeeValidationDetailService.findByAuditCode(auditCode);
        if (CollectionUtils.isEmpty(details)) {
            return;
        }
        //查询活动形式
        List<String> activityFormCodes = details.stream().map(AuditFeeValidationDetailEntity::getActivityFormCode).distinct().collect(Collectors.toList());
        List<ActivityFormVo> activityFormVoList = activityFormService.findByCodes(activityFormCodes);
        Validate.notEmpty(activityFormVoList,"未在‘活动形式管理’查询到相关活动形式信息");
        Map<String, ActivityFormVo> activityFormVoMap = activityFormVoList.stream().collect(Collectors.toMap(ActivityFormVo::getActivityFormCode, Function.identity(), (v1, v2) -> v2));

        //查询明细对应的差异台账数据
        List<String> detailCodes = details.stream().map(AuditFeeValidationDetailEntity::getDetailCode).distinct().collect(Collectors.toList());
        List<AuditFeeDiffTrackDetailLedgerVo> ledgers = this.auditFeeDiffTrackDetailVoService.findLedgerByDetailCodes(detailCodes);
        //每个追踪，及其关联的差异费用台账map
        Map<String, List<AuditFeeDiffTrackDetailLedgerVo>> ledgerVoMap = ledgers.stream().collect(Collectors.groupingBy(AuditFeeDiffTrackDetailLedgerVo::getDetailCode));

        //操作差异费用台账list
        List<AuditFeeDiffLedgerDeductionDto> operateDto = Lists.newArrayList();
        //保存抵扣记录
        List<AuditFeeValidationDetailUseLedgerDto> useLedgerList = new ArrayList<>();

        //提交的时候是扣减多出来的。判断当前结案金额+已结案金额是否大于申请金额。大于就要补扣
        details.forEach(detail -> {
            detail.setAlreadyAuditAmount(ObjectUtils.defaultIfNull(detail.getAlreadyAuditAmount(), BigDecimal.ZERO));
            detail.setThisAuditAmount(ObjectUtils.defaultIfNull(detail.getThisAuditAmount(), BigDecimal.ZERO));
            detail.setApplyAmount(ObjectUtils.defaultIfNull(detail.getApplyAmount(), BigDecimal.ZERO));
            BigDecimal amount = detail.getAlreadyAuditAmount().add(detail.getThisAuditAmount());
            //超额抵扣关联的差异费用台账
            if (amount.compareTo(detail.getApplyAmount()) > 0 ) {

                ActivityFormVo activityFormVo = activityFormVoMap.get(detail.getActivityFormCode());
                Validate.notNull(activityFormVo,"未在‘活动形式管理’查询到,差异费用追踪【"+detail.getDetailCode()+"】关联的活动形式【" + detail.getActivityFormCode() + "】");
                Validate.isTrue(activityFormVo.getIsAllowExcessAudit(),"差异费用追踪【"+detail.getDetailCode()+"】,不允许超额核销，请修改“本次结案金额”！");

                if (CollectionUtils.isEmpty(ledgerVoMap) || CollectionUtils.isEmpty(ledgerVoMap.get(detail.getDetailCode()))) {
                    throw new RuntimeException("差异费用追踪【"+detail.getDetailCode()+"】,没有关联的差异费用台账，无法抵扣金额");
                }
                //待抵扣金额
                BigDecimal recoveredAmountTotal = amount.subtract(detail.getApplyAmount());

                List<AuditFeeDiffTrackDetailLedgerVo> ledgerVoList = ledgerVoMap.get(detail.getDetailCode());
                //按时间先后排序
                ledgerVoList = ledgerVoList.stream().sorted(Comparator.comparing(AuditFeeDiffTrackDetailLedgerVo::getCreateTime)).collect(Collectors.toList());

                for (AuditFeeDiffTrackDetailLedgerVo auditFeeDiffTrackDetailLedgerVo : ledgerVoList) {
                    if (recoveredAmountTotal.compareTo(BigDecimal.ZERO) == 0) {
                        break;
                    }
                    BigDecimal beRecoveredAmount = Optional.ofNullable(auditFeeDiffTrackDetailLedgerVo.getBeRecoveredAmount()).orElse(BigDecimal.ZERO);
                    if (beRecoveredAmount.compareTo(BigDecimal.ZERO) == 0) {
                        continue;
                    }
                    //抵扣台账dto
                    AuditFeeDiffLedgerDeductionDto dto = new AuditFeeDiffLedgerDeductionDto();
                    dto.setOperationType(AuditFeeDiffLedgerOperationTypeEnum.DEDUCTION_AMOUNT.getCode());
                    dto.setFeeDiffLedgerCode(auditFeeDiffTrackDetailLedgerVo.getFeeDiffLedgerCode());
                    dto.setDataSource(DiffLedgerDataSourceEnum.DIFF_CLOSED.getCode());
                    dto.setBusinessCode(detail.getAuditDetailCode());
                    dto.setFeeDiffLedgerDisposeCode(detail.getDetailCode());
                    dto.setActivityFormCode(detail.getActivityFormCode());
                    dto.setActivityFormName(detail.getActivityFormName());
                    dto.setActivityTypeCode(detail.getActivityTypeCode());
                    dto.setActivityTypeName(detail.getActivityTypeName());
                    dto.setResaleCommercialCode(detail.getSystemCode());
                    dto.setResaleCommercialName(detail.getSystemName());
                    dto.setTerminalCode(detail.getTerminalCode());
                    dto.setTerminalName(detail.getTerminalName());
                    if (!Objects.isNull(loginUserDetails)){
                        dto.setOperatorAccount(loginUserDetails.getAccount());
                    }
                    if (recoveredAmountTotal.compareTo(beRecoveredAmount) > 0) {

                        dto.setRecoveredAmount(beRecoveredAmount);
                        recoveredAmountTotal = recoveredAmountTotal.subtract(beRecoveredAmount);

                    }else {
                        dto.setRecoveredAmount(recoveredAmountTotal);
                        recoveredAmountTotal = BigDecimal.ZERO;
                    }
                    operateDto.add(dto);

                    //生成抵扣记录
                    AuditFeeValidationDetailUseLedgerDto useLedget = new AuditFeeValidationDetailUseLedgerDto();
                    useLedget.setOperationType(dto.getOperationType());
                    useLedget.setFeeDiffLedgerCode(dto.getFeeDiffLedgerCode());
                    useLedget.setRecoveredAmount(dto.getRecoveredAmount());
                    useLedget.setIsReturned(BooleanEnum.FALSE.getCapital());

                    useLedget.setAuditCode(detail.getAuditCode());
                    useLedget.setAuditDetailCode(detail.getAuditDetailCode());
                    useLedget.setPlanCode(detail.getPlanCode());
                    useLedget.setPlanName(detail.getPlanName());
                    useLedget.setDetailCode(detail.getDetailCode());
                    useLedgerList.add(useLedget);
                }
            }
        });

        if (CollectionUtils.isEmpty(operateDto)) {
            return;
        }
        //抵扣台账
        this.operateAmount(currentOperateId, operateDto);
        //保存抵扣记录
        if (!CollectionUtils.isEmpty(useLedgerList)) {
            auditFeeValidationDetailUseLedgerService.saveBatch(useLedgerList);
        }
    }

    //操作差异台账
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void operateAmount(String currentOperateId, List<AuditFeeDiffLedgerDeductionDto> operateDto) {
        List<String> lockKeys = operateDto.stream().map(AuditFeeDiffLedgerDeductionDto::getFeeDiffLedgerCode).distinct().collect(Collectors.toList());
        //加锁
        boolean lock = this.auditFeeDiffLedgerLockService.lock(lockKeys, TimeUnit.SECONDS, 60);
        Validate.isTrue(lock, "系统其他单据正在处理差异费用台账,请稍后再操作");
        this.redisService.set(String.format(AuditFeeValidationConstant.AUDIT_FEE_VALIDATION_AMOUNT_LOCK, currentOperateId), lockKeys);
        operateDto.forEach(e -> this.auditFeeDiffLedgerVoService.useAmount(e));
    }

    @Override
    @Async
    public void batchRelevance(UserIdentity loginUser, AuditFeeDiffTrackDetailDto dto) {
        loginUserService.refreshAuthentication(loginUser);
        auditFeeValidationHelper.sendMsg("开始关联数据");
        int page = 1;
        int size = 1000;
        Pageable pageable = PageRequest.of(page, size);
        Page<AuditFeeDiffTrackDetailVo> resultPage = auditFeeDiffTrackDetailVoService.auditFindByConditions(pageable, dto);
        List<AuditFeeDiffTrackDetailVo> records = resultPage.getRecords();
        if (CollectionUtils.isEmpty(records)) {
            auditFeeValidationHelper.sendMsg("未查询到数据");
            return;
        }
        auditFeeValidationHelper.sendMsg("查询第" + page + "页数据，共有" + resultPage.getTotal() + "条数据");
        addDataToCache(dto.getCacheKey(), records);
        while (resultPage.hasNext()) {
            pageable = PageRequest.of(page++, size);
            resultPage = auditFeeDiffTrackDetailVoService.auditFindByConditions(pageable, dto);
            auditFeeValidationHelper.sendMsg("查询第" + page + "页数据，共有" + resultPage.getTotal() + "条数据");
            records = resultPage.getRecords();
            addDataToCache(dto.getCacheKey(), records);
        }

        auditFeeValidationHelper.sendMsg("关联数据完成", ExecStatusEnum.FINISH.getDictCode());
    }

    private void addDataToCache(String cacheKey, List<AuditFeeDiffTrackDetailVo> records) {
        if (CollectionUtils.isEmpty(records)) {
            return;
        }
        List<AuditFeeValidationDetailDto> detailDtos = (List<AuditFeeValidationDetailDto>) this.nebulaToolkitService.copyCollectionByWhiteList(records, AuditFeeDiffTrackDetailVo.class, AuditFeeValidationDetailDto.class, LinkedHashSet.class, ArrayList.class);
        auditFeeValidationDetailService.addItemCache(cacheKey, detailDtos);
        auditFeeValidationHelper.sendMsg("保存关联数据");
    }

    /**
     * 删除
     *
     * @param ids
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "主键ID集合不能为空");
        this.auditFeeValidationRepository.deleteBatch(ids);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "主键ID集合不能为空");
        this.auditFeeValidationRepository.enableBatch(ids);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "主键ID集合不能为空");
        this.auditFeeValidationRepository.disableBatch(ids);
    }

    @Override
    public AuditFeeValidationVo findConditionByAuditCode(String auditCode) {
        if (StringUtils.isBlank(auditCode)) {
            return null;
        }
        AuditFeeValidationEntity entity = this.auditFeeValidationRepository.findByAuditCode(auditCode);
        if (Objects.isNull(entity)) {
            return null;
        }
        AuditFeeValidationVo vo = nebulaToolkitService.copyObjectByWhiteList(entity, AuditFeeValidationVo.class, null, null);
        List<AuditFeeValidationDetailEntity> detailEntityList = auditFeeValidationDetailService.findByAuditCode(auditCode);
        if (CollectionUtils.isEmpty(detailEntityList)) {
            return vo;
        }
        List<AuditFeeValidationDetailVo> detailVos = (List<AuditFeeValidationDetailVo>) nebulaToolkitService.copyCollectionByWhiteList(detailEntityList, AuditFeeValidationDetailEntity.class, AuditFeeValidationDetailVo.class, HashSet.class, ArrayList.class);
        vo.setDetailList(detailVos);
        return vo;
    }

    @Override
    public void updateProcessStatusByAuditCode(String auditCode, ProcessStatusEnum statusEnum, String processNo) {
        if (StringUtils.isBlank(auditCode)){
            return;
        }
        this.auditFeeValidationRepository.updateProcessStatusByAuditCode(auditCode,statusEnum,processNo);
    }
}
