package com.biz.crm.tpm.business.daily.sales.data.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgRangeDto;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.dataphin.sdk.DataphinService;
import com.biz.crm.mn.third.system.dataphin.sdk.vo.DailySalesDataVo;
import com.biz.crm.tpm.business.daily.sales.data.local.entity.TpmDailySalesDataEntity;
import com.biz.crm.tpm.business.daily.sales.data.local.mapper.TpmDailySalesDataMapper;
import com.biz.crm.tpm.business.daily.sales.data.local.repository.TpmDailySalesDataRepository;
import com.biz.crm.tpm.business.daily.sales.data.local.service.TpmDailySalesDataTransService;
import com.biz.crm.tpm.business.daily.sales.data.sdk.constant.DailySalesDataConstant;
import com.biz.crm.tpm.business.daily.sales.data.sdk.dto.TpmDailySalesDataDto;
import com.biz.crm.tpm.business.daily.sales.data.sdk.dto.TpmDailySalesDataTotalDto;
import com.biz.crm.tpm.business.daily.sales.data.sdk.service.TpmDailySalesDataService;
import com.biz.crm.tpm.business.daily.sales.data.sdk.vo.TpmDailySalesDataTotalVo;
import com.biz.crm.tpm.business.daily.sales.data.sdk.vo.TpmDailySalesDataVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * 日销售数据(TpmDailySalesData)表服务实现类
 *
 * @author hd
 * @create 2023-02-06 15:01:38
 */
@Slf4j
@Service("tpmDailySalesDataService")
public class TpmDailySalesDataServiceImpl implements TpmDailySalesDataService {
    @Autowired(required = false)
    private TpmDailySalesDataTransService tpmDailySalesDataTransService;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private LoginUserService loginUserService;
    @Autowired(required = false)
    private DataphinService dataphinService;
    @Autowired(required = false)
    private TpmDailySalesDataRepository tpmDailySalesDataRepository;

    @Autowired
    private TpmDailySalesDataMapper tpmDailySalesDataMapper;
    @Autowired
    private SalesOrgVoService salesOrgVoService;


    /**
     * 通过时间分区定时拉取数据
     */
    @DynamicTaskService(cornExpression = "0 0 2 * * ?", taskDesc = "拉取前天0346北极星日销售数据-分子公司日销售报表")
    @Override
    public void pullDataList() {
        loginUserService.refreshAuthentication(null);
        Date nowTime = new Date();
        Date date = DateUtil.dateAddDay(nowTime, -2);
        String ds = DateUtil.format(date, "yyyyMMdd");
        String sql = DailySalesDataConstant.DAILY_SALES_DATA_SQL;
        pullDataOne(nowTime, ds, sql);
    }


    /**
     * 拉取数据
     *
     * @param nowTime
     * @param ds
     * @param sql
     */
    private void pullDataOne(Date nowTime, String ds, String sql) {
        AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
        Integer total = dataphinService.getDailySalesDataTotal(DailySalesDataConstant.TOTAL_NUMBER_SQL + ds + "'");
        String tenantCode = TenantUtils.getTenantCode();
        if (total > 0) {
            log.info("----------------------------------------------------------------------------------------------------------" +
                    "-------------------------------------拉取033数据开始--------时间{" + ds + "}----------总条数：{" + total + "}--------------------------------------------------------");
            int pageSize = 1000;
            int count = 20;
            int pageNumber = 0;
            int sum = total % pageSize == 0 ? total / pageSize : (total / pageSize) + 1;
            int errorRetryCount = 0;
            for (; sum > 0; ) {
                if (sum < count) {
                    count = sum;
                }
                List<DailySalesDataVo> dataList = null;
                try {
                    dataList = dataphinService.getDailySalesData(sql + " DS = '" + ds + "'" + " order by invoice_no,invoicing_items  LIMIT ", pageNumber, pageSize, count);
                    if (CollectionUtils.isEmpty(dataList)) {
                        if (pageNumber == 0) {
                            throw new RuntimeException("本次拉取" + ds + "数据为空，请检查！");
                        } else {
                            return;
                        }
                    }
                } catch (Exception e) {
                    log.error("方法名：getDailySalesData=============北极星0346日销售数据分页拉取失败，页码={}，条数={}，count={}，执行sql={}",
                            pageNumber, pageSize, count, sql);
                    log.error(e.getMessage(), e);
                    if (errorRetryCount < 3) {
                        //重试2次
                        errorRetryCount++;
                        continue;
                    }
                }
                errorRetryCount = 0;//重试次数置零

                pageNumber += count;
                sum -= count;
                if (CollectionUtils.isEmpty(dataList)) {
                    continue;
                }
                int partialLimit = 1000;
                int size = dataList.size();
                if (partialLimit < size) {
                    int start = 0;
                    for (; size > 0; ) {
                        if (size < partialLimit) {
                            partialLimit = size;
                        }
                        List<DailySalesDataVo> partList = dataList.subList(start, partialLimit);
                        size -= partialLimit;
                        saveData(nowTime, loginUser, tenantCode, partialLimit, partList);
                    }
                } else {
                    saveData(nowTime, loginUser, tenantCode, size, dataList);
                }
            }
        }
    }

    private void saveData(Date nowTime, AbstractCrmUserIdentity loginUser, String tenantCode, int partialLimit, List<DailySalesDataVo> partList) {
        if (CollectionUtils.isEmpty(partList)) {
            return;
        }
        Set<TpmDailySalesDataEntity> salesDataEntities = new TreeSet<>(Comparator.comparing(TpmDailySalesDataEntity::getVerifyCode));
        for (int i = 0; i < partialLimit; i++) {
            DailySalesDataVo dailySalesDataVo = partList.get(i);
            TpmDailySalesDataEntity o = nebulaToolkitService.copyObjectByWhiteList(dailySalesDataVo,
                    TpmDailySalesDataEntity.class, HashSet.class, ArrayList.class);
            o.setTenantCode(tenantCode);
            o.setCreateName(loginUser.getRealName());
            o.setCreateAccount(loginUser.getAccount());
            String materialCode = o.getMaterialCode();
            if (StringUtils.isNotEmpty(materialCode) && materialCode.startsWith("000000")) {
                o.setMaterialCode(materialCode.substring(6));
            }
            if (StringUtils.isNotEmpty(o.getVoucherDate())
                    && o.getVoucherDate().length() > 0) {
                o.setYearMonthLy(o.getVoucherDate().substring(0, 6));
            }
            o.setCustomerCode(o.getDealerCode() + o.getSalesOrgCode() + o.getChannelCode() + o.getSpartCode());
            o.setSalesInstitutionCode(o.getChannelCode() + o.getSpartCode() + o.getSalesOrgCode());
            o.setTpmSalesOrgCode(o.getSalesInstitutionCode() + dailySalesDataVo.getSaleOfficeCode());
            o.setSalesGroupCode(o.getTpmSalesOrgCode() + o.getSalesCode());
            o.setCreateTime(nowTime);
            o.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            o.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            // 校验字段 暂时校验发票号+发票行号 TODO
            o.setVerifyCode(o.getInvoiceNumber().concat(o.getInvoicingItems()));
            salesDataEntities.add(o);
        }
        partList.clear();
        List<String> verifyCodeList = salesDataEntities.stream()
                .map(TpmDailySalesDataEntity::getVerifyCode).collect(Collectors.toList());
        List<TpmDailySalesDataEntity> byVerifyCodes = tpmDailySalesDataRepository.findByVerifyCodes(verifyCodeList);
        if (CollectionUtils.isEmpty(byVerifyCodes)) {
            dealSalesOrgCode(Lists.newArrayList(salesDataEntities));
            tpmDailySalesDataTransService.saveBatchData(salesDataEntities);
            return;
        }
        Map<String, TpmDailySalesDataEntity> map = byVerifyCodes.stream().collect(Collectors
                .toMap(TpmDailySalesDataEntity::getVerifyCode, Function.identity(), (a, b) -> a));
        byVerifyCodes.clear();
        List<TpmDailySalesDataEntity> saveList = Lists.newArrayList();
        salesDataEntities.forEach(pull -> {
            if (!map.containsKey(pull.getVerifyCode())) {
                saveList.add(pull);
            }
        });
        map.clear();
        dealSalesOrgCode(saveList);
        tpmDailySalesDataTransService.saveBatchData(saveList);
    }

    private void dealSalesOrgCode(List<TpmDailySalesDataEntity> list) {
        List<String> collect = list.stream()
                .filter(item -> StringUtils.isNotEmpty(item.getSalesInstitutionCode()))
                .map(TpmDailySalesDataEntity::getSalesInstitutionCode)
                .distinct()
                .collect(Collectors.toList());

        Map<String, SalesOrgVo> salesOrgVoMap = salesOrgVoService.getSalesMapByTpmCodes(collect);
        if (CollectionUtils.isEmpty(salesOrgVoMap)) {
            return;
        }
        for (TpmDailySalesDataEntity entity : list) {
            if (StringUtils.isNotEmpty(entity.getSalesInstitutionCode())) {
                SalesOrgVo salesOrgVo = salesOrgVoMap.get(entity.getSalesInstitutionCode());
                if (null != salesOrgVo) {
                    entity.setSalesInstitutionName(salesOrgVo.getSalesOrgName());
                }
            }
        }
    }

    /**
     * 通过时间分区拉取数据
     *
     * @param ds
     */
    @Override
    public void pullDataList(String ds) {
        Date nowTime = new Date();
        if (StringUtils.isEmpty(ds)) {
            Date date = DateUtil.dateAddDay(nowTime, -1);
            ds = DateUtil.format(date, "yyyyMMdd");
        }
        String sql = DailySalesDataConstant.DAILY_SALES_DATA_SQL;
        pullDataOne(nowTime, ds, sql);
    }

    @Override
    public TpmDailySalesDataVo findGift(TpmDailySalesDataDto tpmDailySalesDataDto) {
        if (Objects.isNull(tpmDailySalesDataDto)) {
            return null;
        }
        TpmDailySalesDataVo tpmDailySalesDataVo = tpmDailySalesDataMapper.findGiftSum(tpmDailySalesDataDto);
        return tpmDailySalesDataVo;
    }

    @Override
    public void pullByCondition(TpmDailySalesDataDto dto) {
        Validate.notNull(dto, "参数不能为空！");
        Date nowTime = new Date();
        String ds = dto.getDs();
        Validate.notBlank(ds, "时间分区不能为空！");
        String sql = DailySalesDataConstant.DAILY_SALES_DATA_SQL;
        String salesInstitutionCode = dto.getSalesInstitutionCode();
        String saleOfficeCode = dto.getSaleOfficeCode();
        List<String> dsList = dto.getDsList();
        log.info("=====>    条件拉取033数据 start [{}]   <=====", JSON.toJSONString(dto));
        if (!CollectionUtils.isEmpty(dsList) && dsList.size() > 0) {
            if (StringUtils.isNotBlank(salesInstitutionCode)) {
                sql = sql + " sales_org_code = '" + salesInstitutionCode + "' and ";
            }
            if (StringUtils.isNotBlank(saleOfficeCode)) {
                sql = sql + " sale_office_code = '" + saleOfficeCode + "' and ";
            }
            for (int i = 0; i < dsList.size(); i++) {
                pullDataOne(nowTime, dsList.get(i), sql);
            }
        } else {
            pullDataOne(nowTime, ds, sql);
        }
        log.info("=====>    条件拉取033数据 end [{}]   <=====", JSON.toJSONString(dto));
    }

    /**
     * 根据条件统计 订单数量，单价*订单数量
     *
     * @param dto 参数
     * @return TpmDailySalesDataTotalVo
     */
    @Override
    public TpmDailySalesDataTotalVo statisticsByCondition(TpmDailySalesDataTotalDto dto) {
        if (StringUtils.isBlank(dto.getYearMonthLy()) || CollectionUtils.isEmpty(dto.getCustomerCodeList())) {
            return null;
        }
        return this.tpmDailySalesDataRepository.statisticsByCondition(dto);
    }

    @Override
    public List<TpmDailySalesDataVo> statisticsListByCondition(TpmDailySalesDataTotalDto dto) {
        if (StringUtils.isBlank(dto.getYearMonthLy()) || CollectionUtils.isEmpty(dto.getCustomerCodeList())) {
            return Lists.newArrayList();
        }
        return this.tpmDailySalesDataRepository.statisticsListByCondition(dto);
    }
    @Override
    public List<TpmDailySalesDataVo> afterDiscountAmtByCondition(TpmDailySalesDataTotalDto dto) {
        return this.tpmDailySalesDataRepository.afterDiscountAmtByCondition(dto);
    }

    /**
     * 按条件查询
     *
     * @param dto
     * @return
     */
    @Override
    public List<TpmDailySalesDataVo> findByCondition(TpmDailySalesDataTotalDto dto) {
        return tpmDailySalesDataRepository.findByCondition(dto);
    }
}
