package com.biz.crm.tpm.business.deduction.detail.mapping.local.service.internal;


import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.deduction.detail.mapping.local.entity.TpmDeductionDetailMapping;
import com.biz.crm.tpm.business.deduction.detail.mapping.local.entity.TpmDeductionDetailMappingCustomer;
import com.biz.crm.tpm.business.deduction.detail.mapping.local.entity.TpmDeductionDetailMappingRelationActivityConfig;
import com.biz.crm.tpm.business.deduction.detail.mapping.local.entity.TpmDeductionDetailMappingText;
import com.biz.crm.tpm.business.deduction.detail.mapping.local.repository.TpmDeductionDetailMappingCustomerRepository;
import com.biz.crm.tpm.business.deduction.detail.mapping.local.repository.TpmDeductionDetailMappingRelationActivityConfigRepository;
import com.biz.crm.tpm.business.deduction.detail.mapping.local.repository.TpmDeductionDetailMappingRepository;
import com.biz.crm.tpm.business.deduction.detail.mapping.local.repository.TpmDeductionDetailMappingTextRepository;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.constant.TpmDeductionDetailMappingConstant;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.dto.*;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.dto.log.TpmDeductionDetailMappingLogEventDto;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.event.log.TpmDeductionDetailMappingEvent;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.event.log.TpmDeductionDetailMappingLogEventListener;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.service.TpmDeductionDetailMappingService;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingCustomerVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingRelationActivityConfigVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingTextVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * TPM-商超扣费明细映射表(TpmDeductionDetailMapping)表服务实现类
 *
 * @author duyiran
 * @since 2022-11-07 19:04:48
 */
@Service("tpmDeductionDetailMappingService")
@Slf4j
public class TpmDeductionDetailMappingServiceImpl implements TpmDeductionDetailMappingService {

    @Autowired(required = false)
    private TpmDeductionDetailMappingRepository tpmDeductionDetailMappingRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private TpmDeductionDetailMappingRelationActivityConfigRepository deductionDetailMappingRelationActivityConfigRepository;

    @Autowired(required = false)
    private TpmDeductionDetailMappingTextRepository deductionDetailMappingTextRepository;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private TpmDeductionDetailMappingCustomerRepository tpmDeductionDetailMappingCustomerRepository;

    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param dto      dto对象
     * @return
     */
    @Override
    public Page<TpmDeductionDetailMappingVo> findByConditions(Pageable pageable, TpmDeductionDetailMappingDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmDeductionDetailMappingDto();
        }
        String tenantCode = TenantUtils.getTenantCode();
        dto.setTenantCode(tenantCode);
        return this.tpmDeductionDetailMappingRepository.findByConditions(pageable, dto);
    }

    /**
     * 根据编码查询数据
     *
     * @param codeList codeList
     * @return
     */
    @Override
    public List<TpmDeductionDetailMappingVo> findByCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Lists.newArrayList();
        }
        List<String> distinctCodeList = codeList.stream().distinct().collect(Collectors.toList());
        String tenantCode = TenantUtils.getTenantCode();
        List<TpmDeductionDetailMappingCustomerVo> customerVoList = this.tpmDeductionDetailMappingCustomerRepository.findByCodes(distinctCodeList,tenantCode);
        Map<String, List<TpmDeductionDetailMappingCustomerVo>> customerMap = customerVoList.stream().collect(Collectors.groupingBy(TpmDeductionDetailMappingCustomerVo::getCode));
        List<TpmDeductionDetailMappingVo> detailMappingVoList = this.tpmDeductionDetailMappingRepository.findByCodes(distinctCodeList, tenantCode);
        List<TpmDeductionDetailMappingRelationActivityConfigVo> activityConfigVoList = deductionDetailMappingRelationActivityConfigRepository.findByCodes(distinctCodeList, tenantCode);
        List<TpmDeductionDetailMappingTextVo> textVoList = deductionDetailMappingTextRepository.findByCodes(distinctCodeList, tenantCode);
        Map<String, List<TpmDeductionDetailMappingRelationActivityConfigVo>> codeToActivityConfigVoList = activityConfigVoList.stream().collect(Collectors.groupingBy(TpmDeductionDetailMappingRelationActivityConfigVo::getCode));
        Map<String, List<TpmDeductionDetailMappingTextVo>> codeToTextVoList = textVoList.stream().collect(Collectors.groupingBy(TpmDeductionDetailMappingTextVo::getCode));
        detailMappingVoList.forEach(e -> {
            e.setDeductionDetailMappingRelationActivityConfigList(codeToActivityConfigVoList.getOrDefault(e.getCode(), Lists.newArrayList()));
            e.setDeductionDetailMappingTextList(codeToTextVoList.getOrDefault(e.getCode(), Lists.newArrayList()));
            e.setCustomerList(customerMap.getOrDefault(e.getCode(),Lists.newArrayList()));
        });
        return detailMappingVoList;
    }

    /**
     * 新增数据
     *
     * @param dto dto对象
     * @return 新增结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public TpmDeductionDetailMappingVo create(TpmDeductionDetailMappingDto dto) {
        // 根据配置校验必填字段、同纬度数据库验重
        this.createValidate(dto);
        if (!BusinessUnitEnum.SON_COMPANY.getCode().equals(dto.getBusinessUnitCode())) {
            //验重
            this.repeatValidate(Lists.newArrayList(dto));
        }
        //List<TpmDeductionDetailMappingDto> detailMappingDtos = this.repeatValidate(Lists.newArrayList(dto));
        //Validate.isTrue(detailMappingDtos.isEmpty(),"数据库已存在重复的扣费明细记录或者活动关联记录，请检查");

        String tenantCode = TenantUtils.getTenantCode();
        dto.setTenantCode(tenantCode);
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
//        String ruleCode = StringUtils.join(TpmDeductionDetailMappingConstant.DEDUCTION_DETAIL_MAPPING_FORMULA_PREFIX, DateFormatUtils.format(new Date(), "yyyyMMdd"));
        String code = this.generateCodeService.generateCode(TpmDeductionDetailMappingConstant.DEDUCTION_DETAIL_MAPPING_FORMULA_PREFIX, 1, 5, 2, TimeUnit.DAYS).get(0);
        dto.setCode(code);
        // 保存
        TpmDeductionDetailMapping entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, TpmDeductionDetailMapping.class, LinkedHashSet.class, ArrayList.class);
        dto.setId(entity.getId());
        this.tpmDeductionDetailMappingRepository.saveOrUpdate(entity);

        List<TpmDeductionDetailMappingRelationActivityConfig> configList = new ArrayList<>();
        Set<String> formCodes = dto.getDeductionDetailMappingRelationActivityConfigList().stream().map(TpmDeductionDetailMappingRelationActivityConfigDto::getActivityFormCode).collect(Collectors.toSet());
        Validate.isTrue(formCodes.size() == dto.getDeductionDetailMappingRelationActivityConfigList().size(),"活动形式重复！");
        for (TpmDeductionDetailMappingRelationActivityConfigDto tpmDeductionDetailMappingRelationActivityConfigDto : dto.getDeductionDetailMappingRelationActivityConfigList()) {
            TpmDeductionDetailMappingRelationActivityConfig config = new TpmDeductionDetailMappingRelationActivityConfig();
            config.setCode(code);
            config.setTenantCode(tenantCode);
            config.setActivityTypeCode(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityTypeCode());
            config.setActivityTypeName(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityTypeName());
            config.setActivityFormCode(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityFormCode());
            config.setActivityFormName(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityFormName());
            configList.add(config);
        }
        this.deductionDetailMappingRelationActivityConfigRepository.saveBatch(configList);
        List<TpmDeductionDetailMappingText> textList = new ArrayList<>();
        for (TpmDeductionDetailMappingTextDto textDto : dto.getDeductionDetailMappingTextList()) {
            TpmDeductionDetailMappingText text = new TpmDeductionDetailMappingText();
            text.setCode(code);
            text.setText(textDto.getText());
            text.setTenantCode(tenantCode);
            textList.add(text);
        }
        this.deductionDetailMappingTextRepository.saveBatch(textList);

        List<TpmDeductionDetailMappingCustomer> customerList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(dto.getCustomerList())) {
            for (TpmDeductionDetailMappingCustomerDto customerDto : dto.getCustomerList()) {
                TpmDeductionDetailMappingCustomer customer = new TpmDeductionDetailMappingCustomer();
                customer.setCode(code);
                customer.setCustomerCode(customerDto.getCustomerCode());
                customer.setCustomerName(customerDto.getCustomerName());
                customer.setErpCode(customerDto.getErpCode());
                customer.setTenantCode(TenantUtils.getTenantCode());
                customer.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                customerList.add(customer);
            }
        }
        if (CollectionUtils.isNotEmpty(customerList)) {
            tpmDeductionDetailMappingCustomerRepository.saveBatch(customerList);
        }
        // 业务日志新增
        TpmDeductionDetailMappingLogEventDto logEventDto = new TpmDeductionDetailMappingLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<TpmDeductionDetailMappingLogEventListener, TpmDeductionDetailMappingLogEventDto> onCreate =
                TpmDeductionDetailMappingLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, TpmDeductionDetailMappingLogEventListener.class, onCreate);
        return null;
    }

    /**
     * 修改新据
     *
     * @param dto dto对象
     * @return 修改结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public TpmDeductionDetailMappingVo update(TpmDeductionDetailMappingDto dto) {
        // 根据配置校验必填字段、同纬度数据库验重
        this.updateValidate(dto);
        TpmDeductionDetailMapping oldEntity = tpmDeductionDetailMappingRepository.getById(dto.getId());
        if (ObjectUtils.isEmpty(oldEntity)) {
            throw new RuntimeException("数据不存在，请刷新后重试！");
        }
        if (!BusinessUnitEnum.SON_COMPANY.getCode().equals(dto.getBusinessUnitCode())) {
            //验重
            this.repeatValidate(Lists.newArrayList(dto));
        }
        //List<TpmDeductionDetailMappingDto> detailMappingDtos = this.repeatValidate(Lists.newArrayList(dto));
        //Validate.isTrue(detailMappingDtos.isEmpty(),"数据库已存在重复的扣费明细记录或者活动关联记录，请检查");

        String tenantCode = TenantUtils.getTenantCode();
        // 更新
        TpmDeductionDetailMapping entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, TpmDeductionDetailMapping.class, LinkedHashSet.class, ArrayList.class);
        if (StringUtils.isBlank(entity.getSalesInstitutionCode())) {
            entity.setSalesInstitutionCode("");
            entity.setSalesInstitutionErpCode("");
            entity.setSalesInstitutionName("");
        }
        if (StringUtils.isBlank(entity.getResaleCommercialCode())) {
            entity.setResaleCommercialCode("");
            entity.setResaleCommercialName("");
        }
        this.tpmDeductionDetailMappingRepository.saveOrUpdate(entity);

        deductionDetailMappingRelationActivityConfigRepository.remove(Wrappers.lambdaUpdate(TpmDeductionDetailMappingRelationActivityConfig.class)
                .eq(TpmDeductionDetailMappingRelationActivityConfig::getCode, oldEntity.getCode())
                .eq(TpmDeductionDetailMappingRelationActivityConfig::getTenantCode, tenantCode));
        deductionDetailMappingTextRepository.remove(Wrappers.lambdaUpdate(TpmDeductionDetailMappingText.class)
                .eq(TpmDeductionDetailMappingText::getCode, oldEntity.getCode())
                .eq(TpmDeductionDetailMappingText::getTenantCode, tenantCode));
        List<TpmDeductionDetailMappingRelationActivityConfig> configList = new ArrayList<>();
        for (TpmDeductionDetailMappingRelationActivityConfigDto tpmDeductionDetailMappingRelationActivityConfigDto : dto.getDeductionDetailMappingRelationActivityConfigList()) {
            TpmDeductionDetailMappingRelationActivityConfig config = new TpmDeductionDetailMappingRelationActivityConfig();
            config.setCode(oldEntity.getCode());
            config.setTenantCode(tenantCode);
            config.setActivityTypeCode(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityTypeCode());
            config.setActivityTypeName(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityTypeName());
            config.setActivityFormCode(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityFormCode());
            config.setActivityFormName(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityFormName());
            configList.add(config);
        }
        this.deductionDetailMappingRelationActivityConfigRepository.saveBatch(configList);
        List<TpmDeductionDetailMappingText> textList = new ArrayList<>();
        for (TpmDeductionDetailMappingTextDto textDto : dto.getDeductionDetailMappingTextList()) {
            TpmDeductionDetailMappingText text = new TpmDeductionDetailMappingText();
            text.setCode(oldEntity.getCode());
            text.setText(textDto.getText());
            text.setTenantCode(tenantCode);
            textList.add(text);
        }
        this.deductionDetailMappingTextRepository.saveBatch(textList);

        tpmDeductionDetailMappingCustomerRepository.remove(Wrappers.lambdaUpdate(TpmDeductionDetailMappingCustomer.class)
                .eq(TpmDeductionDetailMappingCustomer::getCode, oldEntity.getCode())
                .eq(TpmDeductionDetailMappingCustomer::getTenantCode, tenantCode));
        List<TpmDeductionDetailMappingCustomer> customerList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(dto.getCustomerList())) {
            for (TpmDeductionDetailMappingCustomerDto customerDto : dto.getCustomerList()) {
                TpmDeductionDetailMappingCustomer customer = new TpmDeductionDetailMappingCustomer();
                customer.setCode(oldEntity.getCode());
                customer.setCustomerCode(customerDto.getCustomerCode());
                customer.setCustomerName(customerDto.getCustomerName());
                customer.setErpCode(customerDto.getErpCode());
                customer.setTenantCode(TenantUtils.getTenantCode());
                customer.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                customerList.add(customer);
            }
        }
        if (CollectionUtils.isNotEmpty(customerList)) {
            tpmDeductionDetailMappingCustomerRepository.saveBatch(customerList);
        }

        //通知更新匹配模板
        SerializableBiConsumer<TpmDeductionDetailMappingEvent, TpmDeductionDetailMappingEventDto> update =
                TpmDeductionDetailMappingEvent::onUpdate;
        TpmDeductionDetailMappingEventDto eventDto = new TpmDeductionDetailMappingEventDto();
        eventDto.setCode(entity.getCode());
        eventDto.setSalesInstitutionCode(entity.getSalesInstitutionCode());
        eventDto.setSalesInstitutionErpCode(entity.getSalesInstitutionErpCode());
        eventDto.setSalesInstitutionName(entity.getSalesInstitutionName());
        eventDto.setResaleCommercialCode(entity.getResaleCommercialCode());
        eventDto.setResaleCommercialName(entity.getResaleCommercialName());
        this.nebulaNetEventClient.publish(eventDto, TpmDeductionDetailMappingEvent.class, update);


        //日志
        TpmDeductionDetailMappingLogEventDto logEventDto = new TpmDeductionDetailMappingLogEventDto();
        TpmDeductionDetailMappingDto old = nebulaToolkitService.copyObjectByWhiteList(oldEntity, TpmDeductionDetailMappingDto.class, LinkedHashSet.class, ArrayList.class);
        logEventDto.setOriginal(old);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<TpmDeductionDetailMappingLogEventListener, TpmDeductionDetailMappingLogEventDto> onUpdate =
                TpmDeductionDetailMappingLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, TpmDeductionDetailMappingLogEventListener.class, onUpdate);
        return this.nebulaToolkitService.copyObjectByWhiteList(dto, TpmDeductionDetailMappingVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     * @return 删除结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        String tenantCode = TenantUtils.getTenantCode();
        List<TpmDeductionDetailMapping> entities = this.tpmDeductionDetailMappingRepository.listByIds(idList).stream()
                .filter(e -> tenantCode.equals(e.getTenantCode())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(entities)) {
            entities.forEach(entity -> {
                entity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            });
            this.tpmDeductionDetailMappingRepository.updateBatchById(entities);
        }
    }

    /**
     * 批量新增
     *
     * @param importList 对象集合
     * @return 删除结果
     */
    @Override
    public void importSave(List<TpmDeductionDetailMappingDto> importList) {
        if (CollectionUtils.isEmpty(importList)) {
            return;
        }
        log.info("扣费映射导入保存开始");
        List<TpmDeductionDetailMapping> entityList = new ArrayList<>();
        List<TpmDeductionDetailMappingRelationActivityConfig> configList = new ArrayList<>();
        List<TpmDeductionDetailMappingText> textList = new ArrayList<>();
        List<TpmDeductionDetailMappingCustomer> customerList = new ArrayList<>();


        String tenantCode = TenantUtils.getTenantCode();
        List<String> codes = this.generateCodeService.generateCode(TpmDeductionDetailMappingConstant.DEDUCTION_DETAIL_MAPPING_FORMULA_PREFIX, importList.size(), 5, 2, TimeUnit.DAYS);
        int i = 0;
        for (TpmDeductionDetailMappingDto dto : importList) {
            this.createValidate(dto);

            dto.setTenantCode(tenantCode);
            String code = codes.get(i);
            Validate.notBlank(code,"扣费映射导入保存时，生成映射编码失败");
            dto.setCode(code);
            i++;

            TpmDeductionDetailMapping entity = nebulaToolkitService.copyObjectByWhiteList(dto, TpmDeductionDetailMapping.class, HashSet.class, ArrayList.class);
            entityList.add(entity);

            Set<String> formCodes = dto.getDeductionDetailMappingRelationActivityConfigList().stream().map(TpmDeductionDetailMappingRelationActivityConfigDto::getActivityFormCode).collect(Collectors.toSet());
            Validate.isTrue(formCodes.size() == dto.getDeductionDetailMappingRelationActivityConfigList().size(),"活动形式重复！");
            for (TpmDeductionDetailMappingRelationActivityConfigDto tpmDeductionDetailMappingRelationActivityConfigDto : dto.getDeductionDetailMappingRelationActivityConfigList()) {
                TpmDeductionDetailMappingRelationActivityConfig config = new TpmDeductionDetailMappingRelationActivityConfig();
                config.setCode(code);
                config.setTenantCode(tenantCode);
                config.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                config.setActivityTypeCode(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityTypeCode());
                config.setActivityTypeName(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityTypeName());
                config.setActivityFormCode(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityFormCode());
                config.setActivityFormName(tpmDeductionDetailMappingRelationActivityConfigDto.getActivityFormName());
                configList.add(config);
            }
            for (TpmDeductionDetailMappingTextDto textDto : dto.getDeductionDetailMappingTextList()) {
                TpmDeductionDetailMappingText text = new TpmDeductionDetailMappingText();
                text.setCode(code);
                text.setText(textDto.getText());
                text.setTenantCode(tenantCode);
                text.setTenantCode(tenantCode);
                text.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                textList.add(text);
            }
            if (CollectionUtils.isNotEmpty(dto.getCustomerList())) {
                for (TpmDeductionDetailMappingCustomerDto customerDto : dto.getCustomerList()) {
                    TpmDeductionDetailMappingCustomer customer = new TpmDeductionDetailMappingCustomer();
                    customer.setCode(code);
                    customer.setCustomerCode(customerDto.getCustomerCode());
                    customer.setCustomerName(customerDto.getCustomerName());
                    customer.setErpCode(customerDto.getErpCode());
                    customer.setTenantCode(TenantUtils.getTenantCode());
                    customer.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    customerList.add(customer);
                }
            }
        }

        this.importSaveData(entityList,configList,textList,customerList);
        log.info("扣费映射导入保存结束");
    }

    @Transactional(rollbackFor = Exception.class)
    public void importSaveData(List<TpmDeductionDetailMapping> entityList, List<TpmDeductionDetailMappingRelationActivityConfig> configList, List<TpmDeductionDetailMappingText> textList, List<TpmDeductionDetailMappingCustomer> customerList) {
        if (CollectionUtils.isNotEmpty(entityList)) {
            this.tpmDeductionDetailMappingRepository.saveBatch(entityList);
        }
        if (CollectionUtils.isNotEmpty(configList)) {
            this.deductionDetailMappingRelationActivityConfigRepository.saveBatch(configList);
        }
        if (CollectionUtils.isNotEmpty(textList)) {
            this.deductionDetailMappingTextRepository.saveBatch(textList);
        }
        if (CollectionUtils.isNotEmpty(customerList)) {
            tpmDeductionDetailMappingCustomerRepository.saveBatch(customerList);
        }
    }

    /**
     * 根据id查询
     *
     * @param id id
     * @return 删除结果
     */
    @Override
    public TpmDeductionDetailMappingVo queryById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        TpmDeductionDetailMapping entity = tpmDeductionDetailMappingRepository.getById(id);
        Validate.notNull(entity, "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        List<TpmDeductionDetailMappingRelationActivityConfig> configList = this.deductionDetailMappingRelationActivityConfigRepository.lambdaQuery()
                .eq(TpmDeductionDetailMappingRelationActivityConfig::getCode, entity.getCode())
                .eq(TpmDeductionDetailMappingRelationActivityConfig::getTenantCode, tenantCode)
                .list();

        TpmDeductionDetailMappingVo tpmDeductionDetailMappingVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmDeductionDetailMappingVo.class, LinkedHashSet.class, ArrayList.class);

        if (CollectionUtils.isNotEmpty(configList)) {
            Collection<TpmDeductionDetailMappingRelationActivityConfigVo> tpmDeductionDetailMappingRelationActivityConfigVos = this.nebulaToolkitService.copyCollectionByBlankList(configList, TpmDeductionDetailMappingRelationActivityConfig.class, TpmDeductionDetailMappingRelationActivityConfigVo.class, LinkedHashSet.class, ArrayList.class);
            tpmDeductionDetailMappingVo.setDeductionDetailMappingRelationActivityConfigList((List<TpmDeductionDetailMappingRelationActivityConfigVo>) tpmDeductionDetailMappingRelationActivityConfigVos);
        }

        List<TpmDeductionDetailMappingText> textList = this.deductionDetailMappingTextRepository.lambdaQuery()
                .eq(TpmDeductionDetailMappingText::getCode, entity.getCode())
                .eq(TpmDeductionDetailMappingText::getTenantCode, tenantCode)
                .list();
        if (CollectionUtils.isNotEmpty(textList)) {
            Collection<TpmDeductionDetailMappingTextVo> tpmDeductionDetailMappingTextVos = this.nebulaToolkitService.copyCollectionByBlankList(textList, TpmDeductionDetailMappingText.class, TpmDeductionDetailMappingTextVo.class, LinkedHashSet.class, ArrayList.class);
            tpmDeductionDetailMappingVo.setDeductionDetailMappingTextList((List<TpmDeductionDetailMappingTextVo>) tpmDeductionDetailMappingTextVos);
        }
        List<TpmDeductionDetailMappingCustomer> customerList = this.tpmDeductionDetailMappingCustomerRepository.lambdaQuery()
                .eq(TpmDeductionDetailMappingCustomer::getCode, entity.getCode())
                .eq(TpmDeductionDetailMappingCustomer::getTenantCode, tenantCode)
                .list();
        if (CollectionUtils.isNotEmpty(customerList)) {
            List<TpmDeductionDetailMappingCustomerVo> customerVoList = (List<TpmDeductionDetailMappingCustomerVo>) nebulaToolkitService.copyCollectionByWhiteList(customerList, TpmDeductionDetailMappingCustomer.class, TpmDeductionDetailMappingCustomerVo.class, HashSet.class, ArrayList.class);
            tpmDeductionDetailMappingVo.setCustomerList(customerVoList);
        }

        return tpmDeductionDetailMappingVo;
    }

    /**
     * 验重
     *
     * 一条扣费明细项+业务单元+销售组织+零售商只能找到一个映射，一条活动形式+业务单元+销售组织+零售商只能找到一个映射
     *
     * @param dtoList 条件
     * @return 查询结果
     */
    @Override
    public List<TpmDeductionDetailMappingDto> repeatValidate(List<TpmDeductionDetailMappingDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)){
            return Lists.newArrayList();
        }

        ArrayList<TpmDeductionDetailMappingDto> repeatList = new ArrayList<>();

        for (TpmDeductionDetailMappingDto dto : dtoList) {
            List<TpmDeductionDetailMappingTextDto> textList = dto.getDeductionDetailMappingTextList();
            List<TpmDeductionDetailMappingRelationActivityConfigDto> configList = dto.getDeductionDetailMappingRelationActivityConfigList();

            List<TpmDeductionDetailMapping> mappingList = this.tpmDeductionDetailMappingRepository.lambdaQuery()
                    .eq(TpmDeductionDetailMapping::getTenantCode,TenantUtils.getTenantCode())
                    .eq(StringUtils.isNotBlank(dto.getBusinessFormatCode()),TpmDeductionDetailMapping::getBusinessFormatCode,dto.getBusinessFormatCode())
                    .eq(StringUtils.isNotBlank(dto.getBusinessUnitCode()), TpmDeductionDetailMapping::getBusinessUnitCode, dto.getBusinessUnitCode())
                    .eq(StringUtils.isNotBlank(dto.getSalesInstitutionCode()), TpmDeductionDetailMapping::getSalesInstitutionCode, dto.getSalesInstitutionCode())
                    .eq(StringUtils.isNotBlank(dto.getResaleCommercialCode()), TpmDeductionDetailMapping::getResaleCommercialCode, dto.getResaleCommercialCode())
                    .eq(StringUtils.isNotBlank(dto.getFirstChannelCode()),TpmDeductionDetailMapping::getFirstChannelCode,dto.getFirstChannelCode())
                    .eq(StringUtils.isNotBlank(dto.getSecondChannelCode()),TpmDeductionDetailMapping::getSecondChannelCode,dto.getSecondChannelCode())
                    .eq(StringUtils.isNotBlank(dto.getChannelCode()),TpmDeductionDetailMapping::getChannelCode,dto.getChannelCode())
                    .eq(StringUtils.isNotBlank(dto.getBuyWay()),TpmDeductionDetailMapping::getBuyWay,dto.getBuyWay())
                    .eq(TpmDeductionDetailMapping::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                    .ne(StringUtils.isNotBlank(dto.getId()),TpmDeductionDetailMapping::getId,dto.getId())
                    .list();

            if (CollectionUtils.isEmpty(mappingList)){
                continue;
            }

            List<String> codeList = mappingList.stream().map(o -> o.getCode()).collect(Collectors.toList());

            List<TpmDeductionDetailMappingCustomerDto> customerList = dto.getCustomerList();
            if (CollectionUtils.isNotEmpty(customerList)) {
                //客户
                List<TpmDeductionDetailMappingCustomerVo> existsCustomerVoList = tpmDeductionDetailMappingCustomerRepository.findByCodes(codeList, TenantUtils.getTenantCode());
                //
                existsCustomerVoList = existsCustomerVoList.stream().filter(s -> StringUtils.isNotBlank(s.getCustomerCode()) && StringUtils.isNotBlank(s.getCode())).collect(Collectors.toList());

                List<String> existsCustomerCodes = existsCustomerVoList.stream().map(TpmDeductionDetailMappingCustomerVo::getCustomerCode).collect(Collectors.toList());
                for (TpmDeductionDetailMappingCustomerDto customerDto : customerList) {
                    if (existsCustomerCodes.contains(customerDto.getCustomerCode())) {

                        //扣费明细项
                        List<TpmDeductionDetailMappingTextVo> existsTextVos = this.deductionDetailMappingTextRepository.findByCodes(codeList,TenantUtils.getTenantCode());
                        existsTextVos = existsTextVos.stream().filter(h -> StringUtils.isNotBlank(h.getText()) && StringUtils.isNotBlank(h.getCode())).collect(Collectors.toList());
                        Map<String, TpmDeductionDetailMappingTextVo> existsTextVoMap = existsTextVos.stream().collect(Collectors.toMap(TpmDeductionDetailMappingTextVo::getText, Function.identity(), (v1, v2) -> v2));
                        List<String> textStringList = existsTextVos.stream().map(o -> o.getText()).collect(Collectors.toList());
                        for (TpmDeductionDetailMappingTextDto textDto : textList) {
                            if (textStringList.contains(textDto.getText())){
                                repeatList.add(dto);
                                TpmDeductionDetailMappingTextVo tpmDeductionDetailMappingTextVo = existsTextVoMap.get(textDto.getText());
                                throw new IllegalArgumentException("在映射【"+ tpmDeductionDetailMappingTextVo.getCode() +"】下，已存在’扣费明细项名称’配置【"+textDto.getText()+"】");
                            }
                        }

                        //活动形式
                        List<TpmDeductionDetailMappingRelationActivityConfigVo> configVos = this.deductionDetailMappingRelationActivityConfigRepository.findByCodes(codeList, TenantUtils.getTenantCode());
                        configVos = configVos.stream().filter(s -> StringUtils.isNotBlank(s.getActivityFormCode()) && StringUtils.isNotBlank(s.getCode())).collect(Collectors.toList());
                        Map<String, TpmDeductionDetailMappingRelationActivityConfigVo> configVoMap = configVos.stream().collect(Collectors.toMap(TpmDeductionDetailMappingRelationActivityConfigVo::getActivityFormCode, Function.identity(), (v1, v2) -> v2));
                        List<String> formCodes = configVos.stream().map(o -> o.getActivityFormCode()).collect(Collectors.toList());
                        for (TpmDeductionDetailMappingRelationActivityConfigDto configDto : configList) {
                            if (formCodes.contains(configDto.getActivityFormCode())){
                                repeatList.add(dto);
                                TpmDeductionDetailMappingRelationActivityConfigVo con = configVoMap.get(configDto.getActivityFormCode());
                                throw new IllegalArgumentException("在映射【"+ con.getCode() +"】下，已存在‘活动形式’配置【"+configDto.getActivityFormCode()+"】配置");
                            }
                        }

                    }
                }
            } else {

                //扣费明细项
                List<TpmDeductionDetailMappingTextVo> existsTextVos = this.deductionDetailMappingTextRepository.findByCodes(codeList,TenantUtils.getTenantCode());
                existsTextVos = existsTextVos.stream().filter(h -> StringUtils.isNotBlank(h.getText()) && StringUtils.isNotBlank(h.getCode())).collect(Collectors.toList());
                Map<String, TpmDeductionDetailMappingTextVo> existsTextVoMap = existsTextVos.stream().collect(Collectors.toMap(TpmDeductionDetailMappingTextVo::getText, Function.identity(), (v1, v2) -> v2));
                List<String> textStringList = existsTextVos.stream().map(o -> o.getText()).collect(Collectors.toList());
                for (TpmDeductionDetailMappingTextDto textDto : textList) {
                    if (textStringList.contains(textDto.getText())){
                        repeatList.add(dto);
                        TpmDeductionDetailMappingTextVo tpmDeductionDetailMappingTextVo = existsTextVoMap.get(textDto.getText());
                        throw new IllegalArgumentException("在映射【"+ tpmDeductionDetailMappingTextVo.getCode() +"】下，已存在’扣费明细项名称’配置【"+textDto.getText()+"】");
                    }
                }

                //活动形式
                List<TpmDeductionDetailMappingRelationActivityConfigVo> configVos = this.deductionDetailMappingRelationActivityConfigRepository.findByCodes(codeList, TenantUtils.getTenantCode());
                configVos = configVos.stream().filter(s -> StringUtils.isNotBlank(s.getActivityFormCode()) && StringUtils.isNotBlank(s.getCode())).collect(Collectors.toList());
                Map<String, TpmDeductionDetailMappingRelationActivityConfigVo> configVoMap = configVos.stream().collect(Collectors.toMap(TpmDeductionDetailMappingRelationActivityConfigVo::getActivityFormCode, Function.identity(), (v1, v2) -> v2));
                List<String> formCodes = configVos.stream().map(o -> o.getActivityFormCode()).collect(Collectors.toList());
                for (TpmDeductionDetailMappingRelationActivityConfigDto configDto : configList) {
                    if (formCodes.contains(configDto.getActivityFormCode())){
                        repeatList.add(dto);
                        TpmDeductionDetailMappingRelationActivityConfigVo con = configVoMap.get(configDto.getActivityFormCode());
                        throw new IllegalArgumentException("在映射【"+ con.getCode() +"】下，已存在‘活动形式’配置【"+configDto.getActivityFormCode()+"】配置");
                    }
                }
            }

        }
        return repeatList;
    }

    @Override
    public TpmDeductionDetailMappingVo findByCode(String applyMappingCode) {
        TpmDeductionDetailMapping entity = tpmDeductionDetailMappingRepository.findByCode(applyMappingCode);
        String tenantCode = TenantUtils.getTenantCode();
        List<TpmDeductionDetailMappingRelationActivityConfig> configList = this.deductionDetailMappingRelationActivityConfigRepository.lambdaQuery()
                .eq(TpmDeductionDetailMappingRelationActivityConfig::getCode, entity.getCode())
                .eq(TpmDeductionDetailMappingRelationActivityConfig::getTenantCode, tenantCode)
                .list();

        TpmDeductionDetailMappingVo tpmDeductionDetailMappingVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmDeductionDetailMappingVo.class, LinkedHashSet.class, ArrayList.class);

        if (CollectionUtils.isNotEmpty(configList)) {
            Collection<TpmDeductionDetailMappingRelationActivityConfigVo> tpmDeductionDetailMappingRelationActivityConfigVos = this.nebulaToolkitService.copyCollectionByBlankList(configList, TpmDeductionDetailMappingRelationActivityConfig.class, TpmDeductionDetailMappingRelationActivityConfigVo.class, LinkedHashSet.class, ArrayList.class);
            tpmDeductionDetailMappingVo.setDeductionDetailMappingRelationActivityConfigList((List<TpmDeductionDetailMappingRelationActivityConfigVo>) tpmDeductionDetailMappingRelationActivityConfigVos);
        }

        List<TpmDeductionDetailMappingText> textList = this.deductionDetailMappingTextRepository.lambdaQuery()
                .eq(TpmDeductionDetailMappingText::getCode, entity.getCode())
                .eq(TpmDeductionDetailMappingText::getTenantCode, tenantCode)
                .list();
        if (CollectionUtils.isNotEmpty(textList)) {
            Collection<TpmDeductionDetailMappingTextVo> tpmDeductionDetailMappingTextVos = this.nebulaToolkitService.copyCollectionByBlankList(textList, TpmDeductionDetailMappingText.class, TpmDeductionDetailMappingTextVo.class, LinkedHashSet.class, ArrayList.class);
            tpmDeductionDetailMappingVo.setDeductionDetailMappingTextList((List<TpmDeductionDetailMappingTextVo>) tpmDeductionDetailMappingTextVos);
        }
        return tpmDeductionDetailMappingVo;
    }

    /**
     * 根据Dto查询全部
     * @param dto
     * @return
     */
    @Override
    public List<TpmDeductionDetailMappingVo> findByDto(TpmDeductionDetailMappingDto dto) {
        if (Objects.isNull(dto)) {
            dto = new TpmDeductionDetailMappingDto();
        }
        List<TpmDeductionDetailMapping> mappingList = this.tpmDeductionDetailMappingRepository.findByDto(dto);
        if (CollectionUtils.isEmpty(mappingList)) {
            return Lists.newArrayList();
        }
        return (List<TpmDeductionDetailMappingVo>) this.nebulaToolkitService.copyCollectionByWhiteList(mappingList,TpmDeductionDetailMapping.class,TpmDeductionDetailMappingVo.class,HashSet.class,ArrayList.class);
    }


    /**
     * 创建验证
     *
     * @param dto
     */
    private void createValidate(TpmDeductionDetailMappingDto dto) {
        Validate.notNull(dto, "新增时，对象信息不能为空！");
        dto.setId(null);
        Validate.notBlank(dto.getName(),"扣费映射名称不能为空！");
        Validate.isTrue(dto.getName().length() <= 255,"扣费映射名称超长，最长支持255");
        Validate.notBlank(dto.getBusinessFormatCode(), "新增数据时，业态不能为空！");
        Validate.notBlank(dto.getBusinessUnitCode(), "新增数据时，业务单元不能为空！");
//        Validate.notBlank(dto.getResaleCommercialCode(), "新增数据时，直营体系编码不能为空！");
        //Validate.notBlank(dto.getSalesInstitutionCode(), "新增数据时，销售机构编码不能为空！");
        List<TpmDeductionDetailMappingTextDto> deductionDetailMappingTextList = dto.getDeductionDetailMappingTextList();
        Validate.noNullElements(deductionDetailMappingTextList, "新增数据时，扣费明细项不能为空！");
        deductionDetailMappingTextList.forEach(e -> {
            Validate.notBlank(e.getText(), "新增数据时，扣费明细项不能为空！");
            Validate.isTrue(e.getText().length() <= 255,"扣费明细项名称超长，最长支持255");
        });
        List<TpmDeductionDetailMappingRelationActivityConfigDto> deductionDetailMappingRelationActivityConfigList = dto.getDeductionDetailMappingRelationActivityConfigList();
        Validate.noNullElements(deductionDetailMappingRelationActivityConfigList, "新增数据时，关联活动配置不能为空！");
        //活动形式不能选重复
        Set<String> activityFormCode = Sets.newHashSet();
        deductionDetailMappingRelationActivityConfigList.forEach(e -> {
            Validate.notBlank(e.getActivityFormCode(), "新增数据时，活动形式不能为空！");
            Validate.notBlank(e.getActivityTypeCode(), "新增数据时，活动类型不能为空！");
            Validate.isTrue(!activityFormCode.contains(e.getActivityFormCode()),"关联活动形式不能重复选择");
            activityFormCode.add(e.getActivityFormCode());
        });
    }

    /**
     * 修改验证
     *
     * @param dto
     */
    private void updateValidate(TpmDeductionDetailMappingDto dto) {
        Validate.notNull(dto, "修改时，对象信息不能为空！");
        Validate.notBlank(dto.getId(), "修改数据时，ID不能为空！");
        Validate.notBlank(dto.getBusinessFormatCode(), "修改数据时，业态不能为空！");
        Validate.notBlank(dto.getBusinessUnitCode(), "修改数据时，业务单元不能为空！");
        Validate.notBlank(dto.getCode(), "修改数据时，运费价格维护编码不能为空！");
//        Validate.notBlank(dto.getResaleCommercialCode(), "修改数据时，零售商编码不能为空！");
        //Validate.notBlank(dto.getSalesInstitutionCode(), "修改数据时，销售机构编码不能为空！");
        List<TpmDeductionDetailMappingTextDto> deductionDetailMappingTextList = dto.getDeductionDetailMappingTextList();
        Validate.noNullElements(deductionDetailMappingTextList, "修改数据时，扣费明细项不能为空！");
        deductionDetailMappingTextList.forEach(e -> Validate.notBlank(e.getText(), "修改数据时，扣费明细项不能为空！"));
        List<TpmDeductionDetailMappingRelationActivityConfigDto> deductionDetailMappingRelationActivityConfigList = dto.getDeductionDetailMappingRelationActivityConfigList();
        Validate.noNullElements(deductionDetailMappingRelationActivityConfigList, "修改数据时，关联活动配置不能为空！");
        deductionDetailMappingTextList.forEach(e -> {
            Validate.notBlank(e.getText(), "新增数据时，扣费明细项不能为空！");
            Validate.isTrue(e.getText().length() <= 255,"扣费明细项名称超长，最长支持255");
        });
        //活动形式不能选重复
        Set<String> activityFormCode = Sets.newHashSet();
        log.info("商超扣费映射-关联活动形式,{}",JSON.toJSONString(deductionDetailMappingRelationActivityConfigList));
        deductionDetailMappingRelationActivityConfigList.forEach(e -> {
            log.info("商超扣费映射-关联活动形式SET,{}",JSON.toJSONString(activityFormCode));
            Validate.notBlank(e.getActivityFormCode(), "修改数据时，活动形式不能为空！");
            Validate.notBlank(e.getActivityTypeCode(), "修改数据时，活动类型不能为空！");
            Validate.isTrue(!activityFormCode.contains(e.getActivityFormCode()),"关联活动形式不能重复选择");
            activityFormCode.add(e.getActivityFormCode());
        });
    }

}

