package com.biz.crm.tpm.business.event.prepayment.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.third.system.sd.sdk.dto.PaymentReceiptQueryStatusDto;
import com.biz.crm.tpm.business.event.prepayment.local.constant.EventPrepaymentConstant;
import com.biz.crm.tpm.business.event.prepayment.sdk.dto.*;
import com.biz.crm.tpm.business.event.prepayment.sdk.service.EventPrepaymentService;
import com.biz.crm.tpm.business.event.prepayment.sdk.vo.CustomerPrepaymentVo;
import com.biz.crm.tpm.business.event.prepayment.sdk.vo.EventPrepaymentVo;
import com.biz.crm.tpm.business.event.prepayment.sdk.vo.MaterialInformationVo;
import com.biz.crm.tpm.business.event.prepayment.sdk.vo.PrepaidBySupplierVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.GeneralExpensesVo;
import com.biz.crm.mn.common.base.service.RedisLockService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author youjun
 * @date 2022年11月15日 18:57
 */
@RestController
@RequestMapping("/v1/event/prepayment")
@Slf4j
@Api(tags = "活动预付管理")
public class EventPrepaymentController {

    /**
     * 服务对象
     */
    @Autowired(required = false)
    private EventPrepaymentService eventPrepaymentService;
    /**
     * redis服务
     */
    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param eventPrepaymentDto 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByPrepayments")
    public Result<Page<EventPrepaymentVo>> findByPrepayments(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                               @ApiParam(name = "eventPrepaymentDto", value = "活动预付") EventPrepaymentDto eventPrepaymentDto) {
        try {
            Page<EventPrepaymentVo> page =  this.eventPrepaymentService.findByPrepayments(pageable,eventPrepaymentDto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 新增活动预付
     * @param eventPrepaymentDto 实体对象
     * @return 新增结果
     */
    @ApiOperation(value = "新增数据")
    @PostMapping("create")
    public Result<EventPrepaymentVo> create(@ApiParam(name = "eventPrepaymentDto", value = "活动预付管理") @RequestBody EventPrepaymentDto eventPrepaymentDto) {
        try {
            EventPrepaymentVo eventPrepaymentVo = this.eventPrepaymentService.create(eventPrepaymentDto);
            return Result.ok(eventPrepaymentVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
    /**
     * 删除数据
     *
     * @param ids 主键结合
     * @return 删除结果
     */
    @ApiOperation(value = "删除数据")
    @DeleteMapping("delete")
    public Result<?> delete(@ApiParam(name = "ids", value = "主键集合") @RequestParam("ids") List<String> ids) {
        try {
            this.eventPrepaymentService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
    /**
     * 修改数据
     *
     * @param eventPrepaymentDto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "修改数据")
    @PatchMapping
    public Result<EventPrepaymentVo> update(@ApiParam(name = "eventPrepaymentDto", value = "活动预付") @RequestBody EventPrepaymentDto eventPrepaymentDto) {
        try {
            EventPrepaymentVo update = this.eventPrepaymentService.update(eventPrepaymentDto);
            return Result.ok(update);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
    /**
     * 通过主键查询单条数据
     * @param prepaidCoding 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("{prepaidCoding}")
    public Result<EventPrepaymentVo> findById(@PathVariable @ApiParam(name = "prepaidCoding", value = "预付编码", required = true) String prepaidCoding) {
        try {
            EventPrepaymentVo eventPrepaymentVo = this.eventPrepaymentService.findByPrepaidCoding(prepaidCoding);
            return Result.ok(eventPrepaymentVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
    /**
     * 提交审批
     * @param eventPrepaymentSubmitDto
     * @return
     */
    @PostMapping("submit")
    @ApiOperation("提交审批")
    public Result<?> submit(@RequestBody EventPrepaymentSubmitDto eventPrepaymentSubmitDto){
        String lockKey = "";
        boolean lockSuccess = false;
        try {
            Validate.notNull(eventPrepaymentSubmitDto.getId(), "请选择数据！");
            //提交审批前加锁一个小时
            lockKey = EventPrepaymentConstant.LOCK_EVENT_PREPAYMENT_APPROVE + eventPrepaymentSubmitDto.getId();
            lockSuccess = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 3);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            eventPrepaymentService.submit(eventPrepaymentSubmitDto);
            return Result.ok();
        } catch (Exception e){
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess){
                redisLockService.unlock(lockKey);
            }
        }
    }

    /**
     * 保存并提交审批
     * @param eventPrepaymentSubmitDto
     * @return
     */
    @PostMapping("submitAndSetPass")
    @ApiOperation("保存并提交审批")
    public Result<?> submitAndSetPass(@RequestBody EventPrepaymentSubmitDto eventPrepaymentSubmitDto){
        String lockKey = "";
        boolean lockSuccess = false;
        try {
            if (StringUtils.isNotBlank(eventPrepaymentSubmitDto.getId())) {
                //提交审批前加锁一个小时
                lockKey = EventPrepaymentConstant.LOCK_EVENT_PREPAYMENT_APPROVE + eventPrepaymentSubmitDto.getId();
                lockSuccess = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 3);
                Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            }
            eventPrepaymentService.submitAndSetPass(eventPrepaymentSubmitDto);
            return Result.ok();
        } catch (Exception e){
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess){
                redisLockService.unlock(lockKey);
            }
        }
    }

    /**
     * 获取促销规划数据
     * @param
     * @return
     */
    @PostMapping("promotionPlan")
    @ApiOperation(value = "获取促销规划数据")
    public Result<?> promotionPlan(@RequestBody List<GeneralExpensesVo> promotionPlanDtos){
        try {
            List<PrepaidBySupplierVo> prepaidBySupplierVos=eventPrepaymentService.promotionPlan(promotionPlanDtos);
            return Result.ok(prepaidBySupplierVos);
        } catch (Exception e){
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
    /**
     * 获取供应商数据
     * @param
     * @return
     */
    @PostMapping ("getSuppliers")
    @ApiOperation(value = "获取供应商数据")
    public Result<?> getSuppliers(@RequestBody List<CommercePrepaymentDto> commercePrepaymentDtos){
        try {
            List<MaterialInformationVo> commercePrepaymentVos=eventPrepaymentService.getSuppliers(commercePrepaymentDtos);
            return Result.ok(commercePrepaymentVos);
        } catch (Exception e){
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
    /**
     * 获取采购单数据
     * @param
     * @return
     */
    @PostMapping("savePurchase")
    @ApiOperation(value = "提交采购单数据")
    public Result<?> savePurchase(@RequestBody List<PurchaseOrderDto> purchaseOrderDtos){
        try {
            List<MaterialInformationVo> purchaseVos=eventPrepaymentService.savePurchase(purchaseOrderDtos);
            return Result.ok(purchaseVos);
        } catch (Exception e){
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 手动推送EC
     * @param ids 付款单ID
     * @return
     */
    @ApiOperation(value = "手动推送CE")
    @PostMapping("/pushEventPrepaymentToCEHandle")
    public Result<?> pushEventPrepaymentToCEHandle(@ApiParam(name = "ids",value = "预付单ID") @RequestBody List<String> ids){
        boolean lockSuccess = false;
        try {
            Assert.notEmpty(ids, "请选择数据!");
            lockSuccess = redisLockService.batchLock(EventPrepaymentConstant.EVENT_PREPAYMENT_PUSH_REDIS_LOCK, ids, TimeUnit.MINUTES, 10);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            Map<String, String> map = this.eventPrepaymentService.pushEventPrepaymentToCEHandle(ids);
            if (Objects.nonNull(map) && !map.isEmpty()) {
                StringBuilder sb = new StringBuilder();
                for (Map.Entry<String, String> entry : map.entrySet()) {
                    sb.append("[").append(entry.getKey()).append("]:").append(entry.getValue()).append("\n");
                }
                return Result.error(sb.toString());
            }
            return Result.ok();
        } catch (Exception e) {
            e.printStackTrace();
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.batchUnLock(EventPrepaymentConstant.EVENT_PREPAYMENT_PUSH_REDIS_LOCK, ids);
            }
        }
    }

    /**
     * 手动推送SAP
     * @param ids 付款单ID
     * @return
     */
    @ApiOperation(value = "手动推送折扣上账")
    @PostMapping("/pushEventPrepaymentToSAPHandle")
    public Result<?> pushEventPrepaymentToSAPHandle(@ApiParam(name = "id",value = "预付单ID") @RequestBody List<String> ids){
        boolean lockSuccess = false;
        try {
            Assert.notEmpty(ids, "请选择数据!");
            lockSuccess = redisLockService.batchLock(EventPrepaymentConstant.EVENT_PREPAYMENT_PUSH_REDIS_LOCK, ids, TimeUnit.MINUTES, 10);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            Map<String, String> map = this.eventPrepaymentService.pushEventPrepaymentToSAPHandle(ids);
            if (Objects.nonNull(map) && !map.isEmpty()) {
                StringBuilder sb = new StringBuilder();
                for (Map.Entry<String, String> entry : map.entrySet()) {
                    sb.append("[").append(entry.getKey()).append("]:").append(entry.getValue()).append("\n");
                }
                return Result.error(sb.toString());
            }
            return Result.ok();
        } catch (Exception e) {
            e.printStackTrace();
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.batchUnLock(EventPrepaymentConstant.EVENT_PREPAYMENT_PUSH_REDIS_LOCK, ids);
            }

        }
    }

    /**
     * 查询EC状态
     * @param paymentReceiptQueryStatusDto 请求参数
     * @return
     */
    @ApiOperation(value = "查询CE状态")
    @PostMapping("/queryPaymentReceiptECStatus")
    public Result<?> queryPaymentReceiptECStatus(@ApiParam(name = "paymentReceiptQueryStatusDto",value = "请求参数") @RequestBody PaymentReceiptQueryStatusDto paymentReceiptQueryStatusDto){
        try {
            this.eventPrepaymentService.queryPaymentReceiptECStatus(paymentReceiptQueryStatusDto);
            return Result.ok("查询成功");
        }catch (Exception e){
            e.printStackTrace();
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "查询活动细案明细已预付金额数据")
    @PostMapping("findTotalPrepaidAmountByDetailPlanItemList")
    public Result<List<CustomerPrepaymentVo>> findTotalPrepaidAmountByDetailPlanItemList(@RequestBody TotalPrepaidAmountDto dto){
        try {
            return Result.ok(this.eventPrepaymentService.findTotalPrepaidAmountByDetailPlanItemList(dto));
        }catch (Exception e){
            e.printStackTrace();
            return Result.error(e.getMessage());
        }
    }
}
