package com.biz.crm.tpm.business.event.prepayment.local.service.internal;


import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.aliyun.openservices.shade.com.google.common.collect.Lists;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.tpm.business.event.prepayment.local.entity.CustomerPrepaymentEntity;
import com.biz.crm.tpm.business.event.prepayment.local.entity.PrepaidBySupplierEntity;
import com.biz.crm.tpm.business.event.prepayment.local.enums.PayInAdvanceEnum;
import com.biz.crm.tpm.business.event.prepayment.local.repository.CustomerPrepaymentRepository;
import com.biz.crm.tpm.business.event.prepayment.sdk.dto.CustomerPrepaymentDto;
import com.biz.crm.tpm.business.event.prepayment.sdk.dto.EventPrepaymentDto;
import com.biz.crm.tpm.business.event.prepayment.sdk.dto.log.CustomerPrepaymentLogEventDto;
import com.biz.crm.tpm.business.event.prepayment.sdk.event.CustomerPrepaymentLogEventListener;
import com.biz.crm.tpm.business.event.prepayment.sdk.service.CustomerPrepaymentService;
import com.biz.crm.tpm.business.event.prepayment.sdk.vo.CustomerPrepaymentVo;
import com.biz.crm.tpm.business.event.prepayment.sdk.vo.EventPrepaymentVo;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Maps;
import liquibase.pro.packaged.S;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.biz.crm.tpm.business.event.prepayment.local.constant.EventPrepaymentConstant.AUDIT_CUSTOMER_PREPAYMENT_DETAIL_PREFIX;
import static com.biz.crm.tpm.business.event.prepayment.local.constant.EventPrepaymentConstant.AUDIT_UP_ACCOUNT_DETAIL_PREFIX;

/**
 * 活动明细数据(CustomerPrepaymentEntity)服务实现类
 *
 * @author xiaoyoujun
 * @date 2022年11月18日 9:39
 */
@Service("customerPrepaymentService")
public class CustomerPrepaymentServiceImpl implements CustomerPrepaymentService {

    @Autowired(required = false)
    private CustomerPrepaymentRepository customerPrepaymentRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;


    /**
     * 新增数据
     * @param list 实体对象
     * @return 新增结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void create(List<CustomerPrepaymentDto> list, EventPrepaymentDto dto) {
        Validate.notNull(dto, "对象不能为空!");
        Validate.notEmpty(list, "明细不能为空!");
        //校验是否有活动明细已申请预付，但是状态仍未审批通过
        List<String> detailPlanItemCodeList = list.stream()
                .filter(k -> StringUtils.isNotEmpty(k.getActivityDetailItemCode()))
                .map(CustomerPrepaymentDto::getActivityDetailItemCode)
                .collect(Collectors.toList());
        Validate.notEmpty(detailPlanItemCodeList, "活动细案明细编码不能为空!");
        //获取客户的MDG编码
        Map<String, CustomerVo> customerVoMap = Maps.newHashMap();
        Set<String> customerCodeSet = list.stream()
                .filter(e -> StringUtils.isNotBlank(e.getCustomerCode()))
                .map(CustomerPrepaymentDto::getCustomerCode).collect(Collectors.toSet());
        if (CollectionUtil.isNotEmpty(customerCodeSet)) {
            List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(new ArrayList<>(customerCodeSet));
            if (CollectionUtil.isNotEmpty(customerVos)) {
                customerVoMap = customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode,
                        v -> v, (newValue, oldValue) -> newValue));
            }
        }
        List<CustomerPrepaymentEntity> existEntityList = this.customerPrepaymentRepository.findByActivityDetailItemCodeListAndProcessStatus(detailPlanItemCodeList, ProcessStatusEnum.PASS.getDictCode());
        if (!CollectionUtils.isEmpty(existEntityList)) {
            throw new UnsupportedOperationException("已下活动细案明细存在未审批通过的预付数据：" + JSON.toJSONString(existEntityList.stream().map(CustomerPrepaymentEntity::getActivityDetailItemCode).collect(Collectors.toList())));
        }

        List<String> salesOrgCodeList = new ArrayList<>();
        Map<String, CustomerVo> finalCustomerVoMap = customerVoMap;
        list.forEach(e -> {
            Validate.notEmpty(e.getActivityDetailItemCode(), "活动细案明细编码不能为空!");
            Validate.notNull(e.getPrepaidAmount(), "本次预付金额必填");
            if (StringUtils.isNotEmpty(e.getActivityOrgCode())) {
                e.setSalesOrgCode(e.getActivityOrgCode());
                e.setSalesOrgName(e.getActivityOrgName());
            }
            e.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            e.setTenantCode(TenantUtils.getTenantCode());
            e.setPrepaidCoding(dto.getPrepaidCoding());
            if (PayInAdvanceEnum.CUSTOMER.getCode().equals(dto.getPrepaymentType())) {
                Validate.notNull(e.getFeeAmount(), "申请金额必填");
                Validate.isTrue(e.getFeeAmount().compareTo(e.getPrepaidAmount()) >= 0, "本次预付金额必须≤申请金额");
            }

            if (StringUtils.isNotBlank(e.getCustomerCode())
                    && !BusinessUnitEnum.SON_COMPANY.getCode().equals(dto.getBusinessUnitCode())) {
                CustomerVo customerVo = finalCustomerVoMap.get(e.getCustomerCode());
                Validate.notNull(customerVo, "未找到客户编码【" + e.getCustomerCode() + "】所对应的客户数据");
                e.setCustomerMdgCode(customerVo.getErpCode());
            }
            if (StringUtils.isEmpty(e.getSalesOrgName())
                    && StringUtils.isNotEmpty(e.getSalesOrgCode())) {
                salesOrgCodeList.add(e.getSalesOrgCode());
            }
        });

        Set<String> activityDetailItemCodeSet = list.stream()
                .filter(e -> StringUtils.isNotBlank(e.getActivityDetailItemCode()))
                .map(CustomerPrepaymentDto::getActivityDetailItemCode).collect(Collectors.toSet());

        //获取明细编码已有预付数据，计算已预付金额
        List<CustomerPrepaymentEntity> hasPrepaidList = customerPrepaymentRepository.findByActivityDetailItemCode(new ArrayList<>(activityDetailItemCodeSet));
        Map<String, BigDecimal> prepaidAmountMap = Maps.newHashMap();
        if (CollectionUtil.isNotEmpty(hasPrepaidList)) {
            prepaidAmountMap = hasPrepaidList.stream()
                    .filter(k -> StringUtils.isNotEmpty(k.getActivityDetailItemCode()))
                    .filter(k -> Objects.nonNull(k.getPrepaidAmount()))
                    .collect(Collectors.groupingBy(CustomerPrepaymentEntity::getActivityDetailItemCode,
                            Collectors.reducing(BigDecimal.ZERO, CustomerPrepaymentEntity::getPrepaidAmount, BigDecimal::add)));
        }
        int number = list.size();

        Map<String, String> salesMap = Maps.newHashMap();
        if (CollectionUtil.isNotEmpty(salesOrgCodeList)) {
            salesMap = salesOrgVoService.findNameBySalesOrgCodes(Lists.partition(salesOrgCodeList, 800));
        }

        List<String> detailCodeList = this.generateCodeService.generateCode(AUDIT_CUSTOMER_PREPAYMENT_DETAIL_PREFIX, number, 6, 2, TimeUnit.DAYS);
        List<String> upAccountCodeList = this.generateCodeService.generateCode(AUDIT_UP_ACCOUNT_DETAIL_PREFIX, number, 9, 0, TimeUnit.DAYS);
        for (int i = 0; i < list.size(); i++) {
            CustomerPrepaymentDto dto1 = list.get(i);
            list.get(i).setCustomerPrepaymentDetailCode(detailCodeList.get(i));
            list.get(i).setUpAccountCode(upAccountCodeList.get(i));
            list.get(i).setPrepaidAmountApplied(prepaidAmountMap.getOrDefault(dto1.getActivityDetailItemCode(), BigDecimal.ZERO));

            if (StringUtils.isEmpty(dto1.getSalesOrgName())
                    && StringUtils.isNotEmpty(dto1.getSalesOrgCode())) {
                list.get(i).setSalesOrgName(salesMap.get(dto1.getSalesOrgCode()));
            }
        }

        //保存
        Collection<CustomerPrepaymentEntity> entities = this.nebulaToolkitService.copyCollectionByWhiteList(list, CustomerPrepaymentDto.class, CustomerPrepaymentEntity.class, LinkedHashSet.class, ArrayList.class);
        this.customerPrepaymentRepository.saveOrUpdateBatch(entities);
    }

    /**
     * 删除数据
     * @param ids 主键结合
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "删除数据时，主键集合不能为空！");
        List<CustomerPrepaymentEntity> customerPrepaymentEntities = this.customerPrepaymentRepository.listByIds(ids);
        Validate.notEmpty(customerPrepaymentEntities, "根据提供的主键集合信息，未能获取到相应数据");
        this.customerPrepaymentRepository.removeByIds(ids);

        //删除业务日志
        Collection<CustomerPrepaymentDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(customerPrepaymentEntities,
                CustomerPrepaymentEntity.class, CustomerPrepaymentDto.class, HashSet.class, ArrayList.class);
        SerializableBiConsumer<CustomerPrepaymentLogEventListener, CustomerPrepaymentLogEventDto> onDelete =
                CustomerPrepaymentLogEventListener::onDelete;
        for (CustomerPrepaymentDto dto : dtoList) {
            CustomerPrepaymentLogEventDto logEventDto = new CustomerPrepaymentLogEventDto();
            logEventDto.setOriginal(dto);
            this.nebulaNetEventClient.publish(logEventDto, CustomerPrepaymentLogEventListener.class, onDelete);
        }
    }
    /**
     * 修改数据
     * @param list 实体对象
     * @return 修改结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void update(List<CustomerPrepaymentDto> list, EventPrepaymentDto dto) {
        List<String> idListDto = list.stream().filter(e -> e.getId() != null).map(e -> e.getId()).collect(Collectors.toList());
        //获取客户的MDG编码
        Map<String, CustomerVo> customerVoMap = null;
        Set<String> customerCodeSet = list.stream().filter(e -> StringUtils.isNotBlank(e.getCustomerCode())).map(CustomerPrepaymentDto::getCustomerCode).collect(Collectors.toSet());
        List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(new ArrayList<>(customerCodeSet));
        if (!org.apache.commons.collections.CollectionUtils.isEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity()));
        }
        for (CustomerPrepaymentDto customerPrepaymentDto : list) {
            if (StringUtils.isNotEmpty(customerPrepaymentDto.getActivityOrgCode())) {
                customerPrepaymentDto.setSalesOrgCode(customerPrepaymentDto.getActivityOrgCode());
                customerPrepaymentDto.setSalesOrgName(customerPrepaymentDto.getActivityOrgName());
            }
        }
        //删除
        customerPrepaymentRepository.deleteByIdNotInList(idListDto, dto.getPrepaidCoding());
        if (!CollectionUtils.isEmpty(list)) {
            //保存
            Collection<CustomerPrepaymentEntity> entities = this.nebulaToolkitService.copyCollectionByWhiteList(list, CustomerPrepaymentDto.class, CustomerPrepaymentEntity.class, LinkedHashSet.class, ArrayList.class);
            Map<String, CustomerVo> finalCustomerVoMap = customerVoMap;
            entities.forEach(e -> {
                Validate.notNull(e.getPrepaidAmount(), "本次预付金额必填");
//                if(e.getPrepaidAmountApplied().add(e.getPrepaidAmount()).compareTo(e.getFeeAmount()) > 0){
//                    throw new UnsupportedOperationException("【本次预付金额】应小于【申请金额】与【已预付金额】的差");
//                }
                e.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                e.setTenantCode(TenantUtils.getTenantCode());
                e.setPrepaidCoding(dto.getPrepaidCoding());
                if (dto.getPrepaymentType().equals(PayInAdvanceEnum.CUSTOMER.getCode())) {
                    Validate.notNull(e.getFeeAmount(), "申请金额必填");
                    Validate.isTrue(e.getFeeAmount().compareTo(e.getPrepaidAmount()) >= 0, "本次预付金额必须≤申请金额");
                }
                if (StringUtils.isNotBlank(e.getCustomerCode())) {
                    CustomerVo customerVo = finalCustomerVoMap.get(e.getCustomerCode());
                    Validate.notNull(customerVo, "未找到客户编码【"+ e.getCustomerCode() +"】所对应的客户数据");
                    e.setCustomerMdgCode(customerVo.getErpCode());
                }
            });
            for(CustomerPrepaymentEntity entity : entities){
                if(StringUtils.isEmpty(entity.getSalesOrgName()) && StringUtils.isNotEmpty(entity.getSalesOrgCode())){
                    SalesOrgVo salesOrgVo = salesOrgVoService.findBySalesOrgCode(entity.getSalesOrgCode());
                    if(salesOrgVo != null){
                        entity.setSalesOrgName(salesOrgVo.getSalesOrgName());
                    }
                }
            }
            this.customerPrepaymentRepository.saveOrUpdateBatch(entities);
        }
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public CustomerPrepaymentVo findById(String id) {
        CustomerPrepaymentEntity customerPrepaymentEntity = this.customerPrepaymentRepository.getById(id);
        if (customerPrepaymentEntity==null){
            return null;
        }
        CustomerPrepaymentVo vo = nebulaToolkitService.copyObjectByWhiteList(customerPrepaymentEntity,CustomerPrepaymentVo.class,LinkedHashSet.class, ArrayList.class);
        vo.setActivityOrgCode(vo.getSalesOrgCode());
        vo.setActivityOrgName(vo.getSalesOrgName());
        return vo;
    }
    /**
     * 通过预付编码查询单条数据
     *
     * @param prepaidCoding 预付编码
     * @return 单条数据
     */
    @Override
    public CustomerPrepaymentVo findByPrepaidCoding(String prepaidCoding) {
        QueryWrapper<CustomerPrepaymentEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("prepaid_coding",prepaidCoding);
        CustomerPrepaymentEntity customerPrepaymentEntity = customerPrepaymentRepository.getOne(wrapper);
        return this.nebulaToolkitService.copyObjectByWhiteList(customerPrepaymentEntity,CustomerPrepaymentVo.class,LinkedHashSet.class, ArrayList.class);
    }
    /**
     * 通过预付编码查询多条数据
     *
     * @param prepaid 预付编码
     * @return 多条数据
     */
    @Override
    public List<CustomerPrepaymentVo> findByPrepaid(String prepaid) {
        QueryWrapper<CustomerPrepaymentEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("prepaid_coding",prepaid);
        List<CustomerPrepaymentEntity> list = customerPrepaymentRepository.list(wrapper);
        List<CustomerPrepaymentVo> customerPrepaymentVos = new ArrayList<>(nebulaToolkitService.copyCollectionByWhiteList(list, CustomerPrepaymentEntity.class, CustomerPrepaymentVo.class, LinkedHashSet.class, ArrayList.class));
        if (!CollectionUtils.isEmpty(customerPrepaymentVos)) {
            for (CustomerPrepaymentVo customerPrepaymentVo : customerPrepaymentVos) {
                customerPrepaymentVo.setActivityOrgCode(customerPrepaymentVo.getSalesOrgCode());
                customerPrepaymentVo.setActivityOrgName(customerPrepaymentVo.getSalesOrgName());
            }
        }
        return customerPrepaymentVos;
    }

    @Override
    public List<CustomerPrepaymentVo> findTotalPrepaidAmountByDetailPlanItemList(List<String> detailPlanItemCodeList, String processStatus) {
        return this.customerPrepaymentRepository.findTotalPrepaidAmountByDetailPlanItemList(detailPlanItemCodeList, processStatus);
    }


}
