package com.biz.crm.tpm.business.event.prepayment.local.service.internal;


import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.tpm.business.event.prepayment.local.entity.MaterialProcurementEntity;
import com.biz.crm.tpm.business.event.prepayment.local.repository.MaterialProcurementRepository;
import com.biz.crm.tpm.business.event.prepayment.sdk.dto.MaterialProcurementDto;
import com.biz.crm.tpm.business.event.prepayment.sdk.dto.log.MaterialProcurementLogEventDto;
import com.biz.crm.tpm.business.event.prepayment.sdk.event.MaterialProcurementLogEventListener;
import com.biz.crm.tpm.business.event.prepayment.sdk.service.MaterialProcurementService;
import com.biz.crm.tpm.business.event.prepayment.sdk.vo.MaterialProcurementVo;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 采购单数据(MaterialProcurementEntity)服务实现类
 *
 * @author xiaoyoujun
 * @date 2022年11月26日 13:39
 */
@Service("materialProcurementService")
public class MaterialProcurementServiceImpl implements MaterialProcurementService {

    @Autowired(required = false)
    private MaterialProcurementRepository materialProcurementRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;


    /**
     * 新增数据
     * @param list 实体对象
     * @return 新增结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void create(List<MaterialProcurementDto> list, String code) {
        //校验是否有物料明细已申请预付，但是状态仍未审批通过
        Assert.hasLength(code,"编码不能为空!");
        Assert.notEmpty(list,"明细不能为空!");
        list.forEach(e -> {
            Assert.notNull(e,"明细对象不能为空!");
            Assert.hasLength(e.getCode(),"编码不能为空!");
        });
        List<String> codeList = list.stream()
                .filter(k-> StringUtils.isNotEmpty(k.getCode()))
                .map(MaterialProcurementDto::getCode)
                .collect(Collectors.toList());
        List<MaterialProcurementEntity> existEntityList = this.materialProcurementRepository
                .findByActivityDetailItemCodeListAndProcessStatus(codeList, ProcessStatusEnum.PASS.getDictCode());
        if (!CollectionUtils.isEmpty(existEntityList)) {
            throw new UnsupportedOperationException("已下采购单数据存在未审批通过的预付数据：" + JSON.toJSONString(existEntityList.stream().map(MaterialProcurementEntity::getCode).collect(Collectors.toList())));
        }
        Map<String,BigDecimal> amountMap = Maps.newHashMap();
        List<MaterialProcurementEntity> hasPrepaidList = materialProcurementRepository.findByCodeList(codeList);
        if (CollectionUtil.isNotEmpty(hasPrepaidList)) {
            amountMap.putAll(hasPrepaidList.stream()
                    .filter(k -> StringUtils.isNotEmpty(k.getCode()))
                    .filter(k -> Objects.nonNull(k.getPrepaidAmount()))
                    .collect(Collectors.groupingBy(MaterialProcurementEntity::getCode,
                            Collectors.reducing(BigDecimal.ZERO, MaterialProcurementEntity::getPrepaidAmount, BigDecimal::add))));
        }
        list.forEach(e -> {
            e.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            e.setTenantCode(TenantUtils.getTenantCode());
            e.setPrepaidCoding(code);
            e.setPrepaidAmountApplied(amountMap.getOrDefault(e.getCode(), BigDecimal.ZERO));
        });
        //保存
        Collection<MaterialProcurementEntity> entities = this.nebulaToolkitService.copyCollectionByWhiteList(list, MaterialProcurementDto.class, MaterialProcurementEntity.class, LinkedHashSet.class, ArrayList.class);
        this.materialProcurementRepository.saveOrUpdateBatch(entities);
    }

    /**
     * 删除数据
     * @param ids 主键结合
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "删除数据时，主键集合不能为空！");
        List<MaterialProcurementEntity> materialProcurementEntities = this.materialProcurementRepository.listByIds(ids);
        Validate.notEmpty(materialProcurementEntities, "根据提供的主键集合信息，未能获取到相应数据");
        this.materialProcurementRepository.removeByIds(ids);

        //删除业务日志
        Collection<MaterialProcurementDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(materialProcurementEntities,
                MaterialProcurementEntity.class, MaterialProcurementDto.class, HashSet.class, ArrayList.class);
        SerializableBiConsumer<MaterialProcurementLogEventListener, MaterialProcurementLogEventDto> onDelete =
                MaterialProcurementLogEventListener::onDelete;
        for (MaterialProcurementDto dto : dtoList) {
            MaterialProcurementLogEventDto logEventDto = new MaterialProcurementLogEventDto();
            logEventDto.setOriginal(dto);
            this.nebulaNetEventClient.publish(logEventDto, MaterialProcurementLogEventListener.class, onDelete);
        }
    }
    /**
     * 修改数据
     * @param list 实体对象
     * @return 修改结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void update(List<MaterialProcurementDto> list, String code) {
        List<String> idListDto = list.stream().filter(e -> e.getId() != null).map(e -> e.getId()).collect(Collectors.toList());
        //删除
        materialProcurementRepository.deleteByIdNotInList(idListDto, code);
        if (!CollectionUtils.isEmpty(list)) {
            //保存
            Collection<MaterialProcurementEntity> entities = this.nebulaToolkitService.copyCollectionByWhiteList(list, MaterialProcurementDto.class, MaterialProcurementEntity.class, LinkedHashSet.class, ArrayList.class);
            entities.forEach(e -> {
//                if(e.getPrepaidAmountApplied().add(e.getPrepaidAmount()).compareTo(e.getAmount()) > 0){
//                    throw new UnsupportedOperationException("【本次预付金额】应小于【申请金额】与【已预付金额】的差");
//                }
                e.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                e.setTenantCode(TenantUtils.getTenantCode());
                e.setPrepaidCoding(code);
            });
            this.materialProcurementRepository.saveOrUpdateBatch(entities);
        }
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public MaterialProcurementVo findById(String id) {
        MaterialProcurementEntity materialProcurementEntity = this.materialProcurementRepository.getById(id);
        if (materialProcurementEntity==null){
            return null;
        }
        return nebulaToolkitService.copyObjectByWhiteList(materialProcurementEntity,MaterialProcurementVo.class,LinkedHashSet.class, ArrayList.class);
    }
    /**
     * 通过预付编码查询单条数据
     *
     * @param prepaidCoding 预付编码
     * @return 单条数据
     */
    @Override
    public MaterialProcurementVo findByPrepaidCoding(String prepaidCoding) {
        QueryWrapper<MaterialProcurementEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("prepaid_coding",prepaidCoding);
        MaterialProcurementEntity materialProcurementEntity = materialProcurementRepository.getOne(wrapper);
        return this.nebulaToolkitService.copyObjectByWhiteList(materialProcurementEntity,MaterialProcurementVo.class,LinkedHashSet.class, ArrayList.class);
    }
    /**
     * 通过预付编码查询多条数据
     *
     * @param prepaid 预付编码
     * @return 多条数据
     */
    @Override
    public List<MaterialProcurementVo> findByPrepaid(String prepaid) {
        QueryWrapper<MaterialProcurementEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("prepaid_coding",prepaid);
        List<MaterialProcurementEntity> list = materialProcurementRepository.list(wrapper);
        Collection<MaterialProcurementVo> materialProcurementVos = nebulaToolkitService.copyCollectionByWhiteList(list, MaterialProcurementEntity.class, MaterialProcurementVo.class, LinkedHashSet.class, ArrayList.class);
        return new ArrayList<>(materialProcurementVos);
    }

    @Override
    public List<MaterialProcurementVo> findTotalPrepaidAmountByDetailPlanItemList(List<String> detailPlanItemCodeList, String processCode) {
        return this.materialProcurementRepository.findTotalPrepaidAmountByDetailPlanItemList(detailPlanItemCodeList, processCode);
    }

}
