package com.biz.crm.tpm.business.marketing.strategy.local.controller;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.base.util.DataTooLongGenerateRespUtil;
import com.biz.crm.tpm.business.marketing.strategy.local.dto.MarketingStrategyApproveDto;
import com.biz.crm.tpm.business.marketing.strategy.local.dto.MarketingStrategyApproveSubmitDto;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategy;
import com.biz.crm.tpm.business.marketing.strategy.local.service.MarketingStrategyBudgetService;
import com.biz.crm.tpm.business.marketing.strategy.local.service.MarketingStrategyService;
import com.biz.crm.tpm.business.marketing.strategy.local.vo.MarketingStrategyApproveInfoVo;
import com.biz.crm.tpm.business.marketing.strategy.local.vo.MarketingStrategyApproveProcessInfoVo;
import com.biz.crm.tpm.business.marketing.strategy.local.vo.MarketingStrategyBudgetSumVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyOutDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.service.MarketingStrategyLockSdkService;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyItemVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyVo;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.biz.crm.tpm.business.year.budget.sdk.dto.YearBudgetDto;
import com.biz.crm.tpm.business.year.budget.sdk.vo.YearBudgetVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 营销策略(MarketingStrategy)表相关的http接口
 *
 * @author wanghaojia
 * @since 2022-11-05 16:06:03
 */
@RestController
@RequestMapping("/v1/marketingStrategy")
@Slf4j
@Api(tags = "营销策略")
public class MarketingStrategyController {
    /**
     * 服务对象
     */
    @Autowired(required = false)
    private MarketingStrategyService marketingStrategyService;

    /**
     * 服务对象
     */
    @Autowired(required = false)
    private MarketingStrategyBudgetService marketingStrategyBudgetService;

    /**
     * redis服务
     */
    @Autowired(required = false)
    private MarketingStrategyLockSdkService marketingStrategyLockSdkService;

    /**
     * 策略新增保存
     * @return
     */
    @ApiOperation(value = "策略新增保存")
    @PostMapping(value = "saveSingleMarketingStrategyNoCache")
    public Result<MarketingStrategyOutDto> saveSingleMarketingStrategyNoCache(@ApiParam(name = "MarketingStrategyDto", value = "单策略表头")@RequestBody MarketingStrategyOutDto dto){
        try {
            dto.setId(null);
            return Result.ok(this.marketingStrategyService.saveSingleMarketingStrategyNoCache(dto));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(DataTooLongGenerateRespUtil.generateRespStr(e, MarketingStrategy.class));
        }
    }

    /**
     * 策略新增保存-能力中心
     * @return
     */
    @ApiOperation(value = "策略新增保存-能力中心")
    @PatchMapping(value = "saveSingleMarketingStrategyNoCache")
    public Result<MarketingStrategyOutDto> updateSingleMarketingStrategyNoCache(@ApiParam(name = "MarketingStrategyDto", value = "单策略表头")@RequestBody MarketingStrategyOutDto dto){
        try {
            return Result.ok(this.marketingStrategyService.saveSingleMarketingStrategyNoCache(dto));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(DataTooLongGenerateRespUtil.generateRespStr(e, MarketingStrategy.class));
        }
    }

    /**
     * 策略新增保存-能力中心
     * @return
     */
    @ApiOperation(value = "策略新增保存-能力中心")
    @PostMapping(value = "saveSingleMarketingStrategyNoCachePos")
    public Result<MarketingStrategyOutDto> updateSingleMarketingStrategyNoCachePos(@ApiParam(name = "MarketingStrategyDto", value = "单策略表头")@RequestBody MarketingStrategyOutDto dto){
        try {
            return Result.ok(this.marketingStrategyService.saveSingleMarketingStrategyNoCache(dto));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(DataTooLongGenerateRespUtil.generateRespStr(e, MarketingStrategy.class));
        }
    }

    /**
     * 策略新增保存
     * @param cacheKey 明细缓存key
     * @return
     */
    @ApiOperation(value = "策略新增保存")
    @PostMapping(value = "saveSingleMarketingStrategy")
    public Result saveSingleMarketingStrategy(@ApiParam(name = "cacheKey", value = "缓存键")@RequestParam String cacheKey,
                                            @ApiParam(name = "MarketingStrategyDto", value = "单策略表头")@RequestBody MarketingStrategyDto dto){
        try {
            dto.setId(null);
            this.marketingStrategyService.saveSingleMarketingStrategy(dto,cacheKey);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "策略编辑")
    @PatchMapping(value = "saveSingleMarketingStrategy")
    public Result updateSingleMarketingStrategy(@ApiParam(name = "cacheKey", value = "缓存键")@RequestParam String cacheKey,
                                            @ApiParam(name = "MarketingStrategyDto", value = "单策略表头")@RequestBody MarketingStrategyDto dto){
        try {
            if (StringUtils.isEmpty(dto.getId())){
                throw new RuntimeException("编辑策略参数有误");
            }
            this.marketingStrategyService.saveSingleMarketingStrategy(dto,cacheKey);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过id获取营销策略数据
     */
    @ApiOperation(value = "通过id获取营销策略数据")
    @GetMapping("findById")
    public Result<MarketingStrategyVo> findById(@ApiParam(value = "id")@RequestParam String id){
        try {
            MarketingStrategyVo vo = this.marketingStrategyService.findById(id);
            return Result.ok(vo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除策略
     * @param ids 策略ids
     */
    @ApiOperation(value = "删除策略")
    @DeleteMapping(value = "delete")
    public Result<?> deleteMarketingStrategy(@ApiParam(value = "策略id集合")@RequestParam List<String> ids){
        try {
            this.marketingStrategyService.deleteMarketingStrategy(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除策略-能力中心
     * @param ids 策略ids
     */
    @ApiOperation(value = "删除策略-能力中心")
    @PostMapping(value = "deleteForPost")
    public Result<?> deleteForPost(@RequestBody List<String> ids){
        try {
            this.marketingStrategyService.deleteMarketingStrategy(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 获取提交策略信息
     */
    @ApiOperation(value = "获取提交策略信息")
    @PostMapping(value = "findApproveInfoList")
    public Result<List<MarketingStrategyApproveInfoVo>> findApproveInfoList(@ApiParam(value = "提交流程，按业态，业务单元、归属部门、策略类型、策略年度提交")@RequestBody MarketingStrategyApproveDto dto){
        try {
            List<MarketingStrategyApproveInfoVo> list = this.marketingStrategyService.findApproveInfoList(dto);
            return Result.ok(list);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据流程编号获取已经提交的策略信息
     */
    @ApiOperation(value = "根据流程编号获取已经提交的策略信息")
    @PostMapping(value = "findApproveProcessInfo")
    public Result<MarketingStrategyApproveProcessInfoVo> findApproveProcessInfo(@ApiParam(value = "流程编号")@RequestParam String processNo){
        try {
            MarketingStrategyApproveProcessInfoVo processInfoVo = this.marketingStrategyService.findApproveProcessInfo(processNo);
            return Result.ok(processInfoVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据流程编号获取已经提交的策略信息-汇总版
     */
    @ApiOperation(value = "根据流程编号获取已经提交的策略信息-汇总版")
    @GetMapping(value = "findApproveProcessSumInfo")
    public Result<List<MarketingStrategyBudgetSumVo>> findApproveProcessSumInfo(@ApiParam(value = "流程编号")@RequestParam String processNo){
        try {
            List<MarketingStrategyBudgetSumVo> processInfoVo = this.marketingStrategyService.findApproveProcessSumInfo(processNo);
            return Result.ok(processInfoVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据流程编号获取已经提交的策略信息 requestBody
     */
    @ApiOperation(value = "根据流程编号获取已经提交的策略信息")
    @PostMapping(value = "findApproveProcessInfoByBody")
    public Result<MarketingStrategyApproveProcessInfoVo> findApproveProcessInfoByBody(@ApiParam(value = "流程编号")@RequestBody MarketingStrategyDto dto){
        try {
            MarketingStrategyApproveProcessInfoVo processInfoVo = this.marketingStrategyService.findApproveProcessInfo(dto.getProcessNo());
            return Result.ok(processInfoVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 获取提交策略信息
     */
    @ApiOperation(value = "根据流程编号获取已经提交的策略明细信息")
    @GetMapping(value = "findApproveProcessItemPage")
    public Result<Page<MarketingStrategyItemVo>> findApproveInfoItemList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                                @ApiParam(value = "提交流程，按业态，业务单元、归属部门、策略类型、策略年度提交")@RequestBody MarketingStrategyApproveDto dto){
        try {
            Page<MarketingStrategyItemVo> list = this.marketingStrategyService.findApproveProcessItemPage(pageable,dto);
            return Result.ok(list);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 根据流程编号获取已经提交的策略明细信息
     */
    @ApiOperation(value = "根据流程编号获取已经提交的策略明细信息")
    @GetMapping(value = "findApprovedProcessItemPage")
    public Result<Page<MarketingStrategyItemVo>> findApprovedProcessItemPage(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                            @ApiParam(value = "流程对象") MarketingStrategyDto dto){
        try {
            Page<MarketingStrategyItemVo> page = this.marketingStrategyService.findApproveProcessItemPage(pageable,dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 提交流程，按当前登录人业态，业务单元、归属部门、策略类型、策略年度提交
     */
    @ApiOperation(value = "提交流程，按业态，业务单元、归属部门、策略类型、策略年度提交")
    @PostMapping(value = "submitApproval")
    public Result submitApproval(@ApiParam(value = "提交流程，按业态，业务单元、归属部门、策略类型、策略年度提交")@RequestBody MarketingStrategyApproveSubmitDto dto){
        List<String> codeList = null;
        try {
            codeList = marketingStrategyService.findToApprovalCodeList(dto);
            Assert.notEmpty(codeList, "没有待提交的数据！");
            this.marketingStrategyService.submitApproval(codeList, dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询可以选择的月度预算
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "查询可以选择的月度预算")
    @GetMapping("findMonthBudgetByConditions")
    public Result<Page<MonthBudgetVo>> findMonthBudgetByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                 @ApiParam(name = "MonthBudgetDto", value = "月度预算信息") MonthBudgetDto dto) {
        try {
            Page<MonthBudgetVo> page = this.marketingStrategyService.findMonthBudgetByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 查询可以选择的年度预算-垂直用
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "查询可以选择的月度预算")
    @GetMapping("findYearBudgetByConditions")
    public Result<Page<YearBudgetVo>> findYearBudgetByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                 @ApiParam(name = "YearBudgetDto", value = "年度预算信息") YearBudgetDto dto) {
        try {
            Page<YearBudgetVo> page = this.marketingStrategyService.findYearBudgetByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 营销策略占用月度预算金额
     */
    @ApiOperation(value = "营销策略占用月度预算金额")
    @PostMapping("useMonthBudgetByStrategyCodeList")
    public Result useMonthBudgetByStrategyCodeList(@RequestBody List<String> strategyCodeList) {
        boolean lockSuccess = false;
        try {
            Assert.notEmpty(strategyCodeList, "请选择数据!");
            lockSuccess = marketingStrategyLockSdkService.lock(strategyCodeList, TimeUnit.MINUTES, 3);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            this.marketingStrategyBudgetService.useMonthBudgetByCodeList(strategyCodeList);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                marketingStrategyLockSdkService.unlock(strategyCodeList);
            }
        }
    }



}
