package com.biz.crm.tpm.business.marketing.strategy.local.controller;


import com.aliyun.openservices.shade.com.google.common.collect.Lists;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.biz.crm.mn.common.page.cache.controller.MnPageCacheController;
import com.biz.crm.tpm.business.activities.template.config.sdk.dto.ActivitiesTemplateConfigSearchDto;
import com.biz.crm.tpm.business.activities.template.config.sdk.enums.ActivitiesTemplateConfigTypeEnum;
import com.biz.crm.tpm.business.activities.template.config.sdk.service.ActivitiesTemplateSdkService;
import com.biz.crm.tpm.business.activities.template.config.sdk.vo.ActivitiesTemplateConfigVo;
import com.biz.crm.tpm.business.activity.form.sdk.dto.ActivityFormSelectDto;
import com.biz.crm.tpm.business.activity.type.sdk.dto.ActivityTypeSelectDto;
import com.biz.crm.tpm.business.marketing.strategy.local.service.MarketingStrategyItemService;
import com.biz.crm.tpm.business.marketing.strategy.local.vo.MarketingStrategyBudgetSumVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyItemDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.pojo.MarketingStrategyItemScope;
import com.biz.crm.tpm.business.marketing.strategy.sdk.service.MarketingStrategyLockSdkService;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyItemVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 营销策略明细(MarketingStrategyItem)表相关的http接口
 *
 * @author wanghaojia
 * @since 2022-11-03 18:22:08
 */
@RestController
@RequestMapping("/v1/marketingStrategyItem")
@Slf4j
@Api(tags = "营销策略明细")
public class MarketingStrategyItemController extends MnPageCacheController<MarketingStrategyItemVo,MarketingStrategyItemDto> {
    /**
     * 服务对象
     */
    @Autowired(required = false)
    private MarketingStrategyItemService marketingStrategyItemService;

    /**
     * 活动配置模板服务
     */
    @Autowired(required = false)
    private ActivitiesTemplateSdkService activitiesTemplateSdkService;

    /**
     * 加锁处理service
     */
    @Autowired
    private MarketingStrategyLockSdkService marketingStrategyLockSdkService;

    /**
     * 查询策略明细
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据ø
     */
    @ApiOperation(value = "查询策略明细")
    @GetMapping("findByConditions")
    public Result<Page<MarketingStrategyItemVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                               @ApiParam(name = "MarketingStrategyItemDto", value = "查询实体") MarketingStrategyItemDto dto) {
        try {
            Page<MarketingStrategyItemVo> page = this.marketingStrategyItemService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 获取多策略新增模板
     */
    @ApiOperation(value = "获取多策略新增模板")
    @GetMapping("getMutiStrategyColumnConfig")
    public Result<ActivitiesTemplateConfigVo> getMutiStrategyColumnConfig(@ApiParam(name = "dto", value = "查询实体")  ActivitiesTemplateConfigSearchDto dto) {
        String configCode = dto.getConfigCode();
        if (StringUtils.isEmpty(configCode)){
            initColumnConfigSearchParam(dto);
            dto.setStrategyFormType("2");//多策略
            Page<ActivitiesTemplateConfigVo> result = activitiesTemplateSdkService.findByConditions(null, dto);
            List<ActivitiesTemplateConfigVo> records = result.getRecords();
            if (!CollectionUtils.isEmpty(records)){
                configCode = records.get(0).getConfigCode();
            }
        }
        if (StringUtils.isEmpty(configCode)){
            throw new RuntimeException("未获取到对应模板！");
        }
        return getActivitiesTemplateConfigVoResult(configCode);
    }

    /**
     * 获取单策略新增明细模板
     */
    @ApiOperation(value = "获取单策略新增明细模板")
    @GetMapping("getSingleStrategyColumnConfig")
    public Result<ActivitiesTemplateConfigVo> getSingleStrategyColumnConfig(@ApiParam(name = "dto", value = "查询实体")  ActivitiesTemplateConfigSearchDto dto) {
        String configCode = dto.getConfigCode();
        if (StringUtils.isEmpty(configCode)){
            initColumnConfigSearchParam(dto);
            dto.setStrategyFormType("1");//单策略
            Page<ActivitiesTemplateConfigVo> result = activitiesTemplateSdkService.findByConditions(null, dto);
            List<ActivitiesTemplateConfigVo> records = result.getRecords();
            if (!CollectionUtils.isEmpty(records)){
                configCode = records.get(0).getConfigCode();
            }
        }
        if (StringUtils.isEmpty(configCode)){
            throw new RuntimeException("未获取到对应模板！");
        }
        return getActivitiesTemplateConfigVoResult(configCode);
    }

     private void initColumnConfigSearchParam(ActivitiesTemplateConfigSearchDto dto){
         dto.setType(ActivitiesTemplateConfigTypeEnum.STRATEGY.getCode());
         dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
         Validate.notBlank(dto.getBusinessFormatCode(),"业态不能为空！");
         Validate.notBlank(dto.getBusinessUnitCode(),"业务单元不能为空！");
         Validate.notBlank(dto.getStrategyTypeCode(),"策略类型不能为空！");
         //如果是小业务单元的话就查主体的模板
         if (BusinessUnitEnum.isDefaultBusinessUnit(dto.getBusinessUnitCode())){
             dto.setBusinessUnitCode(BusinessUnitEnum.HEADQUARTERS.getCode());
         }
     }

    /**
     * 根据编码获取活动配置模板
     *
     * @param configCode 模板编码
     * @return 接口返回结果
     */
    private Result<ActivitiesTemplateConfigVo> getActivitiesTemplateConfigVoResult(String configCode) {
        try {
            ActivitiesTemplateConfigVo configVo = activitiesTemplateSdkService.findByCode(configCode);
            return Result.ok(configVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 活动类型下拉
     * @param dto 活动类型查询参数
     */
    @ApiOperation(value = "活动类型下拉")
    @PostMapping("findActivityTypeSelectList")
    public Result<List<CommonSelectVo>> findActivityTypeSelectList(@ApiParam(value = "活动类型dto")@RequestBody ActivityTypeSelectDto dto){
        try {
            List<CommonSelectVo> list = marketingStrategyItemService.findActivityTypeSelectList(dto);
            return Result.ok(list);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 活动形式下拉
     * @param dto 活动形式查询参数
     */
    @ApiOperation(value = "活动形式下拉")
    @PostMapping("findActivityFormSelectList")
    public Result<List<CommonSelectVo>> findActivityFormSelectList(@ApiParam(value = "活动形式dto")@RequestBody ActivityFormSelectDto dto){
        try {
            List<CommonSelectVo> list = marketingStrategyItemService.findActivityFormSelectList(dto);
            return Result.ok(list);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }



    /**
     * 获取预算汇总信息
     *
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "获取预算汇总信息")
    @PostMapping("findMarketingStrategyBudgetCacheSumList")
    public Result<List<MarketingStrategyBudgetSumVo>> findMarketingStrategyBudgetCacheSumList(
            @ApiParam(name = "businessUnitCode", value = "业务单元") @RequestParam String businessUnitCode,
            @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey) {
        try {
            List<MarketingStrategyBudgetSumVo> list = this.marketingStrategyItemService.findMarketingStrategyBudgetCacheSumList(businessUnitCode,cacheKey);
            return Result.ok(list);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 获取分摊到范围上的信息
     */
    @ApiOperation(value = "获取分摊到范围上的信息")
    @PostMapping("findShareScopeList")
    public Result<List<MarketingStrategyItemScope>> findShareScopeList(@ApiParam(name = "dto", value = "查询实体") @RequestBody MarketingStrategyItemDto dto){
        try {
            List<MarketingStrategyItemScope> list =  marketingStrategyItemService.findShareScopeList(dto);
            return Result.ok(list);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 营销策略明细关闭
     */
    @ApiOperation(value = "营销策略明细关闭")
    @PostMapping("closeMarketingStrategyItem")
    public Result<?> closeMarketingStrategyItem(@ApiParam(name = "ids", value = "关闭营销策略明细") @RequestBody List<String> ids) {
        boolean lockSuccess = false;
        try {
            lockSuccess = marketingStrategyLockSdkService.lock(ids, TimeUnit.HOURS,1);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            marketingStrategyItemService.closeMarketingStrategyItem(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                marketingStrategyLockSdkService.unlock(ids);
            }
        }
    }


}
