package com.biz.crm.tpm.business.marketing.strategy.local.modify.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.constant.CommonConstant;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.NumberStringDealUtil;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategyItem;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.entity.MarketingStrategyItemModify;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.entity.MarketingStrategyModify;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.entity.MarketingStrategyModifyAdjustBudget;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.repository.MarketingStrategyItemModifyRepository;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.repository.MarketingStrategyModifyAdjustBudgetRepository;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.repository.MarketingStrategyModifyRepository;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.service.MarketingStrategyBudgetModifyService;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.service.MarketingStrategyItemModifyService;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.service.MarketingStrategyModifyAdjustBudgetService;
import com.biz.crm.tpm.business.marketing.strategy.local.modify.service.MarketingStrategyModifyService;
import com.biz.crm.tpm.business.marketing.strategy.local.repository.MarketingStrategyItemRepository;
import com.biz.crm.tpm.business.marketing.strategy.sdk.constant.MarketingStrategyConstant;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.log.MarketingStrategyModifyLogEventDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.event.log.MarketingStrategyModifyLogEventListener;
import com.biz.crm.tpm.business.marketing.strategy.sdk.modify.dto.*;
import com.biz.crm.tpm.business.marketing.strategy.sdk.modify.vo.MarketingStrategyModifyAdjustBudgetVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.modify.vo.MarketingStrategyModifyVo;
import com.biz.crm.tpm.business.month.budget.sdk.dto.OperateMonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantContextHolder;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.swing.text.html.Option;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 营销策略调整服务实现
 * @author wanghaojia
 * @date 2022/12/3 15:35
 */
@Slf4j
@Service
public class MarketingStrategyModifyServiceImpl implements MarketingStrategyModifyService {

    @Autowired(required = false)
    private MarketingStrategyModifyRepository marketingStrategyModifyRepository;

    @Autowired(required = false)
    private MarketingStrategyItemModifyRepository marketingStrategyItemModifyRepository;

    @Autowired(required = false)
    private MarketingStrategyModifyAdjustBudgetRepository marketingStrategyModifyAdjustBudgetRepository;

    @Autowired(required = false)
    private MarketingStrategyItemModifyService marketingStrategyItemModifyService;

    @Autowired(required = false)
    private MarketingStrategyBudgetModifyService marketingStrategyBudgetModifyService;
    @Autowired(required = false)
    private MarketingStrategyModifyAdjustBudgetService marketingStrategyModifyAdjustBudgetService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private ProcessBusinessService processBusinessService;

    @Autowired(required = false)
    private MonthBudgetService monthBudgetService;

    @Override
    public Page<MarketingStrategyModifyVo> findByConditions(Pageable pageable, MarketingStrategyModifyDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
        dto = Optional.ofNullable(dto).orElse(new MarketingStrategyModifyDto());
        Page<MarketingStrategyModifyVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return marketingStrategyModifyRepository.findByConditions(page, dto);
    }

    /**
     * 保存营销策略调整数据
     * @param dto 表头数据
     * @param cacheKey 缓存key
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result<List<MarketingStrategyModifyAdjustBudgetVo>> validateMarketingStrategyModify(MarketingStrategyModifyDto dto, String cacheKey) {
        List<MarketingStrategyModifyAdjustBudgetVo> adjustBudgetVos = null;
        List<MarketingStrategyItemModifyDto> itemList = marketingStrategyItemModifyService.findCacheList(cacheKey);
        if (CollectionUtils.isEmpty(itemList)){
            return Result.ok(adjustBudgetVos);
        }
        marketingStrategyItemModifyService.createValidateList(dto,itemList);
        for (MarketingStrategyItemModifyDto item : itemList) {
            //业务单元不是垂直，清空掉明细上的预算信息，按明细上的预算信息重新保存
            if (!BusinessUnitEnum.VERTICAL.getCode().equals(dto.getBusinessUnitCode())){
                //其他业务单元重新组装预算信息
                MarketingStrategyBudgetModifyDto budgetDto = new MarketingStrategyBudgetModifyDto();
                budgetDto.setMonthBudgetCode(item.getMonthBudgetCode());
                budgetDto.setStrategyCode(item.getStrategyCode());
                budgetDto.setStrategyItemCode(item.getStrategyItemCode());
                budgetDto.setBudgetItemCode(item.getBudgetItemCode());
                budgetDto.setBudgetItemName(item.getBudgetItemName());
                budgetDto.setFeeBelongCode(item.getFeeBelongCode());
                budgetDto.setUseAmountStr(item.getFeeAmountStr());
                budgetDto.setUseAmount(item.getFeeAmount());
                item.setBudgetShares(Lists.newArrayList(budgetDto));
            }else {
                BigDecimal budgetFeeAmount = BigDecimal.ZERO;
                for (MarketingStrategyBudgetModifyDto budgetShare : item.getBudgetShares()) {
                    NumberStringDealUtil.validateNumberStrAndSet(budgetShare.getUseAmountStr(), "新增时，策略明细费用金额", false,budgetShare::setUseAmount, BigDecimal.class);
                    if (null != budgetShare.getUseAmount()){
                        budgetFeeAmount = budgetFeeAmount.add(budgetShare.getUseAmount());
                    }
                }
                //垂直多选预算，校验预算金额之和要等于行上的费用金额
                BigDecimal feeAmount = Optional.ofNullable(item.getFeeAmount()).orElse(BigDecimal.ZERO);
                if (feeAmount.compareTo(budgetFeeAmount) != 0){
                    throw new RuntimeException("费用金额["+feeAmount+"]不等于实际预算使用金额["+budgetFeeAmount+"]");
                }
            }
        }
        //批量校验预算信息
        List<MarketingStrategyBudgetModifyDto> budgetDtos = Lists.newArrayList();
        for (MarketingStrategyItemModifyDto itemDto : itemList) {
            if (CollectionUtils.isEmpty(itemDto.getBudgetShares())){
                continue;
            }
            budgetDtos.addAll(itemDto.getBudgetShares());
        }
        if (!CollectionUtils.isEmpty(dto.getAdjustBudgetList())){
            List<String> budgetCodeList = budgetDtos.stream().map(MarketingStrategyBudgetModifyDto::getMonthBudgetCode).distinct().collect(Collectors.toList());
            for (MarketingStrategyModifyAdjustBudgetDto adjustBudgetDto : dto.getAdjustBudgetList()) {
                if (!budgetCodeList.contains(adjustBudgetDto.getMonthBudgetCode())){
                    throw new RuntimeException("预算["+adjustBudgetDto.getMonthBudgetCode()+"]不存在于本次调整数据中");
                }
            }
        }
        List<OperateMonthBudgetDto> operateMonthBudgetDtos = marketingStrategyItemModifyService.useMonthBudget(budgetDtos,dto.getAdjustBudgetList(),false,false);
//        log.info("策略使用预算结果："+ JSONObject.toJSONString(operateMonthBudgetDtos));
        Map<String, BigDecimal> adjustBudgetAmountMap = operateMonthBudgetDtos.stream().filter(item -> {
            return null != item.getGapAmount() && item.getGapAmount().compareTo(BigDecimal.ZERO) > 0;
        }).collect(Collectors.groupingBy(OperateMonthBudgetDto::getMonthBudgetCode,
                Collectors.reducing(BigDecimal.ZERO, OperateMonthBudgetDto::getGapAmount, BigDecimal::add)));
        boolean isSuccess = false;
        if (adjustBudgetAmountMap.size() == 0){
            isSuccess = true;//没有需要补充扣减的数据
        }
        if (!CollectionUtils.isEmpty(dto.getAdjustBudgetList())){
            for (MarketingStrategyModifyAdjustBudgetDto adjustBudgetDto : dto.getAdjustBudgetList()) {
                adjustBudgetAmountMap.put(adjustBudgetDto.getMonthBudgetCode(), adjustBudgetAmountMap.getOrDefault(adjustBudgetDto.getMonthBudgetCode(),BigDecimal.ZERO).add(adjustBudgetDto.getAdjustAmount()));
            }
        }
        if (adjustBudgetAmountMap.size() == 0){
            return Result.ok(adjustBudgetVos);
        }
        List<MarketingStrategyModifyAdjustBudgetVo> result = adjustBudgetAmountMap.entrySet().stream()
                .map(item -> {
            return new MarketingStrategyModifyAdjustBudgetVo() {{
                this.setMonthBudgetCode(item.getKey());
                this.setAdjustAmount(item.getValue());
            }};
        }).collect(Collectors.toList());
        //补充预算相关属性
        List<String> monthBudgetCodeList = result.stream().map(MarketingStrategyModifyAdjustBudgetVo::getMonthBudgetCode).collect(Collectors.toList());
        monthBudgetCodeList.add("_");//加一个。。。不然会取到缓存，错误数据
        List<MonthBudgetVo> monthBudgetVos = monthBudgetService.listByCodes(monthBudgetCodeList);
        Map<String, MonthBudgetVo> monthBudgetVoMap = monthBudgetVos.stream().collect(Collectors.toMap(MonthBudgetVo::getMonthBudgetCode, Function.identity()));
        for (MarketingStrategyModifyAdjustBudgetVo adjustBudgetVo : result) {
            MonthBudgetVo monthBudgetVo = monthBudgetVoMap.get(adjustBudgetVo.getMonthBudgetCode());
            adjustBudgetVo.setYearMonthLy(monthBudgetVo.getYearMonthLy());
            adjustBudgetVo.setBudgetItemCode(monthBudgetVo.getBudgetItemCode());
            adjustBudgetVo.setBudgetItemName(monthBudgetVo.getBudgetItemName());
            adjustBudgetVo.setFeeBelongCode(monthBudgetVo.getFeeBelongCode());
            adjustBudgetVo.setAfterFreezeAmount(monthBudgetVo.getAfterFreezeAmount());
            adjustBudgetVo.setAccumulatedAvailableBalance(monthBudgetVo.getAccumulatedAvailableBalance());
            adjustBudgetVo.setUsedStrategyAmount(monthBudgetVo.getUsedStrategyAmount());
        }
        Result<List<MarketingStrategyModifyAdjustBudgetVo>> r = new Result<>();
        r.setResult(result);
        r.setMessage("您使用的["+String.join(",",monthBudgetCodeList)+"]当前预算不足，是否要同步进行预算调整？");
        r.setSuccess(isSuccess);
        return r;
    }

    /**
     * 保存营销策略调整数据
     * @param dto 表头数据
     * @param cacheKey 缓存key
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveMarketingStrategyModify(MarketingStrategyModifyDto dto, String cacheKey) {
        List<MarketingStrategyItemModifyDto> itemList = marketingStrategyItemModifyService.findCacheList(cacheKey);
        if (CollectionUtils.isEmpty(itemList)){
            throw new RuntimeException("策略变更数据不能为空!");
        }
        createValidate(dto,itemList);
        MarketingStrategyModify entity;
        MarketingStrategyModifyDto oldDto = new MarketingStrategyModifyDto();
        boolean update = !StringUtils.isBlank(dto.getId());

        //回写单头归属部门、预算项目
        Set<String> budgetItemCodeSet = new HashSet<>();
        Set<String> budgetItemNameSet = new HashSet<>();
        for (MarketingStrategyItemModifyDto item : itemList) {
            budgetItemCodeSet.add(item.getBudgetItemCode());
            budgetItemNameSet.add(item.getBudgetItemName());
        }
        String attrOrgCode = StringUtils.isNotBlank(dto.getAttrOrgCode()) ? dto.getAttrOrgCode() : itemList.get(0).getAttrOrgCode();
        String attrOrgName = StringUtils.isNotBlank(dto.getAttrOrgName()) ? dto.getAttrOrgName() : itemList.get(0).getAttrOrgName();
        String budgetItemCode = StringUtils.isNotBlank(dto.getBudgetItemCode()) ? dto.getBudgetItemCode() : String.join(",", budgetItemCodeSet);
        String budgetItemName = StringUtils.isNotBlank(dto.getBudgetItemName()) ? dto.getBudgetItemName() : String.join(",", budgetItemNameSet);
        dto.setAttrOrgCode(attrOrgCode);
        dto.setAttrOrgName(attrOrgName);
        dto.setBudgetItemCode(budgetItemCode);
        dto.setBudgetItemName(budgetItemName);

        if (!update){
            entity = nebulaToolkitService.copyObjectByWhiteList(dto,MarketingStrategyModify.class,HashSet.class,ArrayList.class);
            // redis生成营销策略编码，编码规则为MS+年月日+5位顺序数。每天都从00001开始
//            String ruleCode = StringUtils.join(MarketingStrategyConstant.MARKETING_STRATEGY_MODIFY_RULE_CODE_PRE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
            String code = this.generateCodeService.generateCode(MarketingStrategyConstant.MARKETING_STRATEGY_MODIFY_RULE_CODE_PRE, 1, 5, 2, TimeUnit.DAYS).get(0);
            entity.setModifyBusinessCode(code);
            entity.setProcessStatus(ProcessStatusEnum.PREPARE.getKey());
            entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            entity.setTenantCode(TenantContextHolder.getTenantInfo().getTenantCode());
            marketingStrategyModifyRepository.save(entity);
        }else{
            MarketingStrategyModify oldEntity = marketingStrategyModifyRepository.getById(dto.getId());
            dto.setModifyBusinessCode(oldEntity.getModifyBusinessCode());
            dto.setTenantCode(oldEntity.getTenantCode());
            entity = nebulaToolkitService.copyObjectByWhiteList(dto,MarketingStrategyModify.class,HashSet.class,ArrayList.class);
            marketingStrategyModifyRepository.updateById(entity);
            oldDto = nebulaToolkitService.copyObjectByWhiteList(oldEntity,MarketingStrategyModifyDto.class,HashSet.class,ArrayList.class);
        }
        marketingStrategyItemModifyService.saveMarketingStrategyModifyItemList(entity,update,itemList,false);
        marketingStrategyBudgetModifyService.saveMarketingStrategyBudgetList(entity,update,itemList);
        marketingStrategyModifyAdjustBudgetService.saveMarketingStrategyBudgetList(entity,update,dto.getAdjustBudgetList());

        //新增修改业务日志
        MarketingStrategyModifyLogEventDto logEventDto = new MarketingStrategyModifyLogEventDto();
        logEventDto.setNewest(dto);
        if (!update) {
            logEventDto.setOriginal(null);
            SerializableBiConsumer<MarketingStrategyModifyLogEventListener, MarketingStrategyModifyLogEventDto> onCreate =
                    MarketingStrategyModifyLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, MarketingStrategyModifyLogEventListener.class, onCreate);
        } else {
            logEventDto.setOriginal(oldDto);
            SerializableBiConsumer<MarketingStrategyModifyLogEventListener, MarketingStrategyModifyLogEventDto> onUpdate =
                    MarketingStrategyModifyLogEventListener::onUpdate;
            this.nebulaNetEventClient.publish(logEventDto, MarketingStrategyModifyLogEventListener.class, onUpdate);
        }
    }

    private void createValidate(MarketingStrategyModifyDto dto,List<MarketingStrategyItemModifyDto> itemList){
        marketingStrategyItemModifyService.createValidateList(dto,itemList);
        for (MarketingStrategyItemModifyDto item : itemList) {
            //业务单元不是垂直，清空掉明细上的预算信息，按明细上的预算信息重新保存
            if (!BusinessUnitEnum.VERTICAL.getCode().equals(dto.getBusinessUnitCode())){
                //其他业务单元重新组装预算信息
                MarketingStrategyBudgetModifyDto budgetDto = new MarketingStrategyBudgetModifyDto();
                budgetDto.setMonthBudgetCode(item.getMonthBudgetCode());
                budgetDto.setStrategyCode(item.getStrategyCode());
                budgetDto.setStrategyItemCode(item.getStrategyItemCode());
                budgetDto.setBudgetItemCode(item.getBudgetItemCode());
                budgetDto.setBudgetItemName(item.getBudgetItemName());
                budgetDto.setFeeBelongCode(item.getFeeBelongCode());
                budgetDto.setUseAmountStr(item.getFeeAmountStr());
                budgetDto.setUseAmount(item.getFeeAmount());
                item.setBudgetShares(Lists.newArrayList(budgetDto));
            }else {
                BigDecimal budgetFeeAmount = BigDecimal.ZERO;
                for (MarketingStrategyBudgetModifyDto budgetShare : item.getBudgetShares()) {
                    NumberStringDealUtil.validateNumberStrAndSet(budgetShare.getUseAmountStr(), "新增时，策略明细费用金额", false,budgetShare::setUseAmount, BigDecimal.class);
                    if (null != budgetShare.getUseAmount()){
                        budgetFeeAmount = budgetFeeAmount.add(budgetShare.getUseAmount());
                    }
                }
                //垂直多选预算，校验预算金额之和要等于行上的费用金额
                BigDecimal feeAmount = Optional.ofNullable(item.getFeeAmount()).orElse(BigDecimal.ZERO);
                if (feeAmount.compareTo(budgetFeeAmount) != 0){
                    throw new RuntimeException("费用金额["+feeAmount+"]不等于实际预算使用金额["+budgetFeeAmount+"]");
                }
            }
        }
        //批量校验预算信息
        List<MarketingStrategyBudgetModifyDto> budgetDtos = Lists.newArrayList();
        for (MarketingStrategyItemModifyDto itemDto : itemList) {
            if (CollectionUtils.isEmpty(itemDto.getBudgetShares())){
                continue;
            }
            budgetDtos.addAll(itemDto.getBudgetShares());
        }
        marketingStrategyItemModifyService.useMonthBudget(budgetDtos,dto.getAdjustBudgetList(),false,true);
    }

    /**
     * 通过id获取营销策略数据
     */
    @Override
    public MarketingStrategyModifyVo findById(String id) {
        if (StringUtils.isEmpty(id)){
            return null;
        }
        MarketingStrategyModify entity = marketingStrategyModifyRepository.getById(id);
        if (null != entity){
            MarketingStrategyModifyVo vo = nebulaToolkitService.copyObjectByWhiteList(entity, MarketingStrategyModifyVo.class, HashSet.class, ArrayList.class);
            //调整预算信息查出来
            List<MarketingStrategyModifyAdjustBudget> adjustBudgets = marketingStrategyModifyAdjustBudgetRepository.findListByModifyCode(vo.getModifyBusinessCode());
            if (!CollectionUtils.isEmpty(adjustBudgets)){
                List<MarketingStrategyModifyAdjustBudgetVo> adjustBudgetVos = (List<MarketingStrategyModifyAdjustBudgetVo>) nebulaToolkitService.copyCollectionByWhiteList(adjustBudgets, MarketingStrategyModifyAdjustBudget.class, MarketingStrategyModifyAdjustBudgetVo.class, HashSet.class, ArrayList.class);
                vo.setAdjustBudgetList(adjustBudgetVos);
            }
            return vo;
        }
        return null;
    }

    /**
     * 删除营销策略数据
     * @param ids
     */
    @Override
    public void delete(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)){
            return;
        }
        List<MarketingStrategyModify> entities = marketingStrategyModifyRepository.listByIds(ids);
        for (MarketingStrategyModify entity : entities) {
            Validate.isTrue(ProcessStatusEnum.PREPARE.getKey().equals(entity.getProcessStatus()) ||
                            ProcessStatusEnum.REJECT.getKey().equals(entity.getProcessStatus()) ||
                            ProcessStatusEnum.RECOVER.getKey().equals(entity.getProcessStatus()),
                    "策略变更["+entity.getModifyBusinessCode()+"]不是待提交、驳回或追回状态，不能删除！");
        }
        marketingStrategyModifyRepository.deleteIds(ids);
        List<String> modifyCodes = entities.stream().map(MarketingStrategyModify::getModifyBusinessCode).collect(Collectors.toList());
        marketingStrategyItemModifyService.deleteByModifyCodes(modifyCodes);
        marketingStrategyBudgetModifyService.deleteByModifyBusinessCodes(modifyCodes);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void submit(MarketingStrategyModifyApproveSubmitDto dto) {
        Validate.notNull(dto.getId(),"ID不能为空");
        MarketingStrategyModify entity = this.marketingStrategyModifyRepository.getById(dto.getId());
        String modifyBusinessCode = entity.getModifyBusinessCode();
        List<MarketingStrategyItemModifyDto> listByModifyCode = this.marketingStrategyItemModifyRepository.findListByModifyBusinessCode(modifyBusinessCode);
        if (!CollectionUtils.isEmpty(listByModifyCode)){
            throw new IllegalArgumentException("存在审批中的数据");
        }
        Validate.notNull(entity,"实例对象不存在！！");
        //待提交、驳回、追回的数据才能提交
        if (!ProcessStatusEnum.PREPARE.getKey().equals(entity.getProcessStatus()) &&
                !ProcessStatusEnum.RECOVER.getKey().equals(entity.getProcessStatus()) &&
                !ProcessStatusEnum.REJECT.getKey().equals(entity.getProcessStatus())){
            throw new RuntimeException("策略变更["+entity.getModifyBusinessCode()+"]不处于待提交、驳回、追回状态，不能提交审批！");
        }
        dto.setModifyBusinessCode(entity.getModifyBusinessCode());
        //预占用预算
        marketingStrategyItemModifyService.useMonthBudgetByModifyCode(entity.getModifyBusinessCode());
        this.commitProcess(dto);
    }

    /**
     * 提交工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
     */
    private void commitProcess(MarketingStrategyModifyApproveSubmitDto dto) {
        ProcessBusinessDto processBusiness = dto.getProcessBusiness();
        Validate.notNull(processBusiness, "提交工作流时，未传工作流对象信息!");
        processBusiness.setBusinessNo(dto.getModifyBusinessCode());
        processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
        processBusiness.setBusinessCode(MarketingStrategyConstant.PROCESS_NAME_MARKETING_STRATEGY_MODIFY);
        ProcessBusinessVo processBusinessVo = this.processBusinessService.processStart(processBusiness);

        //提交成功
        marketingStrategyModifyRepository.updateProcessStatusAndProcessNo(dto.getModifyBusinessCode(),ProcessStatusEnum.COMMIT.getKey(),processBusinessVo.getProcessNo());
    }

    /**i
     * 流程审批通过
     * @param dto 流程数据
     */
    @Override
    public void processPass(ProcessStatusDto dto) {
        marketingStrategyModifyRepository.updateProcessStatus(dto.getBusinessNo(),dto.getProcessStatus());
    }
    /**
     * 流程审批追回/驳回
     * @param dto 流程数据
     */
    @Override
    public void processRejectAndRecover(ProcessStatusDto dto) {
        marketingStrategyItemModifyService.returnMonthBudgetByModifyCode(dto.getBusinessNo());
        marketingStrategyModifyRepository.updateProcessStatus(dto.getBusinessNo(),dto.getProcessStatus());
    }

}
